package it.softecspa.sso.client;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.sso.LogCategory;
import it.softecspa.sso.client.HttpSsoConnection.Method;
import it.softecspa.sso.common.Additional;
import it.softecspa.sso.common.SsoErrorOutcome;
import it.softecspa.sso.common.SsoException;
import it.softecspa.sso.common.remote.AcquireTokenResponse;
import it.softecspa.sso.common.remote.ErrorResponse;
import it.softecspa.sso.common.remote.TokenInfoResponse;
import it.softecspa.sso.proxy.ProxedClientSetting;

import java.io.IOException;
import java.net.ProtocolException;
import java.util.Calendar;
import java.util.List;
import java.util.Map;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;



public abstract class AbstractHttpSsoController {

	protected Logger log = Logger.getLogger(LogCategory.SSO_LOG_CATEGORY);
	
	
	protected HttpClientSetting config;
	
	// Customizzazioni
	protected String usedRedirectURI;
	protected String usedLogoutURI;
	protected String usedErrorURI;
	//
	
	protected AuthStatus authStatus;
	protected Additional additional;
	
	protected HttpServletRequest request;
	protected HttpServletResponse response;
	protected HttpSession session;
	protected Cookie[] cookie;
	
	protected SsoStub stub;
	
	private boolean redirected;
	
	
	private final String SSO_TEMP_CACHE = "sso-temp-cache";
	
	private class TemporaryCache {
		
		private HttpServletRequest request;
		
		private TemporaryCache(HttpServletRequest request) {
			this.request = request;
		}
		
		private void put(AuthStatus object) {
			request.setAttribute(SSO_TEMP_CACHE, object);
		}
		
		private AuthStatus get() {
			return (AuthStatus)request.getAttribute(SSO_TEMP_CACHE);
		}
		
		private void clear() {
			request.removeAttribute(SSO_TEMP_CACHE);
		}
	}
	
	private class SessionCache {
		
		private HttpServletRequest request;
		private HttpSession session;
		
		private SessionCache(HttpServletRequest request) {
			this.request = request;
			this.session = request.getSession(false);
		}
		
		private void put(AuthStatus object) {
			if (session==null) {
				if (log.isDebugEnabled()) log.debug("SSO - Session is required, create it!");
				session = request.getSession(true);				
			}
			session.setAttribute(SSO_TEMP_CACHE, object);
		}
		
		private AuthStatus get() {
			if (session!=null) {
				return (AuthStatus)session.getAttribute(SSO_TEMP_CACHE);
			}
			return null;
		}
		
		private void clear() {
			if (session!=null) {
				session.removeAttribute(SSO_TEMP_CACHE);
			}
		}
	}
	
	private class CookieCache {

		private HttpServletRequest request;
		private HttpServletResponse response;
		
		public CookieCache(HttpServletRequest request, HttpServletResponse response) {
			this.request = request;
			this.response = response;
		}
		
		private String get() {
			if (!config.isCookieEnable()) return null;
			//
			Cookie[] cookies = request.getCookies();
			if (cookies!=null) {
				for (Cookie cookie : cookies) {
					if (cookie.getName().equals(SSO_TEMP_CACHE)) {
						if (log.isTraceEnabled()) {
							log.trace("SSO - found myCookie:\n"+ 
									  "-[COOKIE]-------------------------------------------------------<<\n" +
									  " name = " + XString.toStringBetweenApex(cookie.getName()) + "\n"+
									  " value = " + XString.toStringBetweenApex(cookie.getValue()) + "\n"+
									  " domain = " + XString.toStringBetweenApex(cookie.getDomain()) + "\n"+
									  " path = " + XString.toStringBetweenApex(cookie.getPath()) + "\n"+
									  " maxage = " + cookie.getMaxAge() + "\n"+
									  " version = " + cookie.getVersion() + "\n" +
									  " comment = " + XString.toStringBetweenApex(cookie.getComment()) + "\n" +
									  " secure = " + cookie.getSecure() + "\n" +
									  "--------------------------------------------------------------------<<");						
						}
						String value = cookie.getValue();
						if (XString.isBlankNullTrim(value)) return null;
						String[] values = value.split("\\|");
						
						// Stampo nel log le info lette dai cookie
						if (log.isTraceEnabled()) {
							String m = "";
							for (String s : values) {
								m += (m.length()>0?", ":"") + "'"+s+"'";
							}
							log.trace("COOKIE - split info: "+m);						
						}
						
						if (XString.isBlankNullTrim(values[0])) return null;
						if ("null".equalsIgnoreCase(values[0])) return null;
						log.info("Read token '"+values[0]+"' from cookie");
						return values[0];
					}
				}
			}
			return null;
		}
		
		public void clear() {
			//if (!config.isCookieEnable()) return;
			if (log.isTraceEnabled()) log.trace("SSO - clear myCookie"); 
			//
			Cookie myCookie = new Cookie(SSO_TEMP_CACHE, "");
			myCookie.setVersion(-1);
			if (XString.isNotBlankNullTrim(config.getCookieDomain())) myCookie.setDomain(config.getCookieDomain());
			response.addCookie(myCookie);
		}
		
		private void put(AuthStatus authStatus) {
			if (!config.isCookieEnable()) return;
			//
			String value = authStatus.getToken().getValue()+"|"+
					  	   authStatus.getToken().getName()+"|"+
					  	   authStatus.getExpires().formatISO8601zulu();	
			long maxage = authStatus.getExpires().difference(EnterpriseCalendar.now(),Calendar.SECOND);
			if (maxage<0) maxage = -1;
			//
			Cookie myCookie = new Cookie(SSO_TEMP_CACHE, value);
			myCookie.setMaxAge((int)maxage);
			if (XString.isNotBlankNullTrim(config.getCookieDomain()))  myCookie.setDomain(config.getCookieDomain());
			//
			response.addCookie(myCookie);
		}

	}

	
	
	public AbstractHttpSsoController(HttpClientSetting config) throws SsoException {
		this.config = config;
		this.config.validate();
		//
		this.usedRedirectURI = config.getDefaultRedirectURI();
		this.usedLogoutURI = config.getDefaultLogoutURI();
		this.usedErrorURI = config.getDefaultErrorURI();
		//
		this.authStatus = new AuthStatus();
		this.authStatus.setScopeString(config.getDefaultScope());
		//
		this.stub = new SsoStub(new ProxedClientSetting(config.getProperties()));
	}

	
	void setupScope(String ... scopes) {
		// Metto gli scope
		String s = "";
		for (String elem : scopes) {
			s+= (s.length()==0?"":",") + elem;
		}
		if (log.isDebugEnabled()) log.debug("SSO - Change scopeString to '"+s+"'");
		this.authStatus.setScopeString(s);
	}
	
	
	
	@SuppressWarnings("unused")
	private String[] parseArray(String array_string) {
		if (array_string==null) return new String[0];
		// Uso la virgola come separatore
		return array_string.split(",");
	}
	
	synchronized AuthStatus executeStep(Step step, HttpServletRequest request, HttpServletResponse response) throws SsoException {
		// Setup dei parametri provenienti dall'esterno
		setupField(request, response);
		redirected = false;
		
		// Autodefinizione dello step in funzione di parametri ambientali
		// -----------------------------------------------------
		if (step==null) step=Step.NO_STEP;
		if (step==Step.NO_STEP) {
			// recupero lo status a partire dalla sequenza
			step = authStatus.autoNextStep();
			if (log.isDebugEnabled()) log.debug("SSO - Last step: "+Step.NO_STEP+" ("+Step.NO_STEP.getValue()+"), next step: "+step+" ("+step.getValue()+")"); 
		} else {
			if (log.isDebugEnabled()) log.debug("SSO - Last step: "+authStatus.getStep()+" ("+authStatus.getStep().getValue()+"), next step: "+step+" ("+step.getValue()+")"); 
		}
		// -----------------------------------------------------
		
		// ------ CHIAMATA INTERNA ------
		return executeStepInner(step, request, response);
	}

	private synchronized AuthStatus executeStepInner(Step step, HttpServletRequest request, HttpServletResponse response) throws SsoException {
		try {
			if (step==null) throw new SsoException(SsoErrorOutcome.STEP_IS_NULL);
			if (step==Step.JUMP_OPERATION) return authStatus;
			
			// Validazione dello stato
			authStatus.validateNextStep(this,step);
			
			
			// Richiesta/forward ala server SSO
			switch (step.getValue()) {
			case AuthStatus.NO_STEP:
				throw new SsoException(SsoErrorOutcome.NOSTATE_STEP_NOT_VALID);
			case AuthStatus.GET_AUTH_CODE:
				requestAuthCode();
				break;
			case AuthStatus.ACQUIRE_TOKEN:
				if (callAcquireToken()) {
					// Richiesta implicita delle informazioni legate al token
					// FIXME aggiungere un parametro di configurazione
					executeStepInner(Step.GET_TOKEN_INFO, request, response);
					executeStepInner(Step.REDIRECT, request, response);
				} else {
					// Rimando alla pagina di errore
					redirectToErrorPagePage();
				}
				break;
			case AuthStatus.GET_TOKEN_INFO:
				callGetTokenInfo();
				break;
			case AuthStatus.REDIRECT:
				redirectToLoginPage();
				break;
			case AuthStatus.INVALIDATE_TOKEN:
				redirectToRevokeToken();
				break;
			default:
				authStatus.setLastSsoStatus(SsoErrorOutcome.STEP_NOT_VALID);
				throw new SsoException(authStatus.getLastSsoStatus(),"step #"+step.getValue());
			}
			
			return authStatus;
		} catch (Exception e) {
			redirectToErrorPagePage();
		}
		return authStatus;
	}
	
	private void redirectToErrorPagePage() {
		// Invalido tutta la sequenza
		
		// ----------------------------------------------------
		if (log.isTraceEnabled()) log.trace("SSO - Redirect to base url: '"+usedLogoutURI+"'");
		try {
			invalidateToken();
			authStatus.setStep(Step.NO_STEP);
			redirected = true;
			response.sendRedirect(usedLogoutURI);
		} catch (IOException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR_PAGE);
			log.error(authStatus.getStatusMessage() +  "url: '"+usedLogoutURI+"'",e);
		} catch (Exception e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR_PAGE);
			log.error(authStatus.getStatusMessage() +  "url: '"+usedLogoutURI+"'",e);
		}
	}


	private void requestAuthCode() throws SsoException {
		String url = config.getAuthorizationEndpoint();
		
		if (XString.isBlankNullTrim(authStatus.getScopeString())) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SCOPE_IS_NULL);
			throw new SsoException(authStatus.getLastSsoStatus());
		}
		String organization = getOrganization(authStatus,config);
		
		// -------------------------------------------------
		url += "?response_type=code" +
			   "&client_id="+config.getClientId()+
			   "&redirect_uri="+usedRedirectURI+
			   "&scope="+authStatus.getScopeString()+
			   (XString.isNotBlankNull(organization)?"&organization="+organization:"") +
			   (XString.isNotBlankNull(authStatus.getState())?"&state="+authStatus.getState():"");
		// --------------------------------------------------
		if (log.isTraceEnabled()) log.trace("SSO - Call autorization server: '"+url+"'");
		
		try {
			redirected = true;
			response.sendRedirect(url);
		} catch (IOException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), "url: '"+url+"'", e);
		}
		
	}


	private void redirectToLoginPage() throws SsoException {
		if (log.isTraceEnabled()) log.trace("SSO - Redirect to base url: '"+usedRedirectURI+"'");
		try {
			redirected = true;
			response.sendRedirect(usedRedirectURI);
		} catch (IOException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), "url: '"+usedRedirectURI+"'", e);
		} catch (Exception e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), "url: '"+usedRedirectURI+"'", e);
		}
	}
	
	

	/**
	 * Aquisisce il TOKE
	 * @return
	 * @throws SsoException
	 */
	private boolean callAcquireToken() throws SsoException {
		
		if (XString.isBlankNullTrim(authStatus.getCode())) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.CODE_IS_NULL);
			throw new SsoException(authStatus.getLastSsoStatus());
		}
		if (XString.isBlankNullTrim(authStatus.getScopeString())) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SCOPE_IS_NULL);
			throw new SsoException(authStatus.getLastSsoStatus());
		}
		
		try {
			HttpSsoConnection connection = new HttpSsoConnection(config.getAccessTokenEndPoint());
			connection.setRequestMethod(Method.POST);
			connection.setAuthorization("Basic " + connection.encodeBase64(config.getClientId(),config.getClientSecret()));
			connection.setAttribute("grant_type","authorization_code");
			connection.setAttribute("code",authStatus.getCode());
			connection.setAttribute("redirect_uri",usedRedirectURI);
			String organization = getOrganization(authStatus,config);
			connection.setAttributeNVL("organization",organization);
			connection.setAttributeNVL("authenticator_name", config.getDefaultAuthenticator());
			connection.setAttributeNVL("additional",additional!=null?additional.toJson():config.getJsonAdditional());
			connection.setAttributeNVL("state",authStatus.getState());
			boolean ret = connection.call(new AcquireTokenResponse());
			if (ret) {
				// Risposta positiva, ho ottenuto il token
				AcquireTokenResponse response = (AcquireTokenResponse) connection.getResponse();
				if (log.isTraceEnabled()) {
					log.trace("SSO - ACQUIRE-TOKEN: server response OK:\n"+ 
							  "-[DMSSO]------------------------------------------------------------<<\n" +
							  " Organization = "+ XString.toStringBetweenApex(organization) + "\n"+
							  " Authenticator = "+ XString.toStringBetweenApex(config.getDefaultAuthenticator()) + "\n"+
							  " Access token = " + XString.toStringBetweenApex(response.getAccessToken()) + "\n"+
							  " Expire in = " + XString.toString(response.getExpiresIn()) + "\n" +
							  " Scope = " + XString.toStringBetweenApex(response.getScope()) + "\n" +
							  " Token type = " + XString.toStringBetweenApex(response.getTokenType()) + "\n" +
							  "--------------------------------------------------------------------<<");
				}
				authStatus.setScopeString(response.getScope());
				authStatus.setToken(new AccessToken(response.getAccessToken(), response.getExpiresIn()));
				authStatus.setLastSsoStatus(SsoErrorOutcome.OK_ACQUIRED_TOKEN);
				authStatus.setCode(null);
				
				// Metto l'oggetto nella cache della chiamata
				new TemporaryCache(request).put(authStatus);
				// FINE
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				log.error("SSO - ACQUIRE-TOKEN: server error with code " + response.getResponseCode() + ":\n"+ 
						  "-[DMSSO]------------------------------------------------------------<<\n" +
						  " Organization = "+ XString.toStringBetweenApex(organization) + "\n"+
						  " Authenticator = "+ XString.toStringBetweenApex(config.getDefaultAuthenticator()) + "\n"+
						  " Code = " + response.getResponseCode() + "\n"+
						  " Message = " + XString.toStringBetweenApex(response.getResponseMessage()) + "\n"+
						  " error = " + XString.toStringBetweenApex(response.getError()) + "\n"+
						  " error_description = " + XString.toStringBetweenApex(response.getErrorDescription()) + "\n"+
						  "--------------------------------------------------------------------<<");
				authStatus.setLastSsoStatus(SsoErrorOutcome.ERROR, response.getError());
				authStatus.setCode(null);
				
				// Faccio un reset della eventuale cache della chiamata
				new TemporaryCache(request).clear();
				// FINE
			}
			
			return ret;
			
		
		} catch (ProtocolException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.REMOTE_CALL_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), e);
		} catch (IOException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.REMOTE_CALL_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), e);
		} catch (Exception e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.UNHANDLED_EXCEPTION);
			throw new SsoException(authStatus.getLastSsoStatus(), e);
		}
			
	}

	
	/**
	 * Prende l'organizzaione per priorit
	 * 1) quella passata nel prese della chiamata
	 * 2) quella configurata come default
	 * @param organization
	 * @param defaultOrganization
	 * @return
	 */
	private String getOrganization(AuthStatus authStatus, HttpClientSetting config) {
		if (XString.isNotBlankNull(authStatus.getOrganization())) return authStatus.getOrganization();
		authStatus.setOrganization(config.getDefaultOrganization());
		return authStatus.getOrganization();
	}


	private boolean callGetTokenInfo() throws SsoException {

		if (!authStatus.getToken().isValid()) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.ACCESS_TOKEN_REQUIRED);
			throw new SsoException(authStatus.getLastSsoStatus());
		}
		
		
		try {
			HttpSsoConnection connection = new HttpSsoConnection(config.getAccessTokenInfoEndPoint());
			connection.setRequestMethod(Method.GET);
			connection.setAuthorization("Basic " + connection.encodeBase64(config.getOauth2ServerSignature()));
			connection.setAttribute("access_token", authStatus.getToken().getValue());
			boolean ret = connection.call(new TokenInfoResponse(authStatus.getToken().getValue()));
			if (ret) {
				// Risposta positiva, ho ottenuto il token
				TokenInfoResponse response = (TokenInfoResponse) connection.getResponse();
				if (log.isTraceEnabled()) {
					log.trace("SSO - TOKEN-INFO: server response OK:\n"+ 
							  "-[DMSSO]------------------------------------------------------------<<\n" +
							  " Username = "+ XString.toStringBetweenApex(response.getName()) + "\n"+
							  " Audience = " + XString.toStringBetweenApex(response.getAudience()) + "\n"+
							  " Scopes = " + list2string(response.getScopes()) + "\n"+
							  " Roles = " + list2string(response.getRoles()) + "\n"+
							  " Attributes: " + map2string(response.getAttributes()) + "\n" +
							  " Expires = "+ printTimestampUTC(response.getExpires()) +"\n" +
							  "--------------------------------------------------------------------<<");
				}
				authStatus.setTokenInfo(response);
				//
				authStatus.getToken().setName(response.getName());
				authStatus.setExpires(response.getExpires());
				//
				authStatus.setLastSsoStatus(SsoErrorOutcome.OK_LOGGED);
				authStatus.setLogged(true);
				authStatus.setCode(null);
				
				if (response.getAttributes()!=null) {
					authStatus.setOrganization(response.getAttributes().get("organization"));
				}
				
				// Faccio un reset della eventuale cache della chiamata
				new TemporaryCache(request).clear();
				// Metto l'oggetto nella cache della chiamata
				new SessionCache(request).put(authStatus);
				new CookieCache(request, this.response).put(authStatus);
				// FINE			
				
				
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				//log.error("SSO - token info: server error with code "+response.getResponseCode()+", "+response.getResponseMessage());
				log.error("SSO - TOKEN-INFO: server error with code " + response.getResponseCode() + ":\n"+ 
						  "-[DMSSO]------------------------------------------------------------<<\n" +
						  " Access token = "+ XString.toStringBetweenApex(authStatus.getToken().getValue()) + "\n"+
						  " Code = " + response.getResponseCode() + "\n"+
						  " Message = " + XString.toStringBetweenApex(response.getResponseMessage()) + "\n"+
						  " error = " + XString.toStringBetweenApex(response.getError()) + "\n"+
						  " error_description = " + XString.toStringBetweenApex(response.getErrorDescription()) + "\n"+
						  "--------------------------------------------------------------------<<");
				authStatus.setLastSsoStatus(SsoErrorOutcome.GET_TOKEN_INFO_ERROR, response.getError());
				authStatus.setCode(null);
				invalidateToken();
				// FINE				
				throw new SsoException(authStatus.getLastSsoStatus());
			}
			
			return ret;
			
		
		} catch (ProtocolException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.REMOTE_CALL_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), e);
		} catch (IOException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.REMOTE_CALL_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), e);
		} catch (Exception e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.UNHANDLED_EXCEPTION);
			throw new SsoException(authStatus.getLastSsoStatus(), e);
		}
	}

	private String list2string(List<String> list) {
		if (list==null) return "";
		String s="";
		for (String value : list) {
			s += (s.length()==0?"":", ")+ "'"+value+"'";
		}
		return s;
	}
	
	private String map2string(Map<String,String> map) {
		if (map==null) return "";
		String s="";
		for (String key : map.keySet()) {
			s += "\n\t" + key + " = '" + map.get(key) + "'";
		}
		return s;
	}
	
	private String printTimestampUTC(EnterpriseCalendar timestamp) {
		if (timestamp==null) return "OO";
		return timestamp.toStringUTC();
	}
	
	
	private void redirectToRevokeToken() throws SsoException {
		String token = authStatus.getToken().getValue();	
		if (token==null) {
			// Provo a recuperalo dal cookie
			token = new CookieCache(request, response).get();
		}
		
		String url = config.getLogoutEndPoint() + "?"+(token!=null?"token="+token+"&":"")+"redirect_uri="+usedRedirectURI;
		if (log.isTraceEnabled()) log.trace("SSO - Redirect to base url: '"+url+"'");
		try {
			invalidateToken();
			authStatus.setStep(Step.NO_STEP);
			authStatus.setLastSsoStatus(SsoErrorOutcome.NOT_LOGGED);
			redirected = true;
			response.sendRedirect(url);
		} catch (IOException e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), "url: '"+url+"'", e);
		} catch (Exception e) {
			authStatus.setLastSsoStatus(SsoErrorOutcome.SEND_REDIRECT_ERROR);
			throw new SsoException(authStatus.getLastSsoStatus(), "url: '"+url+"'", e);
		}
		
	}
	
	@SuppressWarnings("unused")
	private void setupField(HttpServletRequest request, HttpServletResponse response) throws SsoException {
		this.request = request;
		this.response = response;
		
		this.session = request.getSession(false);
		
		
		// Se trovo questo parametro valorizzato la richiesta devo validare questo TOKEN perch potenzialmente valido
		String request_token = null;
		if (request.getParameter("access_token")!=null) {
			request_token = authStatus.getToken().getValue();
			authStatus.setToken(new AccessToken(request.getParameter("access_token"), 0L));
			if (log.isTraceEnabled()) log.trace("SSO - Validate token acquired from request: '"+authStatus.getToken().getValue()+"'");
			return;
		} 
				
		// Se trovo questo parametro valorizzato la richiesta arriva dalla Authorization Request
		if (request.getParameter("code")!=null) {
			authStatus.setCode(request.getParameter("code"));
			if (log.isTraceEnabled()) log.trace("SSO - Request return from authorization server with code '"+authStatus.getCode()+"'");
			return;
		} 
		
		// Se trovo questo parametro valorizzato la richiesta arriva dalla Authorization Request
		if (request.getParameter("error")!=null) {
			String error = request.getParameter("error");
			String error_description = request.getParameter("error_description");
			if (log.isDebugEnabled()) log.debug("SSO - Server response with error '"+error+"'. "+error_description);
			authStatus.setLastSsoStatus(SsoErrorOutcome.ERROR, "'"+error+"'. "+error_description);
			throw new SsoException(authStatus.getLastSsoStatus(), "'"+error+"'. "+error_description);
		} 
				
		// Verifico la presenza della cache di appoggio tra chiamate
		AuthStatus temp_cache = new TemporaryCache(request).get();
		if (temp_cache!=null) {
			if (log.isDebugEnabled()) log.debug("SSO - Acquired AuthStatus from temporary cache (attribute)");
			authStatus = temp_cache;
		}
		
		// Cerco di ricavare lo stato del processo a partire dalle condizioni "ambientali"
		if (session!=null) {
			AuthStatus sess_cache = new SessionCache(request).get();
			if (sess_cache!=null) {
				if (log.isDebugEnabled()) log.debug("SSO - Acquired AuthStatus from temporary cache (session)");
				authStatus = sess_cache;
			
				if (request_token != null) {
					/* Faccio un averifica tra il TOKEN in sessione e quello passato da request
					 * Se sono diversi posso:
					 * 1) il token della request vince sulla sessione e la cambio
					 * 2) il token di sessione vince e non prendo in considerazione quello passato
					 * 3) invalido token e sessione
					 * 
					 * Opto per l'opzione 1
					 */
					authStatus.setToken(new AccessToken(request.getParameter("access_token"), 0L));
					log.info("SSO - Change session token with request value 'access_token'");
				}
				return;
			}
			
		}

		// Controllo la presenza di un cookie della precedente session
		String cookie_token = new CookieCache(request, response).get();
		if (XString.isNotBlankNullTrim(cookie_token)) {
			authStatus.setToken(new AccessToken(cookie_token, 0L));
		}
	}



	public HttpClientSetting getConfig() {
		return config;
	}


	/**
	 * Restituisce il bean contenete le informazioni di atorizzazione
	 * @return
	 */
	public AuthStatus getAuthStatus() {
		return authStatus;
	}
	
	
	/**
	 * La chiamata  stata reinderizzata
	 * @return
	 */
	public boolean isRedirected() {
		return redirected;
	}
	
	
	/**
	 * Preimposta l'organizzazione da utilizare nelle chiamate
	 * @param organization
	 */
	public void presetOrganization(String organization) {
		authStatus.setOrganization(organization);
	}
	
	public boolean isLogged() {
		return authStatus.isLogged();
	}


	/**
	 * Invalida localmente il token (e ripulisce i cookie)
	 */
	void invalidateToken() {
		if (log.isDebugEnabled()) log.debug("SSO - Invalidate token and flush all caches");
		try {
			authStatus.invalidateToken(); 
			new TemporaryCache(request).clear();
			new SessionCache(request).clear();
			new CookieCache(request,response).clear();
		} catch (Exception e) {
			log.error("SSO - Unhandled exception in invalidate token", e);
		}
	}

	
	/**
	 * Parametri aggiuntivi legati alla richiesta di login
	 * L'elemento contiene una serie di coppie chiave valore
	 * @return
	 */
	public Additional getAdditional() {
		return additional;
	}


	public String getUsedRedirectURI() {
		return usedRedirectURI;
	}

	public void changeRedirectURI(String redirectURI) {
		this.usedRedirectURI = redirectURI;
	}


	public String getUsedLogoutURI() {
		return usedLogoutURI;
	}

	public void changeLogoutURI(String logoutURI) {
		this.usedLogoutURI = logoutURI;
	}


	public String getUsedErrorURI() {
		return usedErrorURI;
	}

	public void changeErrorURI(String errorURI) {
		this.usedErrorURI = errorURI;
	}
	

}
