package it.softecspa.sso.client;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.Properties;
import it.softecspa.sso.ClientSetting;
import it.softecspa.sso.LogCategory;
import it.softecspa.sso.SettingsCostants;
import it.softecspa.sso.common.SsoErrorOutcome;
import it.softecspa.sso.common.SsoException;

import org.apache.log4j.Logger;

public class HttpClientSetting implements ClientSetting, SettingsCostants {

	private Logger log = Logger.getLogger(LogCategory.SSO_LOG_CATEGORY);
	private Properties properties;
	
	private boolean cookieEnable;
	private String cookieDomain;
		
	private String clientId;
	private String clientSecret;

	private String authorizationEndpoint;
	private String accessTokenEndPoint;
	private String accessTokenInfoEndPoint;
	private String userInfoEndPoint;
	private String changePasswordEndPoint;	
	private String logoutEndPoint;

	private String redirectURI;
	private String logoutURI;
	private String errorURI;

	private String oauth2Server;
	private String oauth2ServerSignature;

	private String defaultScope;
	private String defaultOrganization;
	private String defaultAuthenticator;
	private String jsonAdditional;

	public HttpClientSetting(Properties properties) {
		super();
		this.properties = properties;

		// Gestione cookie
		this.cookieEnable = properties.getBoolean(COOKIE_ENABLE, false);
		this.cookieDomain = properties.get(COOKIE_DOMAIN);
		
		// Identificativi del client sul server SSO
		this.clientId = properties.get(CLIENT_ID);
		this.clientSecret = properties.get(CLIENT_SECRET);
		this.defaultScope = properties.get(DEFAULT_SCOPE);
		this.defaultOrganization = properties.get(DEFAULT_ORGANIZATION);
		this.defaultAuthenticator = properties.get(DEFAULT_AUTHENTICATOR);
		this.jsonAdditional = properties.get(JSON_ADDITIONAL);

		this.oauth2Server = properties.get(OAUT2_SERVER);
		this.oauth2ServerSignature = properties.get(OAUT2_SERVER_SIGNATURE);

		// Endpoint del serve SSO
		this.authorizationEndpoint =  normalizeEndPoint(properties.get(AUTHORIZATION_ENDPOINT));
		this.accessTokenEndPoint =  normalizeEndPoint(properties.get(TOKEN_ENDPOINT));
		this.accessTokenInfoEndPoint =  normalizeEndPoint(properties.get(TOKEN_INFO_ENDPOINT));
		this.userInfoEndPoint =  normalizeEndPoint(properties.get(USER_INFO_ENDPOINT));
		this.changePasswordEndPoint =  normalizeEndPoint(properties.get(CHANGE_PASSWORD_ENDPOINT));
		this.logoutEndPoint =  normalizeEndPoint(properties.get(LOGOUT_ENDPOINT));

		// Url di ritorno del server locale
		this.redirectURI = properties.get(REDIRECT_URI);
		this.logoutURI = properties.get(LOGOUT_URI);
		this.errorURI = properties.get(ERROR_URI);
	}

	private String normalizeEndPoint(String value) {
		if (value==null) return null;
		
		Matcher matcher = Pattern.compile("(\\$\\{"+OAUT2_SERVER+"\\})").matcher(value);
		boolean found = matcher.find();
		if (found) {
			return  matcher.replaceAll(oauth2Server);				
		}
		return value;	
	}
	
	public void validate() throws SsoException {

		checkMandatory(this.clientId, CLIENT_ID);
		checkMandatory(this.clientSecret, CLIENT_SECRET);
		checkMandatory(this.defaultScope, DEFAULT_SCOPE);
		// checkMandatory(this.oauth2Server, OAUT2_SERVER);
		checkMandatory(this.oauth2ServerSignature, OAUT2_SERVER_SIGNATURE);

		//
		checkMandatory(this.authorizationEndpoint, AUTHORIZATION_ENDPOINT);
		checkMandatory(this.accessTokenEndPoint, TOKEN_ENDPOINT);
		checkMandatory(this.accessTokenInfoEndPoint, TOKEN_INFO_ENDPOINT);
		checkMandatory(this.logoutEndPoint, LOGOUT_ENDPOINT);
		checkMandatory(this.userInfoEndPoint, USER_INFO_ENDPOINT);
		checkMandatory(this.changePasswordEndPoint, CHANGE_PASSWORD_ENDPOINT);

		//
		checkMandatory(this.redirectURI, REDIRECT_URI);
		checkMandatory(this.logoutURI, LOGOUT_URI);
		checkMandatory(this.errorURI, ERROR_URI);

		if (this.redirectURI.equalsIgnoreCase(this.logoutURI)) {
			log.warn("configuration properties '" + REDIRECT_URI + "' and '" + LOGOUT_URI + "' are equals; this could be a problem");
		}
		if (this.redirectURI.equalsIgnoreCase(this.errorURI)) {
			log.warn("configuration properties '" + REDIRECT_URI + "' and '" + ERROR_URI + "' are equals; this could be a problem");
		}

	}

	public String getClientId() {
		return clientId;
	}

	public String getClientSecret() {
		return clientSecret;
	}

	public String getAccessTokenEndPoint() {
		return accessTokenEndPoint;
	}

	public String getAccessTokenInfoEndPoint() {
		return accessTokenInfoEndPoint;
	}

	public String getAuthorizationEndpoint() {
		return authorizationEndpoint;
	}
	
	public String getDefaultRedirectURI() {
		return redirectURI;
	}

	public String getDefaultLogoutURI() {
		return logoutURI;
	}
	
	public String getDefaultErrorURI() {
		return errorURI;
	}

	public String getOauth2Server() {
		return oauth2Server;
	}

	public String getOauth2ServerSignature() {
		return oauth2ServerSignature;
	}

	private void checkMandatory(String value, String field) throws SsoException {
		if (value == null) {
			throw new SsoException(SsoErrorOutcome.FIELD_MANDATORY, "'" + field + "'");
		} else if (value instanceof String) {
			if (XString.isBlankNullTrim((String) value)) {
				throw new SsoException(SsoErrorOutcome.FIELD_EMPTY, "'" + field + "'");
			}
		}
	}

	public String getDefaultScope() {
		return defaultScope;
	}

	public String getJsonAdditional() {
		return jsonAdditional;
	}

	public String getLogoutEndPoint() {
		return logoutEndPoint;
	}

	public String getDefaultOrganization() {
		return defaultOrganization;
	}
	
	public String getDefaultAuthenticator() {
		return defaultAuthenticator;
	}
	
	public String getUserInfoEndPoint() {
		return userInfoEndPoint;
	}

	public String getChangePasswordEndPoint() {
		return changePasswordEndPoint;
	}

	public Properties getProperties() {
		return properties;
	}

	public boolean isCookieEnable() {
		return cookieEnable;
	}

	public String getCookieDomain() {
		return cookieDomain;
	}

	

	

}