package it.softecspa.sso.proxy;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.sso.LogCategory;
import it.softecspa.sso.client.HttpSsoConnection;
import it.softecspa.sso.client.HttpSsoConnection.Method;
import it.softecspa.sso.common.Additional;
import it.softecspa.sso.common.SsoErrorOutcome;
import it.softecspa.sso.common.SsoException;
import it.softecspa.sso.common.TokenInfo;
import it.softecspa.sso.common.UserInfo;
import it.softecspa.sso.common.remote.AcquireTokenResponse;
import it.softecspa.sso.common.remote.DummyResponse;
import it.softecspa.sso.common.remote.ErrorResponse;
import it.softecspa.sso.common.remote.TokenInfoResponse;
import it.softecspa.sso.common.remote.UserInfoResponse;

import java.io.IOException;
import java.net.ProtocolException;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;



public abstract class AbstractProxySsoStub {

	protected Logger log = Logger.getLogger(LogCategory.SSO_LOG_CATEGORY);	
	protected ProxedClientSetting config;
	
	
	public AbstractProxySsoStub(ProxedClientSetting config) throws SsoException {
		this.config = config;
		this.config.validate();
	}


	public ProxedClientSetting getConfig() {
		return config;
	}

	
	Authorization login(String organization, String username, String password, Additional additional, String authenticator) throws SsoException {
		return login(organization, username, password, additional!=null?additional.toJson():null, null);
	}
		
	/**
	 * Chiamata effettiva di login
	 * @param organization
	 * @param username
	 * @param password
	 * @param additional
	 * @param authenticator
	 * @return
	 * @throws SsoException
	 */
	Authorization login(String organization, String username, String password, String jsonAdditional, String authenticator) throws SsoException {
		
		if (XString.isBlankNullTrim(username)) {
			throw new SsoException(SsoErrorOutcome.USERNAME_REQUIRED);
		}
		if (XString.isBlankNullTrim(password)) {
			throw new SsoException(SsoErrorOutcome.PASSWORD_REQUIRED);
		}		
		
		Authorization auth = new Authorization();
		if (XString.isNotBlankNull(organization)) {
			auth.setOrganization(organization);
		} else {
			auth.setOrganization(config.getDefaultOrganization());
		}
		if (authenticator==null) {
			authenticator=config.getDefaultAuthenticator();
		}
		
		
		try {
			HttpSsoConnection connection = new HttpSsoConnection(config.getAccessTokenEndPoint());
			connection.setRequestMethod(Method.POST);
			connection.setAuthorization("Basic " + connection.encodeBase64(config.getClientId(),config.getClientSecret()));
			// TEST
			//connection.setAuthorization("Basic " + connection.encodeBase64("dmservice","d1906cca-d858-4420-ba54-f8c0984883bd"));
			//connection.setAuthorization("Basic " + connection.encodeBase64("dmveroni","ac3e3928-4c7d-41ec-b8bb-dc334d091493"));
			
			connection.setAttribute("grant_type","password");
			connection.setAttribute("username", username);
			connection.setAttribute("password", password);
			connection.setAttributeNVL("organization", auth.getOrganization());
			connection.setAttributeNVL("authenticator_name", authenticator);
			if (jsonAdditional!=null) {
				if (log.isDebugEnabled()) log.debug("Found additionals value on login request");
				connection.setAttributeNVL("additional",jsonAdditional);
			} else {
				jsonAdditional = config.getJsonAdditional();
				if (log.isDebugEnabled()) log.debug("Use additional on configuraton file");
				connection.setAttributeNVL("additional",jsonAdditional);
			}
			boolean ret = connection.call(new AcquireTokenResponse());
			if (ret) {
				// Risposta positiva, ho ottenuto il token
				AcquireTokenResponse response = (AcquireTokenResponse) connection.getResponse();
				if (log.isTraceEnabled()) {
					log.trace("SSO - LOGIN: server response OK:\n"+ 
							  "-[DMSSO]------------------------------------------------------------<<\n" +
							  " Organization = "+ XString.toStringBetweenApex(auth.getOrganization()) + "\n"+
							  " Authenticator = "+ XString.toStringBetweenApex(authenticator) + "\n"+
							  " Username = " + XString.toStringBetweenApex(username) + "\n"+
							  " Password = " + XString.toStringBetweenApex(password!=null?"******":null) + "\n"+
							  " Additional = " + XString.toStringBetweenApex(jsonAdditional) + "\n"+
							  " Access token = " + XString.toStringBetweenApex(response.getAccessToken()) + "\n"+
							  " Expire in = " + XString.toString(response.getExpiresIn()) + "\n"+
							  " Scope = " + XString.toStringBetweenApex(response.getScope()) + "\n"+
							  " Token type = " + XString.toStringBetweenApex(response.getTokenType()) + "\n"+
							  "--------------------------------------------------------------------<<");							  
				}
				auth.setScope(response.getScope().split(","));
				auth.setAccessToken(new AccessToken(response.getAccessToken(), response.getExpiresIn()));
				auth.setLastSsoStatus(SsoErrorOutcome.OK_ACQUIRED_TOKEN);
				auth.setLogged(true);
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				log.error("SSO - LOGIN: server error with code " + response.getResponseCode() + ":\n"+ 
						  "-[DMSSO]------------------------------------------------------------<<\n" +
						  " Organization = "+ XString.toStringBetweenApex(auth.getOrganization()) + "\n"+
						  " Authenticator = "+ XString.toStringBetweenApex(config.getDefaultAuthenticator()) + "\n"+
						  " Username = " + XString.toStringBetweenApex(username) + "\n"+
						  " Password = " + XString.toString(password!=null?"******":null) + "\n"+
						  " Additional = " + XString.toStringBetweenApex(jsonAdditional) + "\n"+
						  " Code = " + response.getResponseCode() + "\n"+
						  " Message = " + XString.toStringBetweenApex(response.getResponseMessage()) + "\n"+
						  " error = " + XString.toStringBetweenApex(response.getError()) + "\n"+
						  " error_description = " + XString.toStringBetweenApex(response.getErrorDescription()) + "\n"+
						  "--------------------------------------------------------------------<<");							  
				
				if ("Wrong credentials".equalsIgnoreCase(response.getErrorDescription())) {
					auth.setLastSsoStatus(SsoErrorOutcome.WRONG_CREDENTIAL);
					} else {
					auth.setLastSsoStatus(SsoErrorOutcome.ERROR, response.getError());
				}
			}
			return auth;
		
		} catch (ProtocolException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (IOException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (Exception e) {
			throw new SsoException(SsoErrorOutcome.UNHANDLED_EXCEPTION, e);
		}
	}

	void logout(String access_token) throws SsoException {
		
		if (XString.isBlankNullTrim(access_token)) {
			throw new SsoException(SsoErrorOutcome.ACCESS_TOKEN_REQUIRED);
		}
		
		
		try {
			HttpSsoConnection connection = new HttpSsoConnection(config.getLogoutEndPoint());
			connection.setRequestMethod(Method.GET);
			connection.setAuthorization("bearer " + access_token);
			boolean ret = connection.call(new DummyResponse());
			if (ret) {
				if (log.isTraceEnabled()) {
					log.trace("SSO - logout: server response OK");							  
				}
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				if (!"token_not_found".equalsIgnoreCase(response.getError())) {
					// Errore da non tracciare, esito ugualmente positivo
					log.error("SSO - Server error with code "+response.getResponseCode()+", "+response.getResponseMessage());
					throw new SsoException(SsoErrorOutcome.LOGOUT_ERROR, response.getError());
				} else {
					log.warn("SSO - Server error with code "+response.getResponseCode()+", "+response.getResponseMessage());
				}
				
			}
			
		} catch (ProtocolException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (IOException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (Exception e) {
			throw new SsoException(SsoErrorOutcome.UNHANDLED_EXCEPTION, e);
		}
	}
	
	
	protected TokenInfo acquireTokenInfo(String access_token) throws SsoException {
		
		if (XString.isBlankNullTrim(access_token)) {
			throw new SsoException(SsoErrorOutcome.ACCESS_TOKEN_REQUIRED);
		}
		
		try {
			HttpSsoConnection connection = new HttpSsoConnection(config.getAccessTokenInfoEndPoint());
			connection.setRequestMethod(Method.GET);
			connection.setAuthorization("Basic " + connection.encodeBase64(config.getOauth2ServerSignature()));
			connection.setAttribute("access_token", access_token);
			boolean ret = connection.call(new TokenInfoResponse(access_token));
			if (ret) {
				// Risposta positiva, ho ottenuto il token
				TokenInfoResponse response = (TokenInfoResponse) connection.getResponse();
				if (log.isTraceEnabled()) {
					log.trace("SSO - TOKEN-INFO: server response OK:\n"+ 
							  "-[DMSSO]------------------------------------------------------------<<\n" +
							  " Username = "+ XString.toStringBetweenApex(response.getName()) + "\n"+
							  " Audience = " + XString.toStringBetweenApex(response.getAudience()) + "\n"+
							  " Scopes = " + list2string(response.getScopes()) + "\n"+
							  " Roles = " + list2string(response.getRoles()) + "\n"+
							  " Attributes: " + map2string(response.getAttributes()) + "\n" +
							  " Expires = " + printTimestampUTC(response.getExpires()) + "\n" +
							  "--------------------------------------------------------------------<<");
				}
				return response;
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				log.error("SSO - TOKEN-INFO: server error with code " + response.getResponseCode() + ":\n"+ 
						  "-[DMSSO]------------------------------------------------------------<<\n" +
						  " Access token = "+ XString.toStringBetweenApex(access_token) + "\n"+
						  " Code = " + response.getResponseCode() + "\n"+
						  " Message = " + XString.toStringBetweenApex(response.getResponseMessage()) + "\n"+
						  " error = " + XString.toStringBetweenApex(response.getError()) + "\n"+
						  " error_description = " + XString.toStringBetweenApex(response.getErrorDescription()) + "\n"+
						  "--------------------------------------------------------------------<<");		
				
				throw new SsoException(SsoErrorOutcome.GET_TOKEN_INFO_ERROR, response.getError());
			}
			
		} catch (ProtocolException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (IOException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (Exception e) {
			throw new SsoException(SsoErrorOutcome.UNHANDLED_EXCEPTION, e);
		}
	}
	
	private String list2string(List<String> list) {
		if (list==null) return "";
		String s="";
		for (String value : list) {
			s += (s.length()==0?"":", ")+ "'"+value+"'";
		}
		return s;
	}
	
	private String map2string(Map<String,String> map) {
		if (map==null) return "";
		String s="";
		for (String key : map.keySet()) {
			s += "\n\t" + key + " = '" + XString.toString(map.get(key)) + "'";
		}
		return s;
	}
	
	private String printTimestampUTC(EnterpriseCalendar timestamp) {
		if (timestamp==null) return "OO";
		return timestamp.toStringUTC();
	}
	
	
	
	/**
	 * Recupera le user info
	 * @param access_token
	 * @return
	 * @throws SsoException
	 */
	protected UserInfo acquireUserInfo(String access_token) throws SsoException {
		
		if (XString.isBlankNullTrim(access_token)) {
			throw new SsoException(SsoErrorOutcome.ACCESS_TOKEN_REQUIRED);
		}
				
		try {
			HttpSsoConnection connection = new HttpSsoConnection(config.getUserInfoEndPoint());
			connection.setRequestMethod(Method.GET);
			connection.setAuthorization("bearer " + access_token);
			boolean ret = connection.call(new UserInfoResponse());
			if (ret) {
				// Risposta positiva, ho ottenuto il token
				UserInfoResponse response = (UserInfoResponse) connection.getResponse();
				if (log.isTraceEnabled()) {
					log.trace("SSO - USER-INFO: server response OK:\n"+ 
							  "-[DMSSO]------------------------------------------------------------<<\n" +
							  " Family name = " + XString.toStringBetweenApex(response.getFamilyName()) + "\n"+
							  " Given name = " + XString.toStringBetweenApex(response.getGivenName()) + "\n"+
							  " Username = " + XString.toStringBetweenApex(response.getUsername()) + "\n"+
							  " Email = " + XString.toStringBetweenApex(response.getEmail()) + "\n"+
							  " Picture url = " + XString.toStringBetweenApex(response.getPictureUrl()) + "\n" +
							  " Birthdate = " + XString.toStringBetweenApex(response.getBirthdate()) + "\n"+
							  " Roles = " + list2string(response.getRoles()) + "\n"+
							  "--------------------------------------------------------------------<<");
				}
				return response;
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				log.error("SSO - USER-INFO: server error with code " + response.getResponseCode() + ":\n"+ 
						  "-[DMSSO]------------------------------------------------------------<<\n" +
						  " Access token = "+ XString.toStringBetweenApex(access_token) + "\n"+
						  " Code = " + response.getResponseCode() + "\n"+
						  " Message = " + XString.toStringBetweenApex(response.getResponseMessage()) + "\n"+
						  " error = " + XString.toStringBetweenApex(response.getError()) + "\n"+
						  " error_description = " + XString.toStringBetweenApex(response.getErrorDescription()) + "\n"+
						  "--------------------------------------------------------------------<<");		
				throw new SsoException(SsoErrorOutcome.GET_USER_INFO_ERROR, response.getError());
			}
			
		} catch (ProtocolException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (IOException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (Exception e) {
			throw new SsoException(SsoErrorOutcome.UNHANDLED_EXCEPTION, e);
		}
	}
	
	/**
	 * Cambio password
	 * @param access_token
	 * @param organization
	 * @param username
	 * @param old_password
	 * @param new_password
	 * @throws SsoException
	 */
	protected void changePassword(String access_token, String organization, String username, String old_password, String new_password) throws SsoException {
		
		if (XString.isBlankNullTrim(access_token)) {
			throw new SsoException(SsoErrorOutcome.ACCESS_TOKEN_REQUIRED);
		}
		if (XString.isBlankNullTrim(organization)) {
			throw new SsoException(SsoErrorOutcome.ORGANIZATION_REQUIRED);
		}
		if (XString.isBlankNullTrim(username)) {
			throw new SsoException(SsoErrorOutcome.USERNAME_REQUIRED);
		}
		if (XString.isBlankNullTrim(new_password)) {
			throw new SsoException(SsoErrorOutcome.NEW_PASSWORD_REQUIRED);
		}
		
		try {
			// URL: /admin/organizationname/<organization>/username/<username>/password?oldPassword=<old_password>&newPassword=<new_password>
			String url = config.getChangePasswordEndPoint()+"/organizationname/"+ organization +
														    "/username/" + username +
														    "/password?"+(old_password!=null?"oldPassword="+old_password+"&":"")+
														    "newPassword="+new_password;
			HttpSsoConnection connection = new HttpSsoConnection(url);
			connection.setRequestMethod(Method.POST);
			connection.setAuthorization("bearer " + access_token);
			boolean ret = connection.call(new DummyResponse());
			if (ret) {
				// Risposta positiva, ho cambiato la password
				//DummyResponse response = (DummyResponse) connection.getResponse();
				if (log.isTraceEnabled()) {
					log.trace("SSO - CHANGE-PASSWORD: server response OK:\n"+ 
							  "-[DMSSO]------------------------------------------------------------<<\n" +
							  " Username = " + XString.toStringBetweenApex(username) + "\n"+
							  " Old password = " + XString.toStringBetweenApex(old_password!=null?"******":null) + "\n"+
							  // " Old password = " + XString.toStringBetweenApex(old_password) + "\n"+
							  " New password = " + XString.toStringBetweenApex(new_password!=null?"******":null) + "\n"+
							  // " New password = " + XString.toStringBetweenApex(new_password) + "\n"+
							  "--------------------------------------------------------------------<<");							  
				}
				return;
			} else {
				ErrorResponse response = (ErrorResponse) connection.getResponse();
				log.error("SSO - CHANGE-PASSWORD: server error with code " + response.getResponseCode() + ":\n"+ 
						  "-[DMSSO]------------------------------------------------------------<<\n" +
						  " Organization = " + XString.toStringBetweenApex(organization) + "\n"+
						  " Access token = "+ XString.toStringBetweenApex(access_token) + "\n"+
						  " Username = " + XString.toStringBetweenApex(username) + "\n"+
						  " Old password = " + XString.toStringBetweenApex(old_password!=null?"******":null) + "\n"+
						  " New password = " + XString.toStringBetweenApex(new_password!=null?"******":null) + "\n"+
						  " Code = " + response.getResponseCode() + "\n"+
						  " Message = " + XString.toStringBetweenApex(response.getResponseMessage()) + "\n"+
						  " error = " + XString.toStringBetweenApex(response.getError()) + "\n"+
						  " error_description = " + XString.toStringBetweenApex(response.getErrorDescription()) + "\n"+
						  "--------------------------------------------------------------------<<");				
				throw new SsoException(SsoErrorOutcome.CHANGE_PASSWORD_ERROR, response.getError());
			}
			
		} catch (ProtocolException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (IOException e) {
			throw new SsoException(SsoErrorOutcome.REMOTE_CALL_ERROR, e);
		} catch (Exception e) {
			throw new SsoException(SsoErrorOutcome.UNHANDLED_EXCEPTION, e);
		}
		
		
		
	}

}
