package it.softecspa.sso.proxy;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.Properties;
import it.softecspa.sso.ClientSetting;
import it.softecspa.sso.SettingsCostants;
import it.softecspa.sso.common.SsoErrorOutcome;
import it.softecspa.sso.common.SsoException;

/**
 * Configurazione del proxy client SSO
 */
public class ProxedClientSetting implements ClientSetting, SettingsCostants {

	private String clientId;
	private String clientSecret;
	
	private String authorizationEndpoint;
	private String accessTokenEndPoint;
	private String accessTokenInfoEndPoint;
	private String userInfoEndPoint;
	private String changePasswordEndPoint;
	private String logoutEndPoint;
	
	private String oauth2Server;
	private String oauth2ServerSignature;
		
	private String defaultScope;
	private String defaultOrganization;
	private String defaultAuthenticator;
	private String jsonAdditional;
	
	/**
	 * Questa classe contiene tutte le informazioni che configurano
	 * @param properties configurazione del client SSO
	 */
	public ProxedClientSetting(Properties properties) {
		super();
		
		// Identificativi del client sul server SSO
		this.clientId = properties.get(CLIENT_ID);
		this.clientSecret = properties.get(CLIENT_SECRET);
		this.defaultScope = properties.get(DEFAULT_SCOPE);
		this.defaultOrganization = properties.get(DEFAULT_ORGANIZATION);
		this.defaultAuthenticator = properties.get(DEFAULT_AUTHENTICATOR);
		this.jsonAdditional = properties.get(JSON_ADDITIONAL);
		
		this.oauth2Server = properties.get(OAUT2_SERVER);
		this.oauth2ServerSignature = properties.get(OAUT2_SERVER_SIGNATURE);
		
		// Endpoint del serve SSO
		this.authorizationEndpoint = normalizeEndPoint(properties.get(AUTHORIZATION_ENDPOINT));
		this.accessTokenEndPoint = normalizeEndPoint(properties.get(TOKEN_ENDPOINT));
		this.accessTokenInfoEndPoint = normalizeEndPoint(properties.get(TOKEN_INFO_ENDPOINT));
		this.userInfoEndPoint = normalizeEndPoint(properties.get(USER_INFO_ENDPOINT));
		this.changePasswordEndPoint = normalizeEndPoint(properties.get(CHANGE_PASSWORD_ENDPOINT));
		this.logoutEndPoint = normalizeEndPoint(properties.get(LOGOUT_ENDPOINT));
	}
	
	
	private String normalizeEndPoint(String value) {
		if (value==null) return null;
		
		Matcher matcher = Pattern.compile("(\\$\\{"+OAUT2_SERVER+"\\})").matcher(value);
		boolean found = matcher.find();
		if (found) {
			return  matcher.replaceAll(oauth2Server);				
		}
		return value;	
	}

	
	public void validate() throws SsoException {
		
		checkMandatory(this.clientId, CLIENT_ID);
		checkMandatory(this.clientSecret, CLIENT_SECRET);
		//checkMandatory(this.defaultScope, DEFAULT_SCOPE);
		checkMandatory(this.defaultScope, DEFAULT_ORGANIZATION);
		//checkMandatory(this.oauth2Server, OAUT2_SERVER);
		checkMandatory(this.oauth2ServerSignature, OAUT2_SERVER_SIGNATURE);
		
		//
		checkMandatory(this.authorizationEndpoint, AUTHORIZATION_ENDPOINT);
		checkMandatory(this.accessTokenEndPoint, TOKEN_ENDPOINT);
		checkMandatory(this.accessTokenInfoEndPoint, TOKEN_INFO_ENDPOINT);
		checkMandatory(this.logoutEndPoint, LOGOUT_ENDPOINT);
		checkMandatory(this.userInfoEndPoint, USER_INFO_ENDPOINT);
		checkMandatory(this.changePasswordEndPoint, CHANGE_PASSWORD_ENDPOINT);
		
	}

	
	


	public String getClientId() {
		return clientId;
	}

	public String getClientSecret() {
		return clientSecret;
	}

	public String getAccessTokenEndPoint() {
		return accessTokenEndPoint;
	}

	public String getAccessTokenInfoEndPoint() {
		return accessTokenInfoEndPoint;
	}

	public String getAuthorizationEndpoint() {
		return authorizationEndpoint;
	}

	public String getOauth2Server() {
		return oauth2Server;
	}

	public String getOauth2ServerSignature() {
		return oauth2ServerSignature;
	}

	private void checkMandatory(String value, String field) throws SsoException {
		if (value==null) {
			throw new SsoException(SsoErrorOutcome.FIELD_MANDATORY, "'"+field+"'");
		} else if (value instanceof String) {
			if (XString.isBlankNullTrim((String)value)) {
				throw new SsoException(SsoErrorOutcome.FIELD_EMPTY, "'"+field+"'");
			}
		}	
	}
	
	
	

	public String getDefaultScope() {
		return defaultScope;
	}


	public String getLogoutEndPoint() {
		return logoutEndPoint;
	}


	public String getDefaultOrganization() {
		return defaultOrganization;
	}


	public String getUserInfoEndPoint() {
		return userInfoEndPoint;
	}


	public String getChangePasswordEndPoint() {
		return changePasswordEndPoint;
	}

	public String getJsonAdditional() {
		return jsonAdditional;
	}

	public String getDefaultAuthenticator() {
		return defaultAuthenticator;
	}
	

}