// Source File Name:   WebApp.java

package it.softecspa.mvc;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.kahuna.io.File;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.services.PortalSettings;
import it.softecspa.kahuna.services.PostOffice;
import it.softecspa.kahuna.services.PostmanPat;
import it.softecspa.kahuna.util.Properties;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.mvc.businessobject.DatabaseManager;
import it.softecspa.portal.ApplicationClusterInfo;
import it.softecspa.portal.KeyStorage;
import it.softecspa.portal.LogAnalyzerProperties;
import it.softecspa.portal.Parameters;
import it.softecspa.portal.Version;
import it.softecspa.s4fs.S4Config;
import it.softecspa.s4fs.S4Engine;
import it.softecspa.s4fs.S4Exception;
import it.softecspa.task.TaskManager;

import java.net.URL;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;

@SuppressWarnings("serial")
public abstract class WebApp extends HttpServlet {

	protected Logger log = Logger.getLogger(WebApp.class);

	
	public WebApp() {
		super();
	}
	
	
	public void init(ServletConfig config) throws ServletException {
		super.init(config);
		try {
			
			// Identificazione cartella configurazione
			String logText = null;
			URL url = this.getClass().getClassLoader().getResource("conf");
			if (url != null) {
			} else {
				url = this.getClass().getClassLoader().getResource("WEB-INF/conf");
			}
			if (url == null) {
				logText = "Resources not found!";
				System.out.println("GRAVE: Unable to complete the initialization of the WebApp: "+logText);
				System.out.println("Not found folder 'conf' or 'WEB-INF/conf'");
				return;
			} else {
				logText = "Resources found in path '" + url.getFile() + "'";
			}
			String configroot = XString.convertFromUnicode(url.getFile()) + "/";

			
			// Carica il file di configurazione per LOG4J
			String log4jPropsFileName = configroot + config.getServletContext().getInitParameter("logPropertyFile");
			// init LOG configuration
			DOMConfigurator.configure(log4jPropsFileName);
			
			log.info("Web application initialization (super)\n"+
					"-[START]-------------------------------------------------------------------------------");
			log.info(logText);			

			
			
			// Carico le costanti dell'applicazione
			String configPropertyFile = config.getServletContext().getInitParameter("constantsFile");
			if (XString.isBlankNullTrim(configPropertyFile)) {
				configPropertyFile = config.getServletContext().getInitParameter("configPropertyFile");
			} else {
				log.warn("Found deprecated <param-name>constantsFile</param-name>, change with <param-name>configPropertyFile</param-name>");
			}
			if (XString.isBlankNullTrim(configPropertyFile)) {
				log.error("Not found <param-name>configPropertyFile</param-name> with valid <param-value>");
				throw new NullPointerException();
			}
			log.info("Loading configuration file '" + configPropertyFile + "'");
			Properties constants = new Properties();
			constants.load(configroot + configPropertyFile);
			log.info("File loading conpleted.");

			// m.veroni, 2010-10-18 - Aggiungo nelle properties il path per recuperare il file log4j.xml
			constants.put(PortalSettings.LOG4J,log4jPropsFileName);
			// ---------------------------------------------------------------------
			
			
			// inizializzazione classe Parameters
			log.info("Parameters class initialization");
			Parameters parameters = Parameters.getInstance();
			parameters.initialize(config, constants);
			log.info("Parameters class initialization, completed");
			// ---------------------------------------------------------------------
			
			// Creazione e configurazione ConnectionManager
			log.info("Connections manager initialization");
			DatabaseManager dbmanager = DatabaseManager.getInstance();
			DatabaseManager.getInstance().inizialize(parameters);
			log.info("Default ConnectionManager setup ");
			ConnectionManager.setDefault(dbmanager.getDefaultConnectionManager());
			log.info("Connections manager initialization, completed");
			// ---------------------------------------------------------------------
			
			try {
				// m.veroni, utilizzato singletone per memorizzare keystore
				String keyStoreClassName = config.getServletContext().getInitParameter("keyStore");
				if (XString.isNotBlankNullTrim(keyStoreClassName)) {
					log.info("KeyStore <" + keyStoreClassName + "> initialitation ");
					KeyStorage.getInstance().inizialize(keyStoreClassName);
				} else {
					log.info("KeyStore non initialized, not found web.xml <param-name>keystore</param-name> with valid <param-value>");
				}
			} catch (Exception e) {
				log.error("KeyStore non initialized, exception found: "+e);
			}
			// ---------------------------------------------------------------------
			
			try {
				// Inizializzazione delle statistiche del portale
				String logAnalyzerFileName = config.getServletContext().getInitParameter("logAnalyzerFile");
				if (XString.isNotBlankNullTrim(logAnalyzerFileName)) {
					log.info("Analyzer initialization with file '" + logAnalyzerFileName + "'");
					if (new File(logAnalyzerFileName).exists()) {
						LogAnalyzerProperties.getInstance().initialize(logAnalyzerFileName);
					} else {
						log.error("No configuration file found for Analyzer: '" + logAnalyzerFileName + "'");
					}
				} else {
					log.info("Analyzer not active, not found web.xml <param-name>logAnalyzerFile</param-name> with valid <param-value>");
				}
			} catch (Exception e) {
				log.error("Analyzer non initialized, exception found: "+e);
			}
			// ---------------------------------------------------------------------
			
			/* 
			 * Chiamata utilizzata per caricare il singleton ed effettuare la prima registrazione al gestore cluster
			 * Il sistema registra se stesso nel "registro" dei cluster 
			 * 
			 * - Questo servizio dipende da DatabaseManager
			 */		
			if (log.isDebugEnabled()) log.debug("Create instance of " + ApplicationClusterInfo.class + " service");
			
			// Parametro inserito nella configurazione su file portal_settings.xml
			String mask = Parameters.getInstance().get(Parameters.CLUSTER_MASK,"${hostname}");
			String backplane_mask = Parameters.getInstance().get(Parameters.CLUSTER_BACKPLANE_MASK);
			
			ApplicationClusterInfo clusterInfo = ApplicationClusterInfo.getInstance().init(config, mask, backplane_mask);
			// ---------------------------------------------------------------------
			
			// Configurazione PostOffice
			if (parameters.getBoolean(PostOffice.MAIL_ENABLE, true)) {
				log.info("Configuration of 'PostOffice' service");				
				PostOffice postOffice = PostOffice.getInstance();
				postOffice.setup(parameters);
			} else {
				log.warn("No 'PostOffice' service configurated");	
			}
			// ---------------------------------------------------------------------
			
			
			// Configurazione componente S4 - 08/10/2010
			
			if (parameters.getBoolean(S4Config.S4_ENABLE, false)) {
				log.info("Module 'S4Engine' is enable, configuration");
				
				// Rettifica della configurazione
				if (parameters.get(S4Config.S4_CONTEXT)==null) {
					if (log.isDebugEnabled()) log.debug(S4Config.class.getSimpleName()+" autoconfigure '"+S4Config.S4_CONTEXT+"'");
					parameters.put(S4Config.S4_CONTEXT, parameters.getChannelInfo().getContextName());
				}
				if (parameters.get(S4Config.S4_OWNER)==null) {
					if (log.isDebugEnabled()) log.debug(S4Config.class.getSimpleName()+" autoconfigure '"+S4Config.S4_OWNER+"'");
					parameters.put(S4Config.S4_OWNER, clusterInfo.getHostName());
				}
				if (parameters.get(S4Config.S4_BUCKET)==null) {
					if (log.isDebugEnabled()) log.debug(S4Config.class.getSimpleName()+" autoconfigure '"+S4Config.S4_BUCKET+"'");
					parameters.put(S4Config.S4_BUCKET, parameters.getChannelInfo().getDomainName());
				}	
				
				S4Config s4config = new S4Config(parameters, DatabaseManager.getInstance().getApplicationConnectionManager());
				try {
					S4Engine.getInstance(s4config);
				} catch (S4Exception e) {
					log.fatal("GRAVE: error configuring 'S4Engine'", e);
				}		
								
			} else {
				log.info("'S4Engine' is disable (only lock function enabled)");
			}
			// ---------------------------------------------------------------------
			
			
			/* m.veroni, utilizzato singletone
			   Inizializzazione e avvio del task manager */
			if (parameters.getBoolean(PortalSettings.TIMERTASK_ENABLE, true)) {
				log.info("Starting timer-task manager");
				TaskManager.getInstance().startThreadsOnStartup(config);
			} else {
				log.info("Timer-task manager disabled");
			}
			// ---------------------------------------------------------------------
			
			log.info("WebApp inizialization completed with success (super)");
			// ---------------------------------------------------------------------
			
			// m.veroni, Inserimento di informazioni utili nel log
			Version versione = Version.getInstance();
			String installation = parameters.get(Parameters.APPLICATION_ENVIRONMENT);
			String hostname = ApplicationClusterInfo.getInstance().getHostNameAddress();
			log.info("Start: " + parameters.getChannelInfo().getApplicationTitle() + " - Version " + versione.toString() + " (" + installation + ")" + (hostname != null ? " on host " + hostname : ""));
			// ---------------------------------------------------------------------
			
			// Invio email per tracciamento riavvio
			trackingStartWebservice(hostname);
			
		} catch (Exception e) {
			if (log == null) {
				System.out.println("GRAVE: Unable to complete the initialization of the WebApp");
				e.printStackTrace(System.out);
			} else {
				log.fatal("GRAVE: Unable to complete the initialization of the WebApp", e);
			}
			throw new ServletException(e);
		} finally {
			log.info("--------------------------------------------------------------------------------");
		}
		return;
	}

	
	
	public void destroy() {
		Parameters parameters = Parameters.getInstance();
		log.info("Destroing " + parameters.getChannelInfo().getApplicationTitle() + "....");
		
		// Registrazione della chiusura dell'applicazione
		ApplicationClusterInfo.getInstance().endLife();
		
		
		// Stop dei servizi S4
		S4Engine engine = S4Engine.getInstance();
		if (engine.isConfigured()) {
			if (engine.getCleaner()!=null) {
				engine.getCleaner().stop();
			}
		}
		
		/* Stop di tutti i timertask */
		if (parameters.getBoolean(PortalSettings.TIMERTASK_ENABLE, true)) {
			log.info("Starting timer-task manager");
			TaskManager.getInstance().stopAllThreads();
		} else {
			log.info("Timer-task manager disabled");
		}
		// ---------------------------------------------------------------------
		
		// Deregistrazione del datasorise dal Tomcat
		DatabaseManager.getInstance().unregisterPooledDataSource();

		
		log.info("Application " + Parameters.getInstance().getChannelInfo().getApplicationTitle() + " destroyed!\n" +
		         "-[STOP ]-------------------------------------------------------------------------------" +
				 "\n" +
				 "\n");
		
		super.destroy();
	}

	
	
	private void trackingStartWebservice(String hostname) {
		Parameters parameters = Parameters.getInstance();

		boolean send = parameters.getBoolean(PostOffice.MAIL_ENABLE, false) && 
					   parameters.getBoolean(PortalSettings.MAIL_START_ENABLE, false); 
		if (!send) {
			send = ApplicationClusterInfo.getInstance().changedVersion();
			if (send) log.warn("Changed version, automatic email send!");
		}
		if (!send) return;
			
		log.info("Sending information mail for service start");
		
			
		
		// Recupero l'ora di sistema di Oracle
		EnterpriseCalendar dataSystem = EnterpriseCalendar.now();

		Version version = Version.getInstance();
		String installation = parameters.get(PortalSettings.APPLICATION_ENVIRONMENT);
		
		String domain = parameters.get(PortalSettings.APPLICATION_DOMAIN);
		String context = parameters.get(PortalSettings.APPLICATION_CONTEXT_NAME);
		String cluster_mask = parameters.get(PortalSettings.CLUSTER_MASK);
		String base_url = parameters.get(PortalSettings.BASE_URL);
		
		PostmanPat pat = new PostmanPat();
		pat.setSubject("Start context");
		pat.addMessage("Service is alive");
		pat.addMessage("");
		pat.addMessage("Version: " + version.getVersion() + " (build "+version.getBuild()+") " + installation);
		if (hostname != null) pat.addMessage("Host: " + hostname);
		if (domain != null) pat.addMessage("Domain: " + domain);
		if (context != null) pat.addMessage("Context: " + context);
		pat.addMessage("");
		pat.addMessage("Timestamp (system): " + dataSystem.toStringUTC());
		pat.addMessage("");
		if (cluster_mask != null) pat.addMessage("Mask: " + cluster_mask);
		pat.addMessage("Real path:" +parameters.getWebAppRootDirRealPath());
		if (base_url != null) pat.addMessage("Url: " + base_url);
		pat.addMessage("Real path:" +parameters.getWebAppRootDirRealPath());
		pat.addMessage("");
		pat.send();		
				
	}

}
