package it.softecspa.mvc.businessobject;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.portal.Parameters;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.apache.log4j.Logger;

import com.mchange.v2.c3p0.ComboPooledDataSource;
import com.mchange.v2.c3p0.DataSources;
import com.mchange.v2.c3p0.PooledDataSource;
import com.mchange.v2.c3p0.management.ActiveManagementCoordinator;

public class DatabaseManager {

	private static Logger log = Logger.getLogger(DatabaseManager.class);
	private static DatabaseManager instance;
	
	private Map<String, StandardConnectionManager> pooledDataSource;

	// Datasource di default
	private DataSource defaultDataSource;
	private String defaultDataSourceName;

	// Datasource dedicato al CMS
	private String cmsDataSourceName;
	private String cmsReadonlyDataSourceName;

	// Datasource applicativi
	private List<String> applicationDataSourceName;
	private List<String> applicationReadonlyDataSourceName;

	
	
	private DatabaseManager() {
		super();
		synchronized (DatabaseManager.class) {
			if (instance == null) {
				instance = this;
			}
		}
	}

	public static DatabaseManager getInstance() {
		synchronized (DatabaseManager.class) {
			if (instance == null)
				new DatabaseManager();
		}
		return instance;
	}

	public void inizialize(Parameters parameters) throws Exception {
		boolean setupdefault = true;
		
		pooledDataSource = new HashMap<String, StandardConnectionManager>();

		// DataSource CMS read/write 
		String dsCMS = parameters.get(Parameters.DATASOURCE_CMS);
		if (XString.isNotBlankNullTrim(dsCMS)) {
			StandardDataSourceConfig sdsc = new StandardDataSourceConfig(dsCMS, false);
			addDataSource(sdsc, setupdefault);
			cmsDataSourceName = sdsc.getName();
			setupdefault = false;
			log.info("* add datasource '" + sdsc + "' as cms WRITABLE");
		}
		
		// DataSource CMS readonly
		String dsCMS_readonly = parameters.get(Parameters.DATASOURCE_CMS_READONLY);
		if (dsCMS_readonly != null) {
			StandardDataSourceConfig sdsc = new StandardDataSourceConfig(dsCMS_readonly, true);
			addDataSource(sdsc, setupdefault);
			cmsReadonlyDataSourceName = sdsc.getName();
			setupdefault = false;
			log.info("* add datasource '" + sdsc + "' as cms READONLY");
		}

		// DataSource APPLICATION read/write
		String dsApplication = parameters.get(Parameters.DATASOURCE_APPLICATION);
		if (XString.isNotBlankNullTrim(dsApplication)) {
			String[] lista = dsApplication.trim().split(",");
			for (String elemento : lista) {
				StandardDataSourceConfig sdsc = new StandardDataSourceConfig(elemento, false);
				addDataSource(sdsc, setupdefault);
				if (applicationDataSourceName == null)
					applicationDataSourceName = new ArrayList<String>();
				applicationDataSourceName.add(sdsc.getName());
				setupdefault = false;
				log.info("* add datasource '" + sdsc + "' as application WRITABLE");
			}
		}

		// DataSource APPLICATION readonly
		String dsApplication_readonly = parameters.get(Parameters.DATASOURCE_APPLICATION_READONLY);
		if (XString.isNotBlankNullTrim(dsApplication_readonly)) {
			String[] lista = dsApplication_readonly.trim().split(",");
			for (String elemento : lista) {
				StandardDataSourceConfig sdsc = new StandardDataSourceConfig(elemento, true);
				addDataSource(sdsc, false);
				if (applicationReadonlyDataSourceName == null)
					applicationReadonlyDataSourceName = new ArrayList<String>();				
				applicationReadonlyDataSourceName.add(sdsc.getName());
				log.info("* add datasource '" + sdsc + "' as application READONLY");
			}
		}

		log.info("* datasource '" + defaultDataSourceName + "' used as default");
	}

	public void addCMSDataSource(StandardDataSourceConfig sdsConfig, boolean useAsDefault) throws Exception {
		if (!pooledDataSource.containsKey(sdsConfig.getName())) {

			StandardConnectionManager scm = new StandardConnectionManager(sdsConfig.getName(), sdsConfig.getDbms());
			pooledDataSource.put(sdsConfig.getName(), scm);

			if (useAsDefault) {
				defaultDataSourceName = sdsConfig.getName();
			}
		}
	}

	private void addDataSource(StandardDataSourceConfig sdsConfig, boolean useAsDefault) throws Exception {
		if (!pooledDataSource.containsKey(sdsConfig.getName())) {

			StandardConnectionManager scm = new StandardConnectionManager(sdsConfig.getName(), sdsConfig.getDbms());
			pooledDataSource.put(sdsConfig.getName(), scm);

			if (useAsDefault) {
				defaultDataSourceName = sdsConfig.getName();
			}
		}

	}

	/**
	 * Restituisce una connessione dal DataSource di default
	 * 
	 * @return
	 * @throws Exception
	 */
	public Connection getConnection() throws SQLException {
		return getConnection(defaultDataSourceName);
	}

	/**
	 * Restituisce la connessione relativa al nome richiesto
	 * 
	 * @param dbName
	 * @return
	 * @throws Exception
	 */
	public Connection getConnection(String dbName) throws SQLException {
		Connection conn = ((StandardConnectionManager) pooledDataSource.get(dbName)).getConnection();
		conn.setAutoCommit(false);
		return conn;
	}

	/**
	 * Chiude la connessione richiesta
	 * 
	 * @param connection
	 */
	public static void closeConnection(Connection connection) {
		try {
			if (connection != null) {
				connection.close();
			}
		} catch (Throwable t) {
			log.warn("Errore di chiusura connessione", t);
		}
	}

	/**
	 * Chiude il PreparedStatement richiesto
	 * 
	 * @param pstmt
	 */
	public static void closeStatement(PreparedStatement pstmt) {
		try {
			if (pstmt != null)
				pstmt.close();
		} catch (Throwable t) {
			log.warn("Errore di chiusura PreparedStatement", t);
		}
	}

	/**
	 * Chiude lo Statement riciesto
	 * 
	 * @param stmt
	 */
	public static void closeStatement(Statement stmt) {
		try {
			if (stmt != null)
				stmt.close();
		} catch (Throwable t) {
			log.warn("Errore di chiusura Statement", t);
		}
	}

	/**
	 * Chiude il ResulSet richiesto
	 * 
	 * @param res
	 */
	public static void closeResultSet(ResultSet res) {
		try {
			if (res != null)
				res.close();
		} catch (Throwable t) {
			log.warn("Errore di chiusura ResultSet", t);
		}
	}

	/**
	 * Esegue un ROLLBACK sulla connessione desiderata
	 * 
	 * @param conn
	 */
	public static void rollbackConnection(Connection conn) {
		try {
			if (conn != null)
				conn.rollback();
		} catch (Throwable e) {
			log.warn("Errore di rollback della connessione", e);
		}
	}

	/**
	 * Restitutisce l'elenco delle StandardConnectionManagere
	 * 
	 * @return
	 */
	public Map<String, StandardConnectionManager> getDataSourcePool() {
		return pooledDataSource;
	}

	/**
	 * Restituisce il DataSource di default
	 * 
	 * @return
	 */
	public DataSource getDefaultDataSource() {
		return defaultDataSource;
	}

	/**
	 * Restituisce il nome del DataSource di default
	 * 
	 * @return
	 */
	public String getDefaultDataSourceName() {
		return defaultDataSourceName;
	}

	/**
	 * Restituisce il connection manager delle aconnessione di default
	 * 
	 * @return
	 */
	public ConnectionManager getDefaultConnectionManager() {
		return pooledDataSource.get(defaultDataSourceName);
	}

	/**
	 * Restituisce il ConnectionManager per nome
	 * 
	 * @param dbName
	 * @return
	 */
	public ConnectionManager getConnectionManager(String dbName) {
		return pooledDataSource.get(dbName);
	}

	/**
	 * Restituisce il ConnectionManager utilizzato per il CMS
	 * 
	 * @param readonly
	 * @return
	 */
	public ConnectionManager getCMSConnectionManager(boolean readonly) {
		if (readonly && cmsReadonlyDataSourceName != null) {
			// Sola lettura
			return pooledDataSource.get(cmsReadonlyDataSourceName);
		}
		// Lettura/Scrittura
		return pooledDataSource.get(cmsDataSourceName);
	}

	/**
	 * Restituisce il ConnectionManager utilizzato per il CMS in sola lettura
	 * 
	 * @return
	 */
	public ConnectionManager getCMSConnectionManagerREADONLY() {
		return getCMSConnectionManager(true);
	}

	/**
	 * Restituisce il ConnectionManager utilizzato per il CMS in
	 * lettura/scrittura
	 * 
	 * @return
	 */
	public ConnectionManager getCMSConnectionManager() {
		return getCMSConnectionManager(false);
	}

	/**
	 * Restituisce il ConnectionManager all'indice richiesto recuperandolo dalla
	 * lista dei connection manager applicativi
	 * 
	 * @param index
	 * @param readonly
	 * @return
	 */
	public ConnectionManager getApplicationConnectionManager(int index, boolean readonly) {
		if (readonly && applicationReadonlyDataSourceName != null && 
				        applicationReadonlyDataSourceName.size() > index && 
				        applicationReadonlyDataSourceName.get(index) != null) {
			// Sola lettura
			return pooledDataSource.get(applicationReadonlyDataSourceName.get(index));
		}

		// Lettura/Scrittura
		return pooledDataSource.get(applicationDataSourceName.get(index));
	}

	/**
	 * Restituisce il ConnectionManagere applicativo con il filtro
	 * lettura/scrittura richiesto
	 * 
	 * @param readonly
	 * @return
	 */
	public ConnectionManager getApplicationConnectionManager(boolean readonly) {
		return getApplicationConnectionManager(0, readonly);
	}

	/**
	 * Restituisce il ConnectionManagere applicativo con il filtro sola lettura
	 * 
	 * @return
	 */
	public ConnectionManager getApplicationConnectionManagerREADONLY() {
		return getApplicationConnectionManager(0, true);
	}

	/**
	 * Restituisce il ConnectionManagere applicativo in lettura/scrittura
	 * 
	 * @return
	 */
	public ConnectionManager getApplicationConnectionManager() {
		return getApplicationConnectionManager(0, false);
	}

	/**
	 * Effetta la deregistrazione dal JMX di Tomcat del C3P0 Contestualmente fa
	 * una chiusura manuale del connection pool
	 */
	public synchronized void unregisterPooledDataSource() {
		ActiveManagementCoordinator amc = null;
		try {
			log.info("Unregister C3P0Registry ComboPooledDataSource mbean");
			amc = new ActiveManagementCoordinator();
		} catch (Exception e) {
			log.error("Problem in costructor of ActiveManagementCoordinator", e);
		}

		if (amc != null) {
			if (pooledDataSource != null) {
				for (Iterator<String> keys = pooledDataSource.keySet().iterator(); keys.hasNext();) {
					String key = keys.next();
					DataSource ds = ((StandardConnectionManager) pooledDataSource.get(key)).getDataSource();

					if (ds instanceof PooledDataSource || ds instanceof ComboPooledDataSource) {
						log.info("Found c3p0 ComboPooledDataSource with name '" + key + "' to unregister!");
						PooledDataSource pds = (PooledDataSource) ds;
						try {
							amc.attemptUnmanagePooledDataSource(pds);
						} catch (Exception e) {
							log.error("Problem call attemptUnmanagePooledDataSource('" + key + "')", e);
						}

						/*
						 * Workaround per risolvere il problema delle
						 * connessioni pending al riavvio del contesto m.veroni,
						 * 2013-03-05
						 */
						try {
							log.info("Destroy ComboPooledDataSource!");
							DataSources.destroy(ds);
						} catch (SQLException e) {
							log.error("Problem destroing ComboPooledDataSource", e);
						}
					} else {
						log.info("DataSource '" + key + "' is not c3p0 PooledDataSource!");
					}
				}

			}

		}

	}

}
