// Source File Name:   User.java

package it.softecspa.mvc.businessobject;

import it.softecspa.database.dbconnect.NoRecordFoundException;
import it.softecspa.jwebber.security.CostantsStandardPolicy;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.mvc.exceptions.AppSecurityException;
import it.softecspa.portal.Parameters;
import it.softecspa.portal.db.AbstractUtentePortale;
import it.softecspa.portal.db.UtentePolicies;

import java.lang.reflect.Constructor;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.log4j.Logger;

public abstract class User {

	protected Logger logger = Logger.getLogger(getClass());
	
	
	public static final String ANON_PROFILE = "ANON";
	public static final String UNKNWON_USERNAME = "unknwown";
	
	
	protected int idUtente;
	protected String username;
	protected String password;
	
	protected String idProfilo;	
	protected String stato;
	
	protected HashMap<String, Object> customInfo;
	protected List<String> policies;
	
	protected String language;
	protected Locale locale;

	
	
	protected User() {
		super();
	}
	
	
	/**
	 * Imposta il valore di idUtente
	 */
	public void setIdUtente(String idUtente) {
		this.idUtente = Integer.parseInt(idUtente);
	}

	/**
	 * Restituisce il valore di idUtente
	 */
	public int getIdUtente() {
		return idUtente;
	}
	
	
	public String getIdUtenteAsString() {
		return Integer.toString(idUtente);
	}

	/**
	 * Imposta la lingua
	 */
	public void setLanguage(String language) {
		this.language = language;

		// Aggiorno il locale di conseguenza
		if (XString.isNotBlankNull(language)) {
			this.language = this.language.trim();
			this.locale = new Locale(this.language, this.language.toUpperCase());
		} else {	
			this.language = Parameters.getInstance().getChannelInfo().getDefaultLanguage();
			this.locale = Parameters.getInstance().getChannelInfo().getDefaultLocale();
		}
		
		
		
	}

	/**
	 * Restituisce la lingua
	 */
	public String getLanguage() {
		return language;
	}
	
	
	/**
	 * Restituisce il Locale utilizzato dall'utente
	 */
	public Locale getLocale() {
		return this.locale;
	}
	
	
	
	public abstract void setPassword(String s) throws Exception;
	
	/**
	 * Restituisce il valore di password
	 */
	public String getPassword() {
		return password;
	}

	
	
	/**
	 * Imposta il valore di userName
	 */
	public void setUserName(String userName) {
		this.username = userName;
	}

	/**
	 * Restituisce il valore di idUtente
	 */
	public String getUserName() {
		return username;
	}
	
	/**
	 * Restituisce l'id del profilo dell'utente
	 */
	public String getIdProfilo() {
		return idProfilo;
	}
	

	public abstract String getDenominazione();

	
	

	
	/**
	 * Restituisce il valore di stato
	 */
	public String getStato() {
		return stato;
	}
	
	

	
	// ...utillizzo particolare della variabile
	private Boolean sa;
	
	/**
	 * @deprecated
	 * @param name
	 * @return
	 */
	public String getPolicy(String name) {
		return hasPolicy(name)?name:null;		
	}
		
	
	public List<String> getPolicies() {
		if (policies==null) {
			Connection connection = null;
			try {
				connection = DatabaseManager.getInstance().getConnection();
				loadUserPolicy(connection);			
			} catch (NoRecordFoundException e) {
				logger.warn("No record found for profile '"+this.idProfilo+"'");
				return null;
			} catch (Exception e) {
				logger.warn("Unhandled exception in check policy",e);
				// ...utilizzer il sistema di backup
			} finally {
				DatabaseManager.closeConnection(connection);				
			}
			
			if (logger.isDebugEnabled()) logger.debug("Utente con "+policies.size()+" policy");	
			Parameters parameters = Parameters.getInstance(); 
				
			// Verifico se il profilo  SA
			if (XString.isNotBlankNull(parameters.get(CostantsStandardPolicy.SA))) {
				sa = Boolean.valueOf(policies.contains(parameters.get(CostantsStandardPolicy.SA)));
			} else {
				sa = Boolean.FALSE;
			}
		}
		return policies;
	}
	
	
	/**
	 * Controlla se un utente possiede una certa policy
	 * Ottimizzata da m.veroni 
	 * @param policy
	 * @return
	 */
	public boolean hasPolicy(String ... policy)  {
		getPolicies();
		
		if (policies!=null) {
			/* 
			 * m.veroni
			 * Se l'utente ha il profilo SA puo fare TUTTO e quindi ha tutte le policy
			 */
			if (sa==null) sa = Boolean.FALSE;			
			if (sa.booleanValue()) return true;		
			
			for (String name : policy) {
				// Se una della policy  valida restituisco TRUE
				if (policies.contains(name)) return true;
			}
		
		} else {
			// ....sistema di backup!
			logger.error("Check policy with backup system");
			
			/* m.veroni
			 * Se l'utente ha il profilo SA puo fare TUTTO e quindi ha tutte le policy
			 */
			if (sa==null) {
				Parameters parameters = Parameters.getInstance(); 
				for (String name : policy) {
					try {
						new UtentePolicies(parameters.get(CostantsStandardPolicy.SA), name);
						sa = Boolean.TRUE;
						if (sa.booleanValue()) return true;	
					} catch (NoRecordFoundException e) {
						logger.warn("No record found for profile '"+this.idProfilo+"', policy '"+parameters.get(CostantsStandardPolicy.SA)+"'");
						sa = Boolean.FALSE;
					} catch (Exception e) {
						logger.error("Unhandled exception in check policy",e);
						sa = Boolean.FALSE;
					}		
				}
			}
			if (sa.booleanValue()) return true;		
			
			for (String name : policy) {
				try {
					new UtentePolicies(this.idProfilo, name);
					return true;
				} catch (NoRecordFoundException e) {
					logger.warn("No record found for profile '"+this.idProfilo+"', policy '"+name+"'");
					return false;
				} catch (Exception e) {
					logger.error("Unhandled exception in check policy",e);
					return false;
				}
			}
		}
		
		return false;
	}
	

	// Carica le policy legate al profilo
	protected abstract void loadUserPolicy(Connection connection) throws SQLException;


	
	
	
	/**
	 * Aggiunge informazioniu custom all'utente in modo che possano essere condivise in sessione
	 * @param key
	 * @param value
	 */
	public void addCustomInfo(String key, Object value) {
		getCustomInfoList().put(key, value);
	}

	/**
	 * Recupera l'informazione custom richiesta
	 * @param key
	 * @return
	 */
	public Object getCustomInfo(String key) {
		return getCustomInfoList().get(key);
	}

	/**
	 * Recupera la lista delle informazioni custom associate all'utente
	 * @return
	 */
	public Map<String, Object> getCustomInfoList() {
		if (customInfo==null) customInfo = new HashMap<String, Object>();
		return customInfo;
	}
	
	
	
	public final static String UTENTE_PORTALE = "utente.portale.cache";
	
	/**
	 * Restituisce l'istanza dell'utente del portale
	 * Alla prima richiesta  fatto il caricamento dal database
	 * @return
	 * @throws AppSecurityException
	 * @throws SQLException
	 */
	public <K extends AbstractUtentePortale> K getUtentePortale(Class<K> clazz) throws AppSecurityException, SQLException {
		return getUtentePortale(clazz, false);
	}
	
	@SuppressWarnings("unchecked")
	public <K extends AbstractUtentePortale> K getUtentePortale(Class<K> clazz, boolean forcereload) throws AppSecurityException, SQLException {
		if (forcereload) {
			getCustomInfoList().remove(UTENTE_PORTALE);
		}
		
		K utente = (K)this.getCustomInfo(UTENTE_PORTALE);
		if (utente!=null) return utente;
		
		utente = getActiveInstance(clazz);
		if (utente==null) return null;
		
		addCustomInfo(UTENTE_PORTALE, utente);
		return utente;
	}
	
	
	@SuppressWarnings("unchecked")
	public <K extends AbstractUtentePortale> K getActiveInstance(Class<K> clazz) throws SQLException, AppSecurityException {
		if (logger.isDebugEnabled()) logger.debug("Create istance of "+clazz);
		Integer id = null;
		try {
			id = new Integer(getIdUtente());
		} catch (NumberFormatException e) {
			id = new Integer(-1);
		}
					
		
		Constructor<?> costructor=null;
		
		costructor = findCostructor(clazz, int.class);
		if (costructor==null) {
			costructor = findCostructor(clazz, Integer.class);
		}
		if (costructor==null) {
			costructor = findCostructor(clazz, String.class);
		}
		if (costructor==null) {
			throw new AppSecurityException("Nessun costruttore trovato per estrarre UtentePortale");
		}
		
		
		K utente = null;
		try {
			utente = (K)costructor.newInstance(id);			
		} catch (Exception e) {
			if (e.getCause() instanceof NoRecordFoundException) {
				throw new AppSecurityException("Utente non valido o non trovato");
			}
			throw new AppSecurityException("Exception with class costructor ", e);
		} 
		
		return utente;		
	}
	
	
	private Constructor<?> findCostructor(Class<?> clazz, Class<?> entrypoint) {
		try {
			return  clazz.getDeclaredConstructor(entrypoint);
			
		} catch (SecurityException e) {
			logger.warn("Errore in accesso al costruttore "+clazz+"("+entrypoint+" id)",e);
		} catch (NoSuchMethodException e) {
			logger.warn("Costruttore non trovato:  "+clazz+"("+entrypoint+" id)",e);
		}
		return null;
	}
	
	
	
	
}
