package it.softecspa.portal;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.services.PortalSettings;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.kahuna.util.http.RequestParameterException;
import it.softecspa.mvc.businessobject.DatabaseManager;
import it.softecspa.mvc.businessobject.StandardConnectionManager;

import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.sql.DataSource;

import org.apache.log4j.Appender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;

import com.mchange.v2.c3p0.ComboPooledDataSource;

/**
 * Servlet standard di monitoraggio
 * 
 * @author m.veroni
 * 
 */
@SuppressWarnings("serial")
public class Monitor extends HttpServlet {

	Logger log = Logger.getLogger(Monitor.class);
	Parameters parameters = Parameters.getInstance();

	// TODO implementare query configurabili

	public class RequestParameter extends it.softecspa.kahuna.util.http.RequestParameter {

		public final static String XML 		= "xml";
		public final static String XML_FULL	= "xml-full";
		public final static String NAGIOS 	= "nagios";
		
		// Gestione dinamica LOG4J
		public final static String LOG_STATEMENT = "log";
		public final static String CATEGORY = "category";
		public final static String LEVEL = "level";

		
		//
		private boolean bXML;
		private boolean bXML_full;
		private boolean bNAGIOS;
		private String logStatement;
		private String category;
		private String level;

		
		public RequestParameter(HttpServletRequest request) throws RequestParameterException {
			super(request);
		}

		@Override
		public void validate() throws RequestParameterException {
			setXML(is(XML));
			setXMLfull(is(XML_FULL));
			setNAGIOS(is(NAGIOS));
			//
			setLogStatement(getRequest(LOG_STATEMENT, String.class));
			setCategory(getRequest(CATEGORY, String.class));
			setLevel(getRequest(LEVEL, String.class));			
		}

		public boolean isXML() {
			return bXML;
		}

		private void setXML(boolean bXML) {
			this.bXML = bXML;
		}

		public boolean isXMLfull() {
			return bXML_full;
		}

		private void setXMLfull(boolean bXML_full) {
			this.bXML_full = bXML_full;
		}
		
		public boolean isNAGIOS() {
			return bNAGIOS;
		}

		private void setNAGIOS(boolean bNAGIOS) {
			this.bNAGIOS = bNAGIOS;
		}
		
		public String getCategory() {
			return category;
		}

		public void setCategory(String category) {
			this.category = category;
		}

		
		public String getLevel() {
			return level;
		}

		private void setLevel(String level) {
			this.level = level;
		}

		public String getLogStatement() {
			return logStatement;
		}

		private void setLogStatement(String statement) {
			this.logStatement = statement;
		}

		

	}

	protected void doGet(HttpServletRequest request, HttpServletResponse response) {
		doPost(request, response);
	}

	
	
	// This method recursively visits all thread groups under `group'.
	public String visit(ThreadGroup group, int level) {
		String xml = "<THREAD-GROUP name=\"" + group.getName() + "\" >";

		// Get threads in `group'
		int numThreads = group.activeCount();
		Thread[] threads = new Thread[numThreads * 2];

		numThreads = group.enumerate(threads, false);
		// Enumerate each thread in `group'
		for (int i = 0; i < numThreads; i++) {
			// Get thread
			Thread thread = threads[i];
			xml += "<THREAD id=\"" + thread.getId() + "\"" + " state=\"" + thread.getState() + "\"" + " priority=\"" + thread.getPriority() + "\"" + " is-alive=\"" + thread.isAlive() + "\"" + " is-daemon=\"" + thread.isDaemon() + "\"" + " >"
					+ thread.getName() + "</THREAD>";
		}

		// Get thread subgroups of `group'
		int numGroups = group.activeGroupCount();
		ThreadGroup[] groups = new ThreadGroup[numGroups * 2];
		numGroups = group.enumerate(groups, false);
		// Recursively visit each subgroup
		for (int i = 0; i < numGroups; i++) {
			xml += visit(groups[i], level + 1);
		}

		xml += "</THREAD-GROUP>";
		return xml;
	}

	private String datasourceXML() {
		String xml = "";
		Map<String, StandardConnectionManager> pooledDataSource = DatabaseManager.getInstance().getDataSourcePool();
		if (pooledDataSource != null) {
			xml += "<DATASOURCES>";
			for (Iterator<String> keys = pooledDataSource.keySet().iterator(); keys.hasNext();) {
				String key = keys.next();
				DataSource ds = ((StandardConnectionManager) pooledDataSource.get(key)).getDataSource();
				if (ds instanceof ComboPooledDataSource) {
					ComboPooledDataSource pds = (ComboPooledDataSource) ds;
					try {
						xml += "<DATASOURCE type=\"c3p0\"" + " name=\"" + key + "\"" + " identity-token=\"" + pds.getIdentityToken() + "\"" + " >" + 
									"<CONNETIONS max=\"" + pds.getMaxPoolSize() +"\" increment=\"" + pds.getAcquireIncrement() +"\" used=\"" + pds.getNumConnectionsAllUsers() + "\"" + " busy=\""+ pds.getNumBusyConnectionsAllUsers() + "\"" + " idle=\"" + pds.getNumIdleConnectionsAllUsers() + "\"" + " orphan=\"" + pds.getNumUnclosedOrphanedConnectionsAllUsers() + "\"" + " />" + 
									"<STATEMENT-CACHE number=\""+ pds.getStatementCacheNumStatementsAllUsers() + "\"" + " connection-with=\"" + pds.getStatementCacheNumConnectionsWithCachedStatementsAllUsers() + "\"" + " checked-out=\""+ pds.getStatementCacheNumCheckedOutStatementsAllUsers() + "\"" + " />" + 
									"<THREADS-POOL size=\"" + pds.getThreadPoolSize() + "\"" + " active=\"" + pds.getThreadPoolNumActiveThreads() + "\"" + " idle=\"" + pds.getThreadPoolNumIdleThreads() + "\"" + " pending=\"" + pds.getThreadPoolNumTasksPending() + "\"" + " />";
					} catch (SQLException e) {
						xml += "<DATASOURCE type=\"c3p0\"" + " name=\"" + key + "\">" + "<EXCEPTION value=\"" + e.toString() + "\" />";
					}
					xml += "</DATASOURCE>";
				} else {
					xml += "<DATASOURCE type=\"" + ds.getClass().getSimpleName() + "\"" + " name=\"" + key + "\"" + " />";
				}
			}

		}
		xml += "</DATASOURCES>";
		return xml;
	}

	protected void doPost(HttpServletRequest request, HttpServletResponse response) {
		RequestParameter input = null;
		try {
			input = new RequestParameter(request);
		} catch (RequestParameterException e) {
			/* Nessuna operazione */
		}

		response.setHeader("Cache-Control", "no-cache, no-store, must-revalidate"); // HTTP 1.1.
		response.setHeader("Pragma", "no-cache"); // HTTP 1.0.
		response.setDateHeader("Expires", 0); // Proxies.
		
		if (input != null) {
			if (input.isNAGIOS()) {
				// Restituisce la risposta in formato XML completo
				responseNAGIOS(response);
			
			} else if (input.isXML()) {
				// Restituisce la risposta in formato XML ridotto
				responseXML(response, false);
			
			} else if (input.isXMLfull()) {
				// Restituisce la risposta in formato XML completo
				responseXML(response, true);	
					
			} else if (input.getLogStatement()!=null) {
				// Cambia il livello del CATEGORY configurato nel LOG4J riportandolo ai valori di default
				logStatement(response, input.getLogStatement());
					
			} else if (input.getCategory()!=null) {
				// Cambia il livello dell CATEGORY configurato nel LOG4J
				changeLevel(response, input.getCategory(), input.getLevel());
				
			} else {
				responseStandard(response);
			}

		} else {
			response(response, "KO", "text/html");
		}

	}

	
	/**
	 * Fa alcune vrifiche interne compreso la connettivit versio i database configurati
	 * @param response
	 */
	private void responseStandard(HttpServletResponse response) {
		if (checkDatabase(response)) {
			response(response, "OK", "text/plain");
		} else {
			response(response, "KO", "text/pain");
		}
		
	}

	private boolean checkDatabase(HttpServletResponse response) {
		// TODO Auto-generated method stub
		boolean error = false;
		
		DatabaseManager dbManager = DatabaseManager.getInstance();
		// Faccio una query di routine su tutti i datasource
		int i=1;
		for (String name : dbManager.getDataSourcePool().keySet()) {
			if (error) break;
			
			if (log.isDebugEnabled()) log.debug("Check integrity for datatasource '"+name+"'");
			response.setHeader("X-datasource-"+i+"-name", name);
			
			java.sql.Connection connection = null;
			Statement stmt = null;
			ResultSet res = null;
			String query = null;
			try {
				connection = dbManager.getConnection(name);
				query = "select sysdate() from dual";
				stmt = connection.createStatement();
				res = stmt.executeQuery(query);
				if (res.next()) {
					EnterpriseCalendar timestamp = new EnterpriseCalendar(res.getTimestamp(1));
					if (log.isDebugEnabled()) 
						log.debug("MONITOR: check integrity for datatasource '"+name+"'; server timestamp is "+timestamp.formatISO8601zulu());
					response.setHeader("X-datasource-"+i+"-timestamp", timestamp.formatRFC1036());
				}
			} catch (SQLException e) {
				error = true;
				log.error("MONITOR: error in check integrityfor datatasource '"+name+"': "+e.toString());
				response.setHeader("X-datasource-"+i+"-error", e.toString());
			} finally {
				DatabaseManager.closeResultSet(res);
				DatabaseManager.closeStatement(stmt);
				DatabaseManager.closeConnection(connection);
			}
			
			i++;
		}
		
		return !error;
	}


	private void logStatement(HttpServletResponse response, String logStatement) {
		if ("reset".equalsIgnoreCase(logStatement)) {
			
			String file = Parameters.getInstance().get(PortalSettings.LOG4J);
			try {
				DOMConfigurator.configure(file);
			} catch (Exception e) {
				response(response, e.getMessage(), "text/html");	
				return;
			}
			log.warn("Logger reset to default value present in file '"+file+"'");
			response(response, "File '"+file+"' reloaded!", "text/html");
		
		} else {
			response(response, "Specify valid value for statement 'log' ('reset')", "text/html");
		}
		
	}



	@SuppressWarnings("unchecked")
	private void changeLevel(HttpServletResponse response, String _category, String _level) {
		
		if (_level!=null) {
			Level levelNuovo = Level.toLevel(_level.toUpperCase());
					
			Logger log = Logger.getLogger(_category);
			Level level = log.getLevel();
			if (level==null) {
				// ...il log specifico non esiste
				log.warn("Logger '"+log.getName()+"' do not exists");
			} else {
				log.warn("Changing log level for category '"+log.getName()+"': "+level.toString()+" > "+levelNuovo.toString());	
				log.setLevel(levelNuovo);
			}
			
		}	
			
		// Stampo lo stato del log richiesto
		Logger log = Logger.getLogger(_category);
		Level level = log.getLevel();
		
		StringBuilder xml = new StringBuilder(); 
		xml.append("<LOGGER>");
		if (level==null) {
			// Il log non esiste, stampo quello di ROOT
			log = Logger.getRootLogger();
			level = log.getLevel();			
			
			xml.append("<ROOT>"); 
			xml.append("<PRIORITY value='").append(level.toString()).append("'/>"); 
			for (Enumeration<Appender> e = log.getAllAppenders(); e.hasMoreElements() ;) {
				Appender appender = e.nextElement();
				xml.append("<APPENDER-REF ref='").append(appender.getName()).append("'/>"); 
			}
			xml.append("</ROOT>");
			
		} else {
			// Category individuato
			xml.append("<CATEGORY name='").append(_category).append("' additivity='"+log.getAdditivity()+"'>"); 
			xml.append("<PRIORITY value='").append(level.toString()).append("' />"); 
			for (Enumeration<Appender> e = log.getAllAppenders(); e.hasMoreElements() ;) {
				Appender appender = e.nextElement();
				xml.append("<APPENDER-REF ref='").append(appender.getName()).append("'/>"); 
			}
			xml.append("</CATEGORY>");			
		}
		xml.append("</LOGGER>");
			
		response(response, xml.toString(), "text/xml");			
		
	}

	
	
	private enum Weight {
		UNKNOWN, OK, WARNING, CRITICAL;
	}

	
	private void responseNAGIOS(HttpServletResponse response) {
		/*
		http://www.mchange.com/projects/c3p0/apidocs/index.html
		http://www.mchange.com/projects/c3p0/apidocs/com/mchange/v2/c3p0/PooledDataSource.html
		*/

		/*
			Esempio Nagios su pi righe
			
			valutazione - text01 | graph01
			
						
			valutazione - text01 | graph01
			text02 | graph02
			
			
			valutazione - text01 | graph01
			text02
			text03
			text04 | graph02
			graph03
			graph04
		
		*/
		String text = "";
		Weight weight = Weight.OK;
				
		Map<String, StandardConnectionManager> pooledDataSource = DatabaseManager.getInstance().getDataSourcePool();
		if (pooledDataSource != null) {
			
			int size = pooledDataSource.keySet().size();
			String[] line = new String[size];
			String[] graph = new String[size];
			
			int i=0;
			for (Iterator<String> keys = pooledDataSource.keySet().iterator(); keys.hasNext();) {
				String key = keys.next();
				DataSource ds = ((StandardConnectionManager) pooledDataSource.get(key)).getDataSource();
				Weight weight_n = Weight.UNKNOWN;
				
				
				line[i] = "DataSource";
				
				if (ds instanceof ComboPooledDataSource ) {
					ComboPooledDataSource pds = (ComboPooledDataSource) ds;
					try {
						int max = pds.getMaxPoolSize();
						
						int busy = pds.getNumBusyConnectionsAllUsers() + pds.getNumIdleConnectionsAllUsers();
						int lock = pds.getNumUnclosedOrphanedConnectionsAllUsers();
						int lev_warn = (int)((double)max/(double)100*(double)75);
						int lev_crit = (int)((double)max/(double)100*(double)90);
						
						if (busy>=lev_crit) {
							weight_n = Weight.CRITICAL;
						} else if (busy>=lev_warn) {
							weight_n = Weight.WARNING;
						} else if (lock>0) {
							weight_n = Weight.WARNING;
						} 
												
						line[i] += " type:c3p0" + 
								   " name:" + key +
								   " max:" + max +
								   " inc:" + pds.getAcquireIncrement() +
								   " used:" + pds.getNumBusyConnectionsAllUsers() +	
								   " idle:" + pds.getNumIdleConnectionsAllUsers() +
								   " orphan:" + pds.getNumUnclosedOrphanedConnectionsAllUsers();
						graph[i] = "c3p0_"+key+"="+busy+";"+lev_warn+";"+lev_crit+";0;"+max;
						
						
					} catch (SQLException e) {
						line[i] += " type:c3p0" + 
								   " name:" + key +
								   " error:" + e.toString();
						graph[i] = "";
						weight_n = Weight.CRITICAL;
					}					
				} else {
					line[i] += " type:" + ds.getClass().getSimpleName() + 
							   " name:" + key;
					graph[i] = "";
				}
				
				if (weight_n.ordinal()>weight.ordinal()) {
					weight=weight_n;
				}	
				i++;
			}
			
			// Estrazione dati secondo NAGIOS
			if (size>=1) {
				text += line[0] + " | " + graph[0];
			} 
			if (i>1) {
				for (int k=1; k<size; k++) {
					text += "\n" + line[k];
				}
				text += " | ";
				for (int k=1; k<size; k++) {
					text += graph[k] + "\n";
				}
			}
		}
		
		text = weight +" - " + text;
		response(response, text, "text/plain");
	}
	

	private void responseXML(HttpServletResponse response, boolean thread) {
		Runtime runtime = Runtime.getRuntime();
		String host = System.getProperty("hostname");
		ApplicationClusterInfo info = ApplicationClusterInfo.getInstance();

		// Verifico il database
		boolean dbok = checkDatabase(response);
		
		ThreadGroup root = null;
		if (thread) {
			root = Thread.currentThread().getThreadGroup().getParent();
			while (root.getParent() != null) {
				root = root.getParent();
			}
		}

		String xml = "<MONITOR status=\"OK\">" + 
					 	"<APPLICATION version=\"" + info.getApplicationVersion() + "\"" + " title=\"" + parameters.getChannelInfo().getApplicationTitle() + "\"" + " name=\"" + parameters.getChannelInfo().getApplicationName() + "\"" + " context=\"" + parameters.getChannelInfo().getContextName() + "\"" + " />" + 
					 	"<HOST name=\"" + info.getHostName() + "\"" + " address=\"" + info.getHostAddress() + "\" >" + 
					 		"<DOMAIN>" + parameters.getChannelInfo().getDomainName() + "</DOMAIN>" + 
					 		"<CONTEXT>" + info.getContext() + "</CONTEXT>" + 
					 		"<MASKED>" + info.getPublicHostnameMasked() + "</MASKED>" + 
					 		"<BACKPLANE>" + info.getBackplaneHostnameMasked() + "</BACKPLANE>" +
					 		(host != null ? "<SYSTEM-PROPERTY>" + host + "</SYSTEM-PROPERTY>" : "") + 
					 	"</HOST>" + 
					 	"<TIMESTAMP>" + EnterpriseCalendar.now().formatISO8601zulu() + "</TIMESTAMP>" + 
					 	"<RUNTIME processor=\""	+ runtime.availableProcessors() + "\">" + 
					 		"<MEMORY total=\"" + XString.formatByte(runtime.totalMemory()) + "\" " + " free=\"" + XString.formatByte(runtime.freeMemory()) + "\"" + " max=\"" + XString.formatByte(runtime.maxMemory()) + "\"" + " />" + 
					 		"<DATABASE>"+(dbok?"OK":"ERROR")+"</DATABASE>" +
					 		datasourceXML() + 
					 		(thread?"<THREADS>" + visit(root, 0) + "</THREADS>":"") + 
					 	"</RUNTIME>" + 
					 "</MONITOR>";
		response(response, xml, "text/xml");
	}



	private void response(HttpServletResponse response, String message, String contentType) {
		// Creo la risposta
		try {

			// Response output stream
			ServletOutputStream out = null;
			try {
				String version = Version.getInstance().toString();
				response.setHeader("X-server-version", version);
				//
				response.setContentType(contentType);
				out = response.getOutputStream();
				out.write(message.getBytes());

			} catch (IOException e) {
				log.error("Comunication exception", e);
				response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			} finally {
				try {
					if (out != null)
						out.close();
				} catch (Exception e) {
					log.warn("Error closing output stream", e);
				}
			}

		} catch (Exception e) {
			log.error("Unhandled exception", e);
			response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			return;
		}
	}

}
