package it.softecspa.task;

import java.sql.SQLException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Timer;

import javax.servlet.ServletConfig;

import org.apache.log4j.Logger;

public class TaskManager {

	// Singleton
	private static TaskManager instance;

	private boolean forceStop;
	
	private TaskManager() {
		super();
		synchronized (TaskManager.class) {
			if (instance == null) {
				instance = this;
			}
		}
		forceStop = false;
	}

	public static TaskManager getInstance() {
		synchronized (TaskManager.class) {
			if (instance == null)
				new TaskManager();
		}
		return instance;
	}

	private Logger logger = Logger.getLogger(getClass());

	private Hashtable<String, WebTimerTask> allThreads = new Hashtable<String, WebTimerTask>(); // Hashtable
																					// di
																					// oggetti
																					// Timer
	private ServletConfig config;
	private int startcount;

	class Delayer implements Runnable {

		private Thread iThread;

		public void run() {
			if (forceStop)  {
				return;
			}
			
			logger.info("Delayer thread is waiting");
			iThread = Thread.currentThread();
			try {
				Thread.sleep(60 * 1000);
			} catch (InterruptedException e) {
				logger.error("Thread interrupted with exception", e);
				stop();
				return;
			}
			logger.info("Delayer thread is awake");
			reloadAllTask();
		}

		public void stop() {
			if (iThread == null) {
				return;
			}
			try {
				iThread.interrupt();
			} catch (Exception e) {
				logger.error(e.getClass().getSimpleName() + " interrupting delayer thread", e);
			}
			logger.info("Delayer thread stopped");
		}
	}

	private void reloadAllTask() {
		if (forceStop)  {
			return;
		}
		
		DBTask[] arrDBTasks = null;
		try {
			arrDBTasks = DBTask.getAllTasks();
		} catch (SQLException e) {
			startcount++;
			logger.warn("Task do not started, retry later ("+startcount+")", e);
			
			// Faccio partire un thread di appoggio con delay 60 secondi
			Thread delayer = new Thread(new Delayer(), "delay-task-starter(" + startcount + ")");
			delayer.setDaemon(true);
			delayer.start();
			return;
		} catch (Exception e) {
			logger.fatal("Unhandled exception starting thread", e);
			return;
		} catch (Error e) {
			logger.fatal("Very grave error starting thread", e);
			return;
		}
		
		try {
			allThreads = new Hashtable<String, WebTimerTask>();
			for (int i = 0; i < arrDBTasks.length; i++) {
				DBTask task = arrDBTasks[i];
				Class<?> c = Class.forName(task.getClassName());
				WebTimerTask timerTask = (WebTimerTask) c.newInstance();
				timerTask.readParams(task.getParamString());
				timerTask.setConfig(config);

				if (task.isRunOnStartup()) {
					Timer timer = new Timer(true);
					long delay = task.getDelay();
					logger.info("Starting new "+timerTask.getClass().getSimpleName()+" (delay: "+delay+")");
					
					if (task.getRate() >= 0) {
						timer.scheduleAtFixedRate(timerTask, delay, task.getRate() * 60000);
					} else {
						timer.schedule(timerTask, delay);
					}
					timerTask.setScheduled(timer);
				} else {
					timerTask.setScheduled(null);
				}				
				
				timerTask.setDbTask(task);
				if (logger.isDebugEnabled()) logger.debug("Timertask #"+task.getIdTask()+" add to threads list");
				allThreads.put(String.valueOf(task.getIdTask()), timerTask);
			}
		} catch (Exception e) {
			logger.error("Errore nel caricamento dei Tasks all'avvio.", e);
		}

	}

	public void startThreadsOnStartup(ServletConfig config) {
		this.config = config;
		reloadAllTask();
	}
	
	
	public void stopAllThreads() {
		forceStop = true; // Utilizzato per stoppare il Delayar
		
		if (getAllThreads()!=null) {
			logger.info("Stopping all timertask thread ("+getAllThreads().size()+")");
			for (Enumeration<String> key = getAllThreads().keys();key.hasMoreElements();) {
				String idTask = key.nextElement();
				WebTimerTask timerTask = (WebTimerTask) allThreads.get(idTask);
				if (!timerTask.isScheduled()) {
					// TimerTask gi stoppato
					//logger.info("Timertask #"+idTask+" just stopped");
				} else {
					timerTask.cancelScheduling();
					// Thread.sleep(3000);
					// Azzero il next run at
					timerTask.getDbTask().resetNextRunAt();
				}
			}
		}
	}
	
	

	public Hashtable<String, WebTimerTask> getAllThreads() {
		return allThreads;
	}

	public ServletConfig getConfig() {
		return config;
	}

	public void setConfig(ServletConfig config) {
		this.config = config;
	}

}
