package it.softecspa.fileproxy.proxyservices.manager;

import it.softecspa.fileproxy.proxyservices.request.UploadRequestType;
import it.softecspa.fileproxy.proxyservices.response.UploadResponseType;
import it.softecspa.fileproxy.services.common.CheckerException;
import it.softecspa.fileproxy.services.common.ManagerException;
import it.softecspa.fileproxy.services.common.ResponseOutcome;
import it.softecspa.kahuna.io.File;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

public class UploadManager extends AbstractHttpFileProxyManager<UploadRequestType, UploadResponseType> {

	private File file_to_upload;

	@Override
	protected void validateRequest() throws CheckerException, ManagerException {
		// Non si possono sovrascrivere le cartelle
		
		checkRequestMandatory("fileName");
		checkWildecard(request.getPath());
		String path = normalizePath(request.getPath());
		
		checkRequestIfNullSetDefault("overwrite", Boolean.FALSE);
		checkRequestIfNullSetDefault("append", Boolean.FALSE);
		checkRequestIfNullSetDefault("bufferLength", 4096);
		
		checkWildecard(request.getFileName());
		file_to_upload = new File (path, request.getFileName());
		if (file_to_upload.exists()) {
			if (!request.getOverwrite().booleanValue() && !request.getAppend().booleanValue()) {
				log.warn("File just not exist: '"+file_to_upload.getAbsolutePath()+"', overwrite or append are FALSE");
				throw new CheckerException(ResponseOutcome.FILE_JUST_EXIST);
			}
			
			if (file_to_upload.isDirectory()) {
				log.warn("File just not exist: '"+file_to_upload.getAbsolutePath()+"', but is a directory");
				throw new CheckerException(ResponseOutcome.FILE_IS_DIRECTORY);
			}
		}
		
		
		
	}

	@Override
	protected void doService(UploadRequestType request, UploadResponseType response) throws ManagerException {
		log.info("Request UPLOAD file '"+ file_to_upload.getAbsolutePath()+"'" 
										+ (request.getOverwrite().booleanValue()?" (overwrite = TRUE)":"") 
										+ (request.getAppend().booleanValue()?" (append = TRUE)":"")
										+ " (bufferLength = "+request.getBufferLength()+") ");
		
		// Prendo input stream e lo salvo su un file (prima temporaneo)
		File file_temp = new File(file_to_upload.getAbsoluteFile()+".tmp");
		// Provo a cancellare il file temporaneo
		file_temp.delete();
		
		if (request.getAppend().booleanValue()) {
			// Devo fare una copia del file per farlo diventara temporaneo
			if (file_to_upload.exists()) {
				// ...naturalmente se il file esite!
				try {
					
					log.info("Creating temporay copy of file '"+ file_to_upload.getAbsolutePath()+"'");
					file_to_upload.copyTo(file_temp);
				} catch (IOException e) {
					throw new ManagerException(new CheckerException(ResponseOutcome.ERROR_COPY_FILE_TEMP));
				}
			}
		}
		
		
		InputStream input = request.takeInputStream();
		OutputStream output = null;
		int BUFFER_LENGTH = request.getBufferLength();
		
		try {
			output = new FileOutputStream(file_temp, request.getAppend().booleanValue());  
		
			byte[] buffer = new byte[BUFFER_LENGTH];  
			int bytesRead;  
			while ((bytesRead = input.read(buffer)) != -1) {  
				output.write(buffer, 0, bytesRead);  
				output.flush();
			}  
			
		 } catch (FileNotFoundException e) {
	    	log.error("File '"+file_temp.getAbsolutePath()+"' not found");
			throw new ManagerException(new CheckerException(ResponseOutcome.FILE_NOT_FOUND));
			
	    } catch (IOException e) {
	    	log.error("IOException writing file '"+file_temp.getAbsolutePath()+"'", e);
			throw new ManagerException(e);
			
	    } finally {
	    	if (output!=null) {
			    try {
			    	output.close();
				} catch (IOException e) {
					log.error("IOException close file '"+file_temp.getAbsolutePath()+"'",e);
				}
	    	}
	    }
		
		log.info("Copy completed, rename temp file to '"+ file_to_upload.getAbsolutePath()+"'");
		if (request.getOverwrite().booleanValue()) {
			if (file_to_upload.exists() && !file_to_upload.delete()) {
				log.info("Impossibility to delete file '"+ file_to_upload.getAbsolutePath()+"' with override flag");
			}
		} else if (request.getAppend().booleanValue()) {
			if (file_to_upload.exists() && !file_to_upload.delete()) {
				log.info("Impossibility to delete old file '"+ file_to_upload.getAbsolutePath()+"' with append flag");
			}
		}
		
		file_temp.renameTo(file_to_upload);
		
		
	   
		// Risposta POSITIVA
		new ResponseBuilder(response).setReturn(ResponseOutcome.OK);
	}

	
}
