package it.softecspa.fileproxy.services;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.database.dbconnect.NoRecordFoundException;
import it.softecspa.fileproxy.DatabaseBalancer;
import it.softecspa.fileproxy.DatabaseBalancer.Balancer;
import it.softecspa.fileproxy.cache.CacheContainer;
import it.softecspa.fileproxy.cache.CacheElement;
import it.softecspa.fileproxy.db.ConfigProperties;
import it.softecspa.fileproxy.db.criterias.ConfigPropertiesCriteria;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.Properties;

import java.sql.SQLException;
import java.util.Set;

import org.apache.log4j.Logger;


/**
 * Contiene lo stato dei servizi e dei domini attivi * 
 * @author m.veroni
 * @param <T>
 */
public class ServerCacheFactory {
	
	private static ServerCacheFactory instance;
	private Logger log = Logger.getLogger(getClass());
	
	
	// Contenitore delle strutture in cache
	private CacheContainer cacheContainer;
	public ListContainer list;
	
	// Parametri di configurazione su database - T001_CONFIG_PROPERTIES
	private Properties dbProperties;
	
	/**
	 * Classe fittizia di rapporto
	 */
	public class ListContainer {
		
		
		/* FIXME esempio
		public Collection<Instance> ofInstances(boolean stage) {
			try {
				SortedMap<String, Instance> container = cacheContainer.getContainer(CacheKey.INSTANCE_ID, getBalancer(stage));
				return container.values();
			} catch (CacheException e) {
				log.error("Cache container problem: "+e.toString());
				return null;
			}
		}
		*/
	}
	
	
	
	private ServerCacheFactory() {
		super();
		synchronized (ServerCacheFactory.class) {
			if (instance == null) {
				instance = this;
				//
				log.info(ServerCacheFactory.class.getSimpleName()+" instance is starting!");
				instance.reloadAll();				
			}
		}
	}

	
	


	public static ServerCacheFactory getInstance() {
		synchronized (ServerCacheFactory.class) {
			if (instance == null) new ServerCacheFactory();			
		}
		return instance;
	}
	
	
	/**
	 * Carica la struttura contenente le abilitazioni al servizio
	 */
	private synchronized void reload(boolean stage) {
		try {
			Balancer balancer = getBalancer(stage);
			if (!DatabaseBalancer.getInstance().isActive()) {
				// non posso ancare avanti se non  presente un database
				return;
			}
			
			if (stage) {
				loadProperties();	
			}
			
			/* Esempio
			cacheContainer.createContainer(new KmTypeRemoteCache(balancer));
			cacheContainer.createContainer(new KmTypeDeviceCache(balancer));
			*/
		
			/*
		} catch (CacheException e) {
			log.fatal(e.toString(),e);
		*/} catch (Exception e) {
			log.fatal("Unhandled exception",e);
		}
	}
	
	
	public synchronized void reloadAll() {
		
		if (cacheContainer==null) {
			if (log.isDebugEnabled()) log.debug("Create instance of " + CacheContainer.class + " service");
			cacheContainer = CacheContainer.getInstance();	
			list = new ListContainer();
		}
		
		reloadOnStage();
		reloadOnMaster();	
	}
	
	
	
	
	public synchronized void reloadOnStage() {				
		// Estraggo i dati dal database STAGE
		log.info("Reload cache on STAGE database");
		reload(true);
	}

	public synchronized void reloadOnMaster() {				
		// Estraggo i dati dal database MASTER
		log.info("Reload cache on MASTER database");
		reload(false);
	}
	
	
	private Balancer getBalancer(boolean stage) {
		return (stage?Balancer.STAGE:Balancer.MASTER);
	}
	
	
	/**
	 * Estrazione parametri di configurazione dal database (MASTER)
	 * Nota: su STAGE  una vista su MASTER 
	 */
	private void loadProperties() {
		ConnectionManager cm = DatabaseBalancer.getInstance().get(Balancer.STAGE);
				
		try {
			log.info("Load 'Properties' configuration list on STAGE");
			ConfigPropertiesCriteria criteria = new ConfigPropertiesCriteria(cm); 
			ConfigProperties[] lista = criteria.select();
			
			if (dbProperties!=null) {
				dbProperties.clear();
			} else {
				dbProperties = new Properties();
			}
			
			for (ConfigProperties elemento : lista) {
				dbProperties.put(elemento.getKey(), elemento.getValue());
			}
			log.info("Loaded configuration from database");
			
			
		} catch (NoRecordFoundException e) {
			log.warn("No properties available on STAGE database, verify configuration");
			
		} catch (SQLException e) {
			log.fatal("Problem reading database",e);
			dbProperties = null;
		}	
	}

	

	
	
	
	/**
	 * Properties estratte dal database (MASTER)
	 * @return
	 */
	public Properties getDbProperties() {
		// Controllo inserito per la gestione dei problemi di connettivit
		if (dbProperties==null) loadProperties();
		
		return dbProperties;
	}


	
	public enum PropertiesKey {
				
		  CLUSTER_PASSWORD 				("cluster_password")
		, CLUSTER_USERNAME 				("cluster_username")
		
		, MASTER_SERVER_URL 			("master.server.url")
		, MASTER_HEARTBEAT_COUNT		("master.heartbeat.count")
		, MASTER_SERVER_USERNAME 		("master.server.username")
		, MASTER_SERVER_PASSWORD 		("master.server.password")
		
		// Url pubblico del semaforo
		//, URL_SEMAFORO	("url.semaforo")
		, CONSOLE_URL					("url.console")
		
		// ...
		, HEARTBEAT_DELAY 				("haertbeat.delay")
		, TRIAL_DEFAULT_TYPE 			("trial.default.type")
		, TRIAL_DEFAULT_VALUE 			("trial.default.value")
		, DAY_ALERT						("day_alert")
		
		// Root di filesystem per salvare le risorse
		, RESOURCE_ROOT_MASTER			("resource.root")
		, RESOURCE_ROOT_STAGE			("resource.root.stage")	
		//, RESOURCE_THUMB_WIDTH			("resource.thumb.width")
		//, RESOURCE_THUMB_HEIGHT			("resource.thumb.height")
		
		// Root di filesystem per salvare i pacchetti da scaricare
		//, DOWNLOAD_ROOT					("download.root")
		//, DOWNLOAD_URL					("download.url")
		/*
		// Email per sincronizzazione
		, SYNCHRO_EMAIL_ERROR			("synchro.mail.error")
		, SYNCHRO_EMAIL					("synchro.mail")
		, SYNCHRO_LOCK_CLIENT			("synchro.lock.client")
		*/
		/*
		// Connettivit verso My-DesktopMate
		, MYDM_URL						("mydesktopmate.url")
		, MYDM_USERNAME					("mydesktopmate.username")
		, MYDM_PASSWORD					("mydesktopmate.password")
		*/
		// Abilitazione/disabilitazione servizi
		//, DISABLE_CHECK_HISTORY			("disable.check.history")
		
		, LANGUAGE_DISABLE_LIST			("language.disable.list")
		/*
		, SHARE_URL						("share.url")
		, MAINTENANCE_ENABLE			("maintenance.enable")
		
		, PROVISIONING_ALERT			("provisioning.alert")
		, DAILY_REPORT_FLUSH_LINE		("daily.report.flush.line")
		, DAILY_REPORT_FLUSH_HYML		("daily.report.flush.html")
		*/
		;
		
		
		private String value;
		
		private PropertiesKey(String value) {
			this.value = value;
		}
		
		public String toString() {
			return value;
		}
	}
	
	
	public enum CacheKey {
		  KM_TYPE_REMOTE
		, KM_TYPE_DEVICE
		;
		
	}
	
	
	/**
	 * @deprecated
	 * Restituisce la ROOT delle risorse 
	 * @param stagemode
	 * @return
	 */
	public String getResourceRoot(boolean stagemode) {
		return getDbProperties().get(stagemode?PropertiesKey.RESOURCE_ROOT_STAGE.toString():PropertiesKey.RESOURCE_ROOT_MASTER.toString());
	}

	public String getResourceRoot() {
		return getDbProperties().get(PropertiesKey.RESOURCE_ROOT_MASTER.toString());
	}


	
	public Set<String> getCachedList(boolean stagemode) {
		return cacheContainer.getCachedTables()[getBalancer(stagemode).index()].keySet();
	}
	
	
	public boolean isCached(String tablename, boolean stagemode) {
		if (XString.isBlankNull(tablename)) return true;
		return cacheContainer.getCachedTables()[getBalancer(stagemode).index()].containsKey(tablename);
	}



	

	public synchronized void reloadAll(String tablename) {
		reloadOnStage(tablename);
		reloadOnMaster(tablename);
	}



	public void reloadOnStage(String tablename) {
		if (XString.isBlankNull(tablename)) {
			reloadOnStage();
			return;
		}
		if (!isCached(tablename, true)) return;
		
		for (CacheElement<?,?> element : cacheContainer.getCachedTables()[getBalancer(true).index()].get(tablename)) {
			element.flush();
		}
	}

	public void reloadOnMaster(String tablename) {
		if (XString.isBlankNull(tablename)) {
			reloadOnMaster();
			return;
		}
		if (!isCached(tablename, false)) return;

		for (CacheElement<?,?> element : cacheContainer.getCachedTables()[getBalancer(false).index()].get(tablename)) {
			element.flush();
		}
	}

	
	
	
	
}
