package it.softecspa.fileproxy.services.common.core;

import it.softecspa.PortalSettings;
import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.database.dbconnect.SQLTimeoutException;
import it.softecspa.fileproxy.DatabaseBalancer;
import it.softecspa.fileproxy.services.ServerCacheFactory;
import it.softecspa.fileproxy.services.ServerCacheFactory.PropertiesKey;
import it.softecspa.fileproxy.services.common.CheckerException;
import it.softecspa.fileproxy.services.common.EnterpriseLog;
import it.softecspa.fileproxy.services.common.ManagerException;
import it.softecspa.fileproxy.services.common.RemoteCallException;
import it.softecspa.fileproxy.services.common.ResponseOutcome;
import it.softecspa.fileproxy.services.common.ResponseResult;
import it.softecspa.fileproxy.services.common.core.request.AddressableRequest;
import it.softecspa.fileproxy.services.common.core.request.AuthIsMandatory;
import it.softecspa.fileproxy.services.common.core.request.InputStreamReaderJSON;
import it.softecspa.fileproxy.services.common.core.request.InputStreamReaderXML;
import it.softecspa.fileproxy.services.common.core.request.JsonMapper;
import it.softecspa.fileproxy.services.common.core.request.UniversalAuthenticatedRequest;
import it.softecspa.fileproxy.services.common.core.request.UniversalRequest;
import it.softecspa.fileproxy.services.common.core.response.UniversalResponse;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.xml.XmlParserException;
import it.softecspa.kahuna.util.xml.XmlRoot;
import it.softecspa.sso.common.SsoException;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.rmi.RemoteException;
import java.sql.SQLException;
import java.util.Locale;
import java.util.ResourceBundle;

import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

public abstract class UniversalManager <REQ extends UniversalRequest, RES extends UniversalResponse, S extends UniversalSession<REQ>> {

	protected static final int STANDARD_THREAD_DELAY = 2000;
	protected Logger log = Logger.getLogger(this.getClass());
	
	
	// Reqest e resposne della chiamata
	protected REQ request;
	protected RES response;
	protected S session;
	
	// Istanza della cache di sistema
	protected ServerCacheFactory cacheInfo;
	
	// Gestione della lingua per la risposta
	//protected ResourceBundle bundle;
	protected String session_language;
	protected ResourceBundle session_bundle;
	
	
	// Balancer del database
	private DatabaseBalancer balance;
	
	
	// Connection manager da utilizzare "normalizzate" dalla chiamata (MASTER/STAGE)
	protected ConnectionManager cmWritable;
	protected ConnectionManager cmReadonly;	
	protected boolean stageMode;
	
	
	
	/**
	 * Metodo invocato prima di eseguire tutte le chiamate.
	 * La request non  stata ancora verificata e la response ha i valori di default
	 */
	protected abstract void initialize() throws ManagerException;
	
	
	/**
	 * Crea e verifica la sessione
	 * @throws CheckerException
	 * @throws ManagerException} 
	 */
	protected abstract S buildAndValidateSession(ServerCacheFactory cacheInfo, boolean stageMode) throws CheckerException, ManagerException;
	
	
	/**
	 * Verifica la corenza dei parametri inseriti
	 * @throws CheckerException
	 */
	protected abstract void validateRequest() throws CheckerException, ManagerException;
	
	
	/**
	 * Core del metodo
	 * @param request
	 * @param response
	 */
	protected abstract void doService(REQ request, RES response) throws ManagerException;
		
	
	/**
	 * Metodo invocato al termine della sequenza, dopo che  stata valorizzata la response
	 * e solo se l'esito  positivo
	 * @param stage
	 * @throws ManagerException
	 */
	protected abstract void completeWithSuccess(boolean stage) throws ManagerException;
	
	
	/**
	 * Crea la response in automatico a partire da {@link Type[]} della definizione della classe
	 * @return
	 */
	protected RES createResponse()  {
		if (log.isDebugEnabled()) log.debug("Auto generate response from class TypeArgument");
		/*
		 * http://stackoverflow.com/questions/75175/create-instance-of-generic-type-in-java
		 */
		try {
			Type type = getClass().getGenericSuperclass();
	        ParameterizedType paramType = (ParameterizedType) type;
	        @SuppressWarnings("unchecked")
			Class<RES> clazz = (Class<RES>) paramType.getActualTypeArguments()[1];
			
	        if (log.isDebugEnabled()) log.debug("Create response: "+clazz);
			RES instance = clazz.newInstance();
			return instance;
			
		} catch(Exception e) {
			log.error("Error create response from TypeArgument",e);
			return null;
		}
	}
	
	
	/**
	 * Metodo pubblico di esecuzione
	 * @param request
	 * @return
	 */
	public RES process(REQ _request) {
		if (log.isDebugEnabled()) log.debug("Calling '"+this.getClass().getSimpleName()+"' operation");
		balance = DatabaseBalancer.getInstance();
		cacheInfo = ServerCacheFactory.getInstance();
		
		// -----------------------------------------------------------------------------------------
		// FASE 0 - Controlli formali
		if (log.isDebugEnabled()) log.debug("Validate input/output parameters and values (phase 0)");
		// RESPONSE
		this.response = createResponse();
		if (response==null) {
			log.fatal("Not valid response, verify 'createResponse' method");
			return null;
		}
		// Imposto un valore di default per la response
		new ResponseBuilder(response).setReturn(ResponseOutcome.RESPONSE_NOT_VALID);
		// REQUEST
		this.request = _request;
		if (this.request==null) {
			log.fatal("Null request is not valid");
			return new ResponseBuilder(response).setReturn(ResponseOutcome.REQUEST_NOT_VALID)
												.decodeMessageErrorWithBundle();
		}
		// --------------------------------------------------------------------------------------------
		
		
		try {
			// -----------------------------------------------------------------------------------------
			// FASE 1 - Inizializzazione
			if (log.isDebugEnabled()) log.debug("Inizialize manager (phase 1)");
			session_language = request.getLanguage();
			initialize();
			
			// -----------------------------------------------------------------------------------------
			// FASE 2 e 3 - Controllo la validit dei parametri in ingresso e creazione sessione
		
			// Definizione dell'ambiente a partire dai dai di istanza
			stageMode = identifyStageMode();
			validateStageMasterConnection();			
			
			// FASE 2a - Controllo sessione
			if (log.isDebugEnabled()) log.debug("Build and validate private session (phase 2)");
			session = buildAndValidateSession(cacheInfo, stageMode);
			if (session==null) {
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SESSION_NOT_VALID)
													.decodeMessageErrorWithBundle();
			}			
			if (log.isDebugEnabled()) log.debug("Request from remote server, session created with success");
			if (session_language==null) session.getLanguage();
			
					
			// Controllo delle credenziali su SSO
			if (request instanceof UniversalAuthenticatedRequest) {
				if (log.isDebugEnabled()) {
					log.debug("Authenticated request, automatic check of credential " + (this.request instanceof AuthIsMandatory?"MANDATORY":"optional"));
				}
				// Validazione delle credenziali
				try {
					session.validateCredential();
					/*
					 * Se queste sono errate o si verifica un errore in questa fase he restituito un HTTP 403 - FORBIDDEN
					 */
				} catch (CheckerException e) {
					MyReturn myReturn = response.takeMyReturn();
					myReturn.changeHttpStatus(HttpServletResponse.SC_FORBIDDEN);
					throw e;
				} catch (ManagerException e) {
					MyReturn myReturn = response.takeMyReturn();
					myReturn.changeHttpStatus(HttpServletResponse.SC_FORBIDDEN);
					throw e;
				}
			}
			
			
			// Controlli custom per ogni singola chiamata
			if (log.isDebugEnabled()) log.debug("Validate Request (phase 3)");
			if (request==null) throw new CheckerException(ResponseOutcome.REQUEST_IS_NULL);
			validateRequest();
			
		} catch (CheckerException e) {
			if (log.isDebugEnabled()) {
				log.debug("CheckerException, message: "+e.getMessage() + (e.getResult()!=null?"; '"+e.getResult()+"'":""));
			}
			checkAndTraceCheckerExceptionErrorIfExist(e);
			
			// Rispondo al client
			return new ResponseBuilder(response).setReturn(e)
												.decodeMessageErrorWithBundle();
		
		} catch (ManagerException e) {
			// CheckerException, nessun invio di mail
			if ((e.getCause()!=null && e.getCause() instanceof CheckerException)) {
				CheckerException ec = (CheckerException)e.getCause();
				if (log.isDebugEnabled()) {
					log.debug("CheckerException, message: "+e.getMessage() + (ec.getResult()!=null?"; '"+ec.getResult()+"'":""));
				}
				checkAndTraceCheckerExceptionErrorIfExist(ec);
				
				// Rispondo al client
				return new ResponseBuilder(response).setReturn(ec)
													.decodeMessageErrorWithBundle();		
			}
			
			if (e.getCause()!=null) {
				log.error("Manager exception checking fields: "+e.getCause().toString(),e);
			} else {
				log.error("Manager exception checking fields: "+e.toString(),e);
			}
			
			// SQLTimeoutException, nessun invio di mail
			if ((e.getCause()!=null && e.getCause() instanceof SQLTimeoutException)) {
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_TIMEOUT)
													.decodeMessageErrorWithBundle();			
			}
			// SQLException, invio della mail
			if ((e.getCause()!=null && e.getCause() instanceof SQLException)) {
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_EXCEPTION)
													.sendEmailAlert(ResponseOutcome.SQL_EXCEPTION, e.getCause())
													.decodeMessageErrorWithBundle();			
			}
			// SsoException
			if ((e.getCause()!=null && e.getCause() instanceof SsoException)) {
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SSO_GENERIC_ERROR)
													.sendEmailAlert(ResponseOutcome.SSO_GENERIC_ERROR, e.getCause())
													.decodeMessageErrorWithBundle();
			}
			
			// ManagerException with cause
			if (e.getCause()!=null) {
				return new ResponseBuilder(response).setReturn(e).getResponse();
			}
			
			return new ResponseBuilder(response).setReturn(ResponseOutcome.MANAGER_EXCEPTION)
												.decodeMessageErrorWithBundle();
			
		} catch (Exception e) {			
			// SQLTimeoutException, nessun invio di mail
			if (e instanceof SQLTimeoutException ||
				(e.getCause()!=null && e.getCause() instanceof SQLTimeoutException)) {
				log.error("SQLTimeoutException checking fields (DB TIMEOUT)");
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_TIMEOUT)
													.decodeMessageErrorWithBundle();			
			}
			// SQLException, invio della mail
			if (e instanceof SQLException ||
				(e.getCause()!=null && e.getCause() instanceof SQLException)) {
				log.error("SQLException checking fields",e);
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_EXCEPTION)
													.sendEmailAlert(ResponseOutcome.SQL_EXCEPTION, e.getCause())
													.decodeMessageErrorWithBundle();
			}
		
			// Eccezione non gestita e non identificata
			log.error("Unhandled exception checking fields",e);
			return new ResponseBuilder(response).setReturn(ResponseOutcome.UNHANDLED_EXCEPTION)
												.sendEmailAlert(ResponseOutcome.UNHANDLED_EXCEPTION, e)
												.decodeMessageErrorWithBundle();
		}
		if (log.isDebugEnabled()) log.debug("Validation completed with success!");
				
		
		// -----------------------------------------------------------------------------------------
		// FASE 4 - Eseguo in core del servizio
		if (log.isDebugEnabled()) log.debug("Processing the service (phase 4)");
		try {
			doService(request, response);
		} catch (ManagerException e) {
			// SQLTimeoutException, nessun invio di mail
			if ((e.getCause()!=null && e.getCause() instanceof SQLTimeoutException)) {
				log.error("SQLTimeoutException processing operation (DB TIMEOUT)");
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_TIMEOUT)
													.decodeMessageErrorWithBundle();			
			}
			// SQLException, invio della mail
			if ((e.getCause()!=null && e.getCause() instanceof SQLException)) {
				log.error("SQLException processing operation",e.getCause());
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_EXCEPTION)
													.sendEmailAlert(ResponseOutcome.SQL_EXCEPTION, e.getCause())
													.decodeMessageErrorWithBundle();
			}
			// CheckerException
			if ((e.getCause()!=null && e.getCause() instanceof CheckerException)) {
				// Stampo direttamente l'errore passato nel check, senza stackTrace
				log.error(e.getCause().toString());
				checkAndTraceCheckerExceptionErrorIfExist((CheckerException)e.getCause());
				
				// Rispondo al client
				return new ResponseBuilder(response).setReturn((CheckerException)e.getCause())
													.decodeMessageErrorWithBundle();
			}
			
			log.error("ManagerException processing operation",e);
			// RemoteException
			if ((e.getCause()!=null && e.getCause() instanceof RemoteException)) {
				return new ResponseBuilder(response).setReturn(ResponseOutcome.REMOTE_EXCEPTION)
													.decodeMessageErrorWithBundle();
			}
			// RemoteCallException
			if ((e.getCause()!=null && e.getCause() instanceof RemoteCallException)) {
				return new ResponseBuilder(response).setReturn((RemoteCallException)e.getCause())
													.decodeMessageErrorWithBundle();
			}
			
			// ManagerException with cause
			if (e.getCause()!=null) {
				return new ResponseBuilder(response).setReturn(e).getResponse();
			}
			
			return new ResponseBuilder(response).setReturn(ResponseOutcome.MANAGER_EXCEPTION)
												.decodeMessageErrorWithBundle();
			
		} catch (Exception e) {			
			// SQLTimeoutException, nessun invio di mail
			if (e instanceof SQLTimeoutException ||
				(e.getCause()!=null && e.getCause() instanceof SQLTimeoutException)) {
				log.error("SQLTimeoutException processing operation (DB TIMEOUT)");
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_TIMEOUT)
													.decodeMessageErrorWithBundle();			
			}
			// SQLException, invio della mail
			if (e instanceof SQLException ||
				(e.getCause()!=null && e.getCause() instanceof SQLException)) {
				log.error("SQLException processing operation",e);
				return new ResponseBuilder(response).setReturn(ResponseOutcome.SQL_EXCEPTION)
													.decodeMessageErrorWithBundle();
			}
			
			// Eccezione non gestita e non identificata
			log.error("Unhandled exception processing operation",e);
			return new ResponseBuilder(response).setReturn(ResponseOutcome.UNHANDLED_EXCEPTION)
												.sendEmailAlert(ResponseOutcome.UNHANDLED_EXCEPTION, e)
												.decodeMessageErrorWithBundle();
			
		}
		if (log.isDebugEnabled()) log.debug("Process completed with success!");
		
		
		// -----------------------------------------------------------------------------------------
		// FASE 5 - Finito
		new ActionCompleted(stageMode).executeWithDelay(STANDARD_THREAD_DELAY);
		//new ActionCompleted(stageMode).executeNow();
				
		
		// -----------------------------------------------------------------------------------------
		// FASE 6 - Rispondo al chiamante
		if (log.isDebugEnabled()) log.debug("Answer to client: ["+response.getReturnCode()+"] "+response.getReturnMessage());
		return new ResponseBuilder(response).decodeMessageErrorWithBundle();
	}
	
	
	
	/** 
	 * Verifica ed eventualemente registra l'errore nella bel
	 * @param e
	 */
	protected abstract void checkAndTraceCheckerExceptionErrorIfExist(CheckerException e);
	
	/**
	 * Identifica la modalit di gestione piattaforma e la variabile stageMode
	 * @throws CheckerException 
	 */
	protected boolean identifyStageMode() throws ManagerException, CheckerException {
		if (request instanceof AddressableRequest) {
			if (log.isDebugEnabled()) log.debug("Identify STAGE/MASTER using specific request attribute");	
			if (((AddressableRequest)request).getStage()==null) {
				if (log.isDebugEnabled()) log.debug("Addressable request do not specify master/stage, force STAGE");			
				stageMode = true;	
				((AddressableRequest)request).setStage(true); 
			} else {
				stageMode = ((AddressableRequest)request).getStage();
				if (log.isDebugEnabled()) log.debug("Request forze use of "+(stageMode?"REALTIME":"PUBLICATION")+" mode");	
			}
		} else {
			// Vado esclusivamente su STAGE
			if (log.isDebugEnabled()) log.debug("Identify stage/master using default configuration, force STAGE");			
			stageMode = true;			
		}
				
		return stageMode;
	}

	/**
	 * Identifica l'ambiente e setta il ConnectionManager specifico
	 */
	private void validateStageMasterConnection() {
		/* Vecchia modalit antecedente al protocollo 3, febbraio 2013
		stageMode = Boolean.TRUE.equals(request.getStage());
		*/
		
		if (stageMode) {
			// uso STAGE
			if (log.isDebugEnabled()) log.debug("Stage mode request, use only WRITABLE connection manager!");			
			cmWritable = balance.getStage();
			cmReadonly = cmWritable;
		} else {
			if (balance.getMaster()==null) {
				// Workaround di protezione da assenza database master
				if (log.isDebugEnabled()) log.debug("Stage mode only, use only WRITABLE connection manager!");	
				cmWritable = balance.getStage();
				cmReadonly = cmWritable;
			} else {
				// uso MASTER
				cmWritable = balance.getMaster();
				cmReadonly = balance.getMasterReadOnly();
			}
		}	
	}
	
	
	/**
	 * Set
	 * @param sorgente
	 * @param variabile
	 * @param prefisso
	 * @return
	 * @throws CheckerException
	 */
	protected Object get(Object sorgente, String variabile, String prefisso) throws CheckerException {
		String nome = prefisso+XString.toUpperCaseFirst(variabile);
		if (log.isDebugEnabled()) log.debug("Acquire value using "+nome+"()");
		try {
			Method method = sorgente.getClass().getMethod(nome);
	        return method.invoke(sorgente);
		/*
		} catch (IllegalArgumentException e) {
		} catch (IllegalAccessException e) {
		} catch (InvocationTargetException e) {
		} catch (SecurityException e) {
		} catch (NoSuchMethodException e) {
		} catch (CloneNotSupportedException e) {
		*/
		} catch (Exception e) {
			log.error("Error invoke instance of "+(sorgente!=null?sorgente.getClass().getSimpleName():"..."),e);
			throw new CheckerException(ResponseOutcome.INVALID_METHOD_CHECKING,"'"+nome+"'");
		}		
	}
	
	
	protected void checkRange(String variabile, Integer min, Integer max) throws CheckerException {
		try {
			Integer ret = (Integer) get(request, variabile, "get");
			
			if ((min!=null && ret.intValue()<min) || 
				(max!=null && ret.intValue()>max)) {
				if (min==null && max!=null) throw new CheckerException(ResponseOutcome.FIELD_OUT_OF_RANGE, "'"+variabile+"' <= "+max);
				if (min!=null && max==null) throw new CheckerException(ResponseOutcome.FIELD_OUT_OF_RANGE, "'"+variabile+"' >= "+min);
				throw new CheckerException(ResponseOutcome.FIELD_OUT_OF_RANGE, "'"+variabile+"', range valid ("+min+":"+max+")");
			} 
			
		} catch (ClassCastException e) {
			throw new CheckerException(ResponseOutcome.FIELD_NOT_VALID, "'"+variabile+"'");
		}
	}
	
	
	@SuppressWarnings("unchecked")
	protected <K> K checkRequestIfNullSetDefault(String variabile, K nullValue) throws CheckerException {
		K ret = (K) get(request, variabile, "get");
		if (ret==null) {			
			set(request, variabile, "set", nullValue);
			return nullValue;
		}
		return ret;
	}
	
	
	/**
	 * Restituisce una eccezione in caso di mancata verifica del campo obbligatorio
	 * @param sorgente
	 * @param variabile
	 * @return
	 * @throws CheckerException
	 */
	protected void checkRequestMandatory(String variabile) throws CheckerException {
		checkMandatory(request, variabile, "get");
	}
	
	protected void checkRequestMandatory(String variabile, String prefisso) throws CheckerException {
		checkMandatory(request, variabile, prefisso);
	}
	
	void checkMandatory(REQ sorgente, String variabile, String prefisso) throws CheckerException {
		Object ret = get(sorgente, variabile, prefisso);
		//if (log.isTraceEnabled()) log.trace("Checking mandatory: "+nome+"()");
		if (ret==null) {
			throw new CheckerException(ResponseOutcome.FIELD_MANDATORY, "'"+variabile+"'");
		} else if (ret instanceof String) {
			if (XString.isBlankNullTrim((String)ret)) {
				throw new CheckerException(ResponseOutcome.FIELD_EMPTY, "'"+variabile+"'");
			}
		}	
	}
	
	
	
	
	
	
	/**
	 * Valorizza il metodo "set" del bean relativo alla request
	 * @param <K>
	 * @param sorgente
	 * @param variabile
	 * @param prefisso
	 * @param value
	 * @throws CheckerException
	 */
	private <K> void set(REQ sorgente, String variabile, String prefisso, K value) throws CheckerException {
		if (value==null) return;
		
		String nome = prefisso+XString.toUpperCaseFirst(variabile);
		Class<?> clazz = value.getClass();
		if (log.isTraceEnabled()) log.trace("Set default value using "+nome+"("+clazz.getSimpleName()+" value)");
		try {
			Method method = request.getClass().getMethod(nome, clazz);
	        method.invoke(request, value);
		/*
		} catch (IllegalArgumentException e) {
		} catch (IllegalAccessException e) {
		} catch (InvocationTargetException e) {
		} catch (SecurityException e) {
		} catch (NoSuchMethodException e) {
		} catch (CloneNotSupportedException e) {
		*/
		} catch (Exception e) {
			log.error("Error invoke method "+nome+"("+clazz.getSimpleName()+" value) of "+(sorgente!=null?sorgente.getClass().getSimpleName():"..."),e);
			throw new CheckerException(ResponseOutcome.INVALID_METHOD_CHECKING,"'"+nome+"'");
		}			
	}
	
	
	protected ResourceBundle getResourceBundle() {
		// Inizializza il bundle per la lingua passata a parametro
		inizializeBundle(session_language);
		return session_bundle;
	}
	
	private void inizializeBundle(String language) {
		if (session_bundle!=null) return;
		Locale.setDefault(Locale.ENGLISH);
		
		if (language!=null) {
			language = language.toLowerCase();
			
			try {
				Locale locale = new Locale(language);
				if (log.isDebugEnabled()) {
					log.debug("Locale: "+locale);					
				}
				
				// -----------------------------------------------------------------------------
				try {
					// Controllo eventuali dizionari da escludere (in test)
					String exclusion = cacheInfo.getDbProperties().get(PropertiesKey.LANGUAGE_DISABLE_LIST.toString(),"ZZ").toLowerCase();
					if (exclusion.indexOf(language)>=0) {
						log.warn("Dictionary exclusion: "+locale);		
						locale = Locale.ENGLISH;
					}
				} catch (Exception e) {
					log.warn("Error in check disable language list:" +e.toString());
				}
				// -----------------------------------------------------------------------------			
				
				session_bundle = ResourceBundle.getBundle(ResponseOutcome.BUNDLE_NAME,  locale);	
				return;
			} catch (Exception e) {
				log.warn("Error loading resource bundle for language '"+language+"'", e);
			}			
		}
		
		// ...riprovo in inglese
		try {
			session_bundle = ResourceBundle.getBundle(ResponseOutcome.BUNDLE_NAME, Locale.ENGLISH);
			return;
		} catch (Exception e) {
			log.warn("Error loading default english (en) resource bundle", e);		
		}		
	}
	

	public boolean isStageMode() {
		return stageMode;
	}

	public DatabaseBalancer getDatabaseBalancer() {
		return balance;
	}
	
	
	
	
	/**
	 * Costruttore della risposta al client
	 */
	public class ResponseBuilder {

		private RES response;
		private String detailReturnMessage;

		
		public ResponseBuilder(RES response) {
			this.response = response;
		}
		
		/**
		 * Setup di un result custom
		 * @param result valore di {@link ResponseResult}
		 * @return
		 */
		public ResponseBuilder setupResult(ResponseResult result) {	
			return setReturn(null, null, result); 
		}
				
		/**
		 * Rispondo con un messaggio codificato nelle {@link ResponseOutcome}
		 * @param outcome
		 * @return
		 */
		public ResponseBuilder setReturn(ResponseOutcome outcome) {	
			return setReturn(outcome,null); 
		}
		
		
		/**
		 * Rispondo con un messaggio codificato nelle {@link ResponseOutcome}, valorizzando anche il {@link ResponseResult}
		 * @param outcome
		 * @param result
		 * @return
		 */
		public ResponseBuilder setReturn(ResponseOutcome outcome, ResponseResult result) {
			return setReturn(outcome.getReturnCode(), outcome.getMessage(), result, outcome.isOk(), outcome.isTraceLog());
		}
		
		/**
		 * Risposta non codificata, recuperata dall'esterno
		 * @param code
		 * @param message
		 * @param result
		 * @return
		 */
		public ResponseBuilder setReturn(String code, String message, ResponseResult result) {
			boolean positive = ResponseOutcome.checkOK(code);
			return setReturn(code, message, result, positive, !positive);			
		}
				
		/**
		 * Setup della risposta
		 * @param code
		 * @param message
		 * @param result
		 * @param positive
		 * @param trace
		 * @return
		 */
		private ResponseBuilder setReturn(String code, String message, ResponseResult result, boolean positive, boolean trace) {
			if (response.takeMyReturn()!=null) {
				// Aggiorno un record precedente
				MyReturn myreturn = response.takeMyReturn();
				// ...imposto solo !=null, quindi se cambia
				if (code!=null) myreturn.setCode(code);
				if (message!=null) myreturn.setMessage(message);
				if (result!=null) myreturn.setResult(result);
				//
				myreturn.setPositive(positive);
				myreturn.setTrace(trace);
			} else {
				response.setReturn(new MyReturn(code, message, result, positive, trace));
			}
			
			return this;
		}
		
		/**
		 * Rispondo con una {@link CheckerException}
		 * @param exception
		 * @return
		 */
		protected ResponseBuilder setReturn(CheckerException exception) {
			// ...server per la successiva internazionalizzazione
			this.detailReturnMessage = exception.getSupplementaryMessage();
			return setReturn(exception.getId(), exception.getStandardMessage(), exception.getResult(), false, exception.isTraceLog());
		}
		
		/**
		 * Rispondo con una {@link RemoteCallException}
		 * @param exception
		 * @return
		 */
		protected ResponseBuilder setReturn(RemoteCallException exception) {
			return setReturn(exception.getId(), exception.getMessage(), null, false, true);
		}
		
		/**
		 * Rispondo con un {@link ManagerException}
		 * @param exception
		 * @return
		 */
		protected ResponseBuilder setReturn(ManagerException exception) {
			// ...server per la successiva internazionalizzazione
			this.detailReturnMessage = exception.getCause().toString();
			return setReturn(ResponseOutcome.MANAGER_EXCEPTION, ResponseResult.ERROR);
		}
		
		
		
		/**
		 * Invia una mail al gestore del servizio per segnalare specifici errori
		 * @param outcome
		 * @param cause
		 */
		public ResponseBuilder sendEmailAlert(ResponseOutcome outcome, Throwable cause) {
			EnterpriseLog elog = new EnterpriseLog(outcome.getFullMessage(), cause);
			if (ResponseOutcome.SQL_EXCEPTION.equals(outcome)) {
				elog.sendEmail(PortalSettings.MAIL_SQL_ERROR_ENABLE, outcome.getMessage());
			} else {
				elog.sendEmail(PortalSettings.MAIL_ERROR_ENABLE, outcome.getMessage());
			}
			return this;
		}
		
		/**
		 * Decodifica il messaggio di errore passato nella response secondo la lingua passata a parametro
		 * @param response
		 * @return 
		 */
		private RES decodeMessageErrorWithBundle() {
			if (response==null) return null;
			ResourceBundle bundle = getResourceBundle();
			
			try {
				String text = bundle.getString(response.getReturnCode());
				response.takeMyReturn().setMessage(text);
			} catch (Exception e) {
				if (bundle!=null) {
					if (!bundle.getLocale().equals(Locale.ENGLISH)) {
						log.warn("Entry '"+response.getReturnCode()+"' not found in bundle file for language '"+bundle.getLocale().getLanguage()+"'");
					}
				} else {
					log.error("Not found any buondle file to decode entry '"+response.getReturnCode()+"'");
				}
			}
			
			if (this.detailReturnMessage!=null) {
				// Aggiungo il messaggio dettagliato
				response.takeMyReturn().setMessage(response.getReturnMessage() + " ["+this.detailReturnMessage+"]");
			}
			
			return response;
		}

		
		/**
		 * Restituisce un puntamento alla RESPONSE
		 * @return
		 */
		public RES getResponse() {
			return response;
		}
	}
	

	protected REQ getRequest() {
		return request;
	}
	
	protected RES getResponse() {
		return response;
	}

	protected S getSession() {
		return session;
	}
	
	
	/**
	 * Esegue il parsing del tracciato Json valorizzando l'opportuna classe
	 * @param json
	 * @param clazz
	 * @return
	 * @throws ManagerException 
	 */
	protected <K> K parseJson(String json, Class<K> clazz) throws ManagerException {
		try {
			return new ObjectMapper().readValue(json, clazz);
		} catch (JsonParseException e) {
			throw new ManagerException(new CheckerException(ResponseOutcome.MALFORMD_JSON_DOCUMENT,"JsonParseException: " +clazz.getName())); 
		} catch (JsonMappingException e) {
			throw new ManagerException(new CheckerException(ResponseOutcome.MALFORMD_JSON_DOCUMENT,"JsonMappingException: "+clazz.getName())); 
		} catch (IOException e) {
			throw new ManagerException("IOException parsing json to class "+clazz.getName(),e); 
		}		
	}
	
	@SuppressWarnings("unchecked")
	protected <J extends Serializable> J parseJSONInputStream(Class<J> clazz) throws ManagerException {
		if (request instanceof InputStreamReaderJSON<?>) {
			if (clazz==null) return null; 
			InputStreamReaderJSON<?> reqJSON = (InputStreamReaderJSON<?>)request;
			try {
				@SuppressWarnings("rawtypes")
				JsonMapper jmapper = new JsonMapper();
				reqJSON.bufferizeInputDocument(jmapper);
				return (J) jmapper.readValue(reqJSON.takeInputStream(), clazz);
				
			} catch (JsonParseException e) {
				throw new ManagerException(new CheckerException(ResponseOutcome.MALFORMD_JSON_DOCUMENT,"JsonParseException: " +clazz.getName())); 
			} catch (JsonMappingException e) {
				throw new ManagerException(new CheckerException(ResponseOutcome.MALFORMD_JSON_DOCUMENT,"JsonMappingException: "+clazz.getName()+":"+e)); 
			} catch (IOException e) {
				throw new ManagerException("IOException parsing json to class "+clazz.getName(),e); 
			}	
		} else {
			log.warn("Not valid request: not extends "+InputStreamReaderJSON.class.getName());
		}
		
		return null;	
		
	}
	
	@SuppressWarnings("unchecked")
	protected <X extends XmlRoot> X parseXMLInputStream(X root) throws ManagerException, CheckerException {
		if (request instanceof InputStreamReaderXML) {
			if (root==null) return null; 
			InputStreamReaderXML<XmlRoot> reqXML = (InputStreamReaderXML<XmlRoot>)request;
			
			if (log.isDebugEnabled()) log.debug("Analyze XML document in request input stream");
			try {
				boolean read = root.parseXML(reqXML.takeInputStream());
				reqXML.bufferizeInputDocument(root);
				if (!read) {
					throw new CheckerException(ResponseOutcome.INPUT_FORWARD_XML_NULL);
				}
				if (log.isDebugEnabled()) {
					String xml = root.extractInputXML();
					if (xml!=null) log.debug("Request XML input stream >>\n"+
											 "-[ XML ]------------------------------------------------------------->>\n"+
											  xml +
											 "--------------------------------------------------------------------->>");
				}
			} catch (XmlParserException e) {
				reqXML.bufferizeInputDocument(root);
				if (e.isGrave()) {
					throw new ManagerException(e);				
				} else {
					throw new CheckerException(ResponseOutcome.MALFORMD_XML_DOCUMENT,e.getMessage());
				}
			}
			if (log.isDebugEnabled()) log.debug("Parsing of XML document completed; generated class "+root.getClass().getSimpleName());
		} else {
			log.warn("Not valid request: not extends "+InputStreamReaderXML.class.getName());
		}
		
		return root;
	}
	
	
	/**
	 * Classe private per il lancio asincrono del metodo complete
	 * Evoluzione 2012-07-04 
	 * Lancio il complete in modo asincrono
	 * Questo mi assicura la chiusura della connessione dello stream della chiamta HTTP
	 *
	 * @author m.veroni
	 *
	 */
	private class ActionCompleted extends DelayedExecution {
		
		private boolean stage;
		
		private ActionCompleted (boolean stage) {
			this.stage = stage;
		}
		
		@Override
		public void core() {
			if (log.isDebugEnabled()) log.debug("Launch deferred asynchronous action after completed with success (phase 5)");			
			try {
				completeWithSuccess(stage);
			} catch (ManagerException e) {
				log.error("Manager exception in complete method",e);			
				// ...non blocco il flusso
				// Invio una mail al gestore del servizio
				new ResponseBuilder(null).sendEmailAlert(ResponseOutcome.UNHANDLED_EXCEPTION, e);
				// --------------------------------------
			} catch (Exception e) {		
				log.error("Unhandled exception in complete method",e);			
				// ...non blocco il flusso
				// Invio una mail al gestore del servizio
				new ResponseBuilder(null).sendEmailAlert(ResponseOutcome.UNHANDLED_EXCEPTION, e);
				// --------------------------------------
			}		
			
		}
		
	}
	
	
}
