package it.softecspa.fileproxy.services.common.core;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.fileproxy.services.ServerCacheFactory;
import it.softecspa.fileproxy.services.common.CheckerException;
import it.softecspa.fileproxy.services.common.ManagerException;
import it.softecspa.fileproxy.services.common.ResponseOutcome;
import it.softecspa.fileproxy.services.common.core.request.AuthIsMandatory;
import it.softecspa.fileproxy.services.common.core.request.Protocollo;
import it.softecspa.fileproxy.services.common.core.request.UniversalAuthenticatedRequest;
import it.softecspa.fileproxy.services.common.core.request.UniversalRequest;
import it.softecspa.portal.DMSsoClientSettings;
import it.softecspa.portal.DMSsoProxedSettings;
import it.softecspa.sso.client.HttpSsoController;
import it.softecspa.sso.common.SsoErrorOutcome;
import it.softecspa.sso.common.SsoException;
import it.softecspa.sso.common.TokenInfo;
import it.softecspa.sso.proxy.Authorization;
import it.softecspa.sso.proxy.ProxySsoStub;

import java.io.IOException;
import java.sql.SQLException;

import org.apache.log4j.Logger;

public abstract class UniversalSession <I extends UniversalRequest> {

	
	protected Logger log = Logger.getLogger(this.getClass());
	protected ConnectionManager cmReadonly;
	
	protected ServerCacheFactory cacheInfo;
	protected boolean stageMode;
	protected String language;
	protected Protocollo protocolVersion;
	
	// Variabile di appoggio per la request
	protected I request;
	
	
	public UniversalSession(ServerCacheFactory cache, I request, boolean stage, ConnectionManager cmReadonly) {
		this.cacheInfo = cache;
		this.stageMode = stage;
		preset(request);
		//
		this.cmReadonly = cmReadonly;
	}
	
	private void preset(I request) {
		this.request = request;
		
		// Bufferizzo la request per chiamate successive
		this.language = request.getLanguage();	
		
		// Setup del default del protocollo
		if (request.getProtocolVersion()==null) {
			request.setProtocolVersion(Protocollo._1);
		}
		
		protocolVersion = request.getProtocolVersion();
		if (log.isDebugEnabled()) log.debug("Request/response required with protocol version "+protocolVersion);
		
	}

	public abstract void validate() throws ManagerException, CheckerException;
	

	public String getLanguage() {
		return language;
	}

	public void setLanguage(String language) {
		this.language = language;
	}


	public ServerCacheFactory getCacheInfo() {
		return cacheInfo;
	}


	public boolean isStageMode() {
		return stageMode;
	}
	
	
	/**
	 * Restituisce la versione del protocollo della chiamta in corso
	 * @param protocolVersion
	 */
	public Protocollo getProtocolVersion() {
		return protocolVersion;
	}
	
	
	
	/**
	 * Verifica le credenziali immesse nella chiamata
	 * @param request
	 * @throws CheckerException 
	 * @throws ManagerException 
	 */
	public void validateCredential() throws CheckerException, ManagerException {
		if (!(this.request instanceof UniversalAuthenticatedRequest)) return;
		UniversalAuthenticatedRequest request = (UniversalAuthenticatedRequest)this.request;
		boolean mandatory = (this.request instanceof AuthIsMandatory);
		
		if (mandatory) {
			if (request.getUsername()!=null) {
				if (request.getPassword()==null)
					throw new CheckerException(ResponseOutcome.PASSWORD_MANDATORY);
			} else if (request.getAccessToken()==null) {
				throw new CheckerException(ResponseOutcome.ACCESS_TOKEN_USERNAME_MANDATORY);
			}
		}
		
		if (request.getAccessToken()!=null) {
			HttpSsoController controller = null;
			try {
				controller = new HttpSsoController(DMSsoClientSettings.getInstance().getSettings());
				if (request.getOrganization()!=null) {
					// Preimposto l'organizzazione di default
					if (log.isDebugEnabled()) log.debug("Found 'organization' parameter in request call with value '"+request.getOrganization()+"'");
					controller.presetOrganization(request.getOrganization());
				}
			} catch (IOException e) {
				throw new ManagerException(e);
			} catch (SsoException e) {
				throw new ManagerException(e);
			}
			
			
			try {
				TokenInfo tokenInfo = controller.getSsoStub().acquireTokenInfo(request.getAccessToken());
				request.setUsername(tokenInfo.getName());
				if (tokenInfo.getAttributes()!=null) {
					request.setOrganization(tokenInfo.getAttributes().get("organization"));
				}
			} catch (SsoException e) {
				throw new ManagerException(new CheckerException(ResponseOutcome.SSO_GET_TOKEN_INFO_ERROR, e.getLocalizedMessage()));
			}

		} else if (request.getUsername()!=null) {
			ProxySsoStub proxy = null;
			try {
				proxy = new ProxySsoStub(DMSsoProxedSettings.getInstance().getSettings());
			} catch (IOException e) {
				throw new ManagerException(e);
			} catch (SsoException e) {
				throw new ManagerException(e);
			}
						
			try {
				if (request.getOrganization()!=null) {
					if (log.isDebugEnabled()) log.debug("Found 'organization' parameter in request call with value '"+request.getOrganization()+"'");
				}
				Authorization auth = proxy.login(request.getOrganization(), request.getUsername(), request.getPassword());
				if (!auth.isLogged()) {
					if (auth.getStatusCode().equals(SsoErrorOutcome.WRONG_CREDENTIAL.getReturnCode())) {
						throw new ManagerException(new CheckerException(ResponseOutcome.SSO_WRONG_CREDENTIAL));
					}
					throw new ManagerException(new CheckerException(ResponseOutcome.SSO_LOGIN_ERROR, auth.getStatusMessage()));
				}
				request.setAccessToken(auth.getToken().getValue());
				
			} catch (SsoException e) {
				throw new ManagerException(new CheckerException(ResponseOutcome.SSO_LOGIN_ERROR, e.getLocalizedMessage()));
			}
		}
	}
	
}
