package it.softecspa.fileproxy.services.common.core;


import it.softecspa.fileproxy.services.ClusterSynchronizer;
import it.softecspa.fileproxy.services.HttpClazzTransformer;
import it.softecspa.fileproxy.services.HttpClazzTransformer.HttpRequestSource;
import it.softecspa.fileproxy.services.common.CheckerException;
import it.softecspa.fileproxy.services.common.ResponseOutcome;
import it.softecspa.fileproxy.services.common.ResponseResult;
import it.softecspa.fileproxy.services.common.TraceableWithIdException;
import it.softecspa.fileproxy.services.common.UniversalStatementException;
import it.softecspa.fileproxy.services.common.comunicator.http.HttpRemote;
import it.softecspa.fileproxy.services.common.core.request.InputStreamReaderJSON;
import it.softecspa.fileproxy.services.common.core.request.InputStreamReaderXML;
import it.softecspa.fileproxy.services.common.core.request.JsonMapper;
import it.softecspa.fileproxy.services.common.core.request.MethodPostMandatory;
import it.softecspa.fileproxy.services.common.core.request.MultipartFormDataRequestReader;
import it.softecspa.fileproxy.services.common.core.request.OutputPrintWriter;
import it.softecspa.fileproxy.services.common.core.request.OutputStreamWriter;
import it.softecspa.fileproxy.services.common.core.request.RemoteHostRequestInfo;
import it.softecspa.fileproxy.services.common.core.request.TrackRequestCustomInfo;
import it.softecspa.fileproxy.services.common.core.request.TrackRequestInfo;
import it.softecspa.fileproxy.services.common.core.request.UniversalRequest;
import it.softecspa.fileproxy.services.common.core.response.AbstractDummyResponse;
import it.softecspa.fileproxy.services.common.core.response.Entry;
import it.softecspa.fileproxy.services.common.core.response.UniversalResponse;
import it.softecspa.fileproxy.services.common.core.response.body.BodyNull;
import it.softecspa.fileproxy.services.common.core.response.body.ErrorBody;
import it.softecspa.fileproxy.services.common.core.response.body.MyBody;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.log.Ticket;
import it.softecspa.kahuna.util.Security;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.kahuna.util.xml.XmlRoot;
import it.softecspa.portal.ApplicationClusterInfo;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.zip.GZIPInputStream;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import javazoom.upload.MultipartFormDataRequest;
import javazoom.upload.UploadException;

import org.apache.log4j.Logger;

@SuppressWarnings("serial")
public abstract class UniversalStatementHttpServlet <UREQ extends UniversalRequest, URES extends UniversalResponse> extends HttpServlet {

	/**
	 * Classe privata che contiene la configurazione della servlet
	 * @author m.veroni
	 */
	private class Configuration {
		
		// Modalit di acquisizione paramatri in ingresso
		ModeRequest config_requestMode;		
		
		// Modalit di risposta al chiamante
		ModeResponse config_responseMode;
		
		// Struttura body di risposta
		ErrorBody config_errorBody;
				
		// Prefisso da utilizzare per il nome delle variabili nell'header http
		String config_headerPrefix;
		// Flag che indica l'utilizzo della variabile di header "RESULT"
		boolean config_useResult;
				
		
		public String getMethodNamePrefix() {
			return config_headerPrefix;
		}
		
		public boolean isUseResult() {
			return config_useResult;
		}

		public ModeRequest getRequestMode() {
			return config_requestMode;
		}

		public ModeResponse getResponseMode() {
			return config_responseMode;
		}
		
		public ErrorBody getErrorBody() {
			return config_errorBody;
		}
		
	}
	
	
	/**
	 * Classe pubblica che consente la modifica della configurazione della servler
	 * @author m.veroni
	 */
	public class Configurable extends Configuration {
		
		public void setHeaderPrefix(String value) {
			if (value==null) value = "";
			super.config_headerPrefix = value.trim();			
		}
		
		public void setUseResult(boolean value) {
			super.config_useResult = value;
		}
				
		private String formatHeader(HeaderKey value) {
			return config_headerPrefix + value.toString();			
		}		
		
		
		public void setRequestMode(ModeRequest mode) {
			super.config_requestMode = mode;
		}		
		
		public void setResponseMode(ModeResponse mode) {
			super.config_responseMode = mode;
		}
			
		public void setErrorBody(ErrorBody errorBody) {
			super.config_errorBody = errorBody;
			
		}
		
		
		public String toString() {
			return "{" + "'"+config_headerPrefix+"'" +
				   "," + config_useResult +
				   "," + config_requestMode.name() +
				   "," + config_responseMode.name() + 
				   "," + (config_errorBody!=null?config_errorBody.getClass().getSimpleName():null) + 
				   "}";		  
		}

		
		
	}
	
	private Configurable myConfig;
	private HttpClazzTransformer clazzT;
	private ApplicationClusterInfo clusterInfo;
	

	private enum HeaderKey {
		
		  STATEMENT			("statement")
		/*
		// Trasmissione attachment
		, CONTENT_LENGTH("Content-Length")
		, ATTACH_LENGTH("Attach-Length")	
		*/
		
		, HEADER_ENCODING	("header-encoding")
		
		/* Risposta */
		, RESULT			("result")
		, RETURN_CODE		("returnCode")
		, RETURN_MESSAGE	("returnMessage")	
		, JARVIS			("jarvis")	
		;  
		
		  
		private HeaderKey() {}
		
		private HeaderKey(String value) {
			this();
			this.value = value;
		}
		
		private String value;
		
		public String toString() {
			return (value!=null?value:super.toString());
		}

		@SuppressWarnings("unused")
		public String getValue() {
			return toString();
		}

	}
	
	public class DummyResponse extends AbstractDummyResponse {

		public DummyResponse(ResponseOutcome outcome) {
			super(outcome);
		}
		
	}
	
	
	protected Logger log = Logger.getLogger(getClass());
	private Logger statementLog = Logger.getLogger("it.softecspa.desktopmate.STATEMENT");
	
	
	
	/**
	 * Metodo invocato in fase di costruzione della servlet
	 */
	public void init(ServletConfig config) throws ServletException {
		super.init(config);
		
		log.info("Starting servlet "+getClass().getName());
		// Configurazione dei parametri dei default della servlet
		myConfig = new Configurable();	
		//
		myConfig.setHeaderPrefix("X-");
		myConfig.setUseResult(false);
		myConfig.setRequestMode(ModeRequest.HEADER);
		myConfig.setResponseMode(ModeResponse.HEADER);
		//
		configure(myConfig);
		log.info("Servlet configuration complete: "+myConfig);
		//		
		afterStart(myConfig, config);
		
				
		// Configurazione e istanza del clazz transformer
		if (myConfig.getRequestMode().equals(ModeRequest.PARAMETER)) {
			clazzT = new HttpClazzTransformer(HttpRequestSource.HTTP_PARAMETER, myConfig.getMethodNamePrefix());
		} else if (myConfig.getRequestMode().equals(ModeRequest.HEADER)) {
			clazzT = new HttpClazzTransformer(HttpRequestSource.HTTP_HEADER, myConfig.getMethodNamePrefix());
		} else /*if (myConfig.getRequestMode().equals(ModeRequest.MIXED))*/ {
			// default
			clazzT = new HttpClazzTransformer(HttpRequestSource.HTTP_HEADER_PARAMETER, myConfig.getMethodNamePrefix());
		}	
		
		// Recupero l'istanza delle informazioni del cluster
		clusterInfo = ApplicationClusterInfo.getInstance();
	}

	
	/**
	 * Configurazione della servlet
	 * Sono da variare i parametri del bean di configurazione
	 * @param config
	 */
	protected abstract void configure(Configurable myConfig);

	
	/**
	 * Attivit da aeseguire all'init della servlet
	 * @param myConfig
	 * @param config
	 */
	protected abstract void afterStart(Configurable myConfig, ServletConfig config);

	
	/**
	 * Attivit da eseguire alla chiusura della servlet, prima del destroy
	 */
	protected abstract void beforeDestroy();


	/**
	 * Metodo invocato in fase di distruzione della servlet
	 */
	public void destroy() {
		beforeDestroy();
		//
		log.info("Stopping servlet "+getClass().getName());
		super.destroy();
	}	
	
		
	/**
	 * Chiamata in modalit GET
	 */
	protected abstract void doGet(HttpServletRequest request, HttpServletResponse response);
	
	/**
	 * Chiamata in modalit POST
	 */
	protected abstract void doPost(HttpServletRequest request, HttpServletResponse response);
		
	/**
	 * Chiamata in modalit OPTION
	 */
	protected void doOptions(HttpServletRequest request, HttpServletResponse response) {
		response.setHeader("Allow", "GET, POST, OPTIONS");
	}
	
	
	/**
	 * Restituisce una risposta di default
	 * @param request
	 * @param response
	 * @param outcome
	 */
	protected void doDummy(HttpServletRequest request, HttpServletResponse response, ResponseOutcome outcome) {
		responseReturn(new MyHttpServletResponse(request, response), new MyReturn(outcome, null), new DummyResponse(outcome));
	}
	
	
	/**
	 * Esecuzione dello statement richiesto
	 * @param request
	 * @param response
	 */
	@SuppressWarnings("unchecked")
	protected void doStatement(HttpServletRequest request, HttpServletResponse response) {
		// Classe di appoggio per la gestione della response
		MyHttpServletResponse myResponse = new MyHttpServletResponse(request, response);
		
			
		// Verifico la validit della operazione cercando un metodo con lo stesso nome
		try {
			if (log.isDebugEnabled()) log.debug("Analyze operation statement");
			myResponse.setStatementKey(XString.toLowerCaseFirst(checkOperation(request)));
		} catch (UniversalStatementException e) {
			// Se non trovo uno statement restituisco un ERRORE
			responseError(myResponse, e);
			return;
		}
		
		
		try {		
			// Estrazione info sul device remoto
			HttpRemote remote = myResponse.getHttpRemote();
			if (log.isDebugEnabled()) log.debug("Remote call from "+remote.getTraceHost());	
						
			if (log.isDebugEnabled()) log.debug("Operation required: '"+myResponse.getStatementKey()+"', method "+request.getMethod());
			// Se trovo un "-" devo mettere a maiuscolo la lettera seguente
			int pos;
			String statement = new String(myResponse.getStatementKey()!=null?myResponse.getStatementKey():"");
			while ((pos = statement.indexOf("-"))>=0) {
				statement = statement.substring(0,pos)+XString.toUpperCaseFirst(statement.substring(pos+1));
			}			
			myResponse.setStatement(statement);
			
			
			// Derivo REQ e RES del metodo dal nome del comando
			// e ne verifico l'esistenza			
			if (log.isDebugEnabled()) log.debug("Statement class implementation is '"+getClass()+"'");
			@SuppressWarnings("rawtypes")
			Class statementRequestClazz = HttpClazzTransformer.forName(getClass(), statement, "request");
			@SuppressWarnings("rawtypes")
			Class statementResponseClazz = HttpClazzTransformer.forName(getClass(), statement, "response");
			
			
			Class<?>[] parameterTypes = new Class[1];
			parameterTypes[0] = statementRequestClazz;
			
			// Verifico l'esistenza di un metodo corrispondente alla operazione
			Method method = null;
			try {
				method = this.getClass().getMethod(statement, parameterTypes);
			} catch (SecurityException e) {
				log.error("SecurityException",e);
				throw new UniversalStatementException(ResponseOutcome.PS_SECURITY_EXCEPTION);
			} catch (NoSuchMethodException e) {
				log.error("NoSuchMethodException",e);
				throw new UniversalStatementException(ResponseOutcome.PS_NO_SUCH_METHOD);
			}
			
			// Valorizzazione della classe di input
			Object[] parameterArgs = new Object[1];
			parameterArgs[0] = clazzT.newInstance(statementRequestClazz, request, true);			
						
			
			// m.veroni - 2014-06-24 - verifica obbligatoriet del metodo
			if (parameterArgs[0] instanceof MethodPostMandatory) {
				if (!request.getMethod().equalsIgnoreCase("POST")) {
					log.error("Request method not valid, required POST (vs '"+request.getMethod()+"')");
					responseError(myResponse, ResponseOutcome.REQUEST_METHOD_NOT_VALID);
					return;
				}
			}
			
			
			// Stampo i parametri di chiamata
			myResponse.setInput(parameterArgs[0]);
			if (log.isDebugEnabled()) log.debug(requestInfo(myResponse, null));				
			
			// Valorizza i metodi delle interfacce custom
			customizeRequestInterface(myResponse, parameterArgs[0]);
			
			
			// Invoke della operazione nel manager
			// ------------------------------------------------------------------------------------------
			Object output = null;
			try {
				if (log.isDebugEnabled()) log.debug("Invoke operation method");
				output = method.invoke(this, parameterArgs);
				myResponse.setOutput(output);
			} catch (IllegalArgumentException e) {
				log.error("IllegalArgumentException",e);
				responseError(myResponse, ResponseOutcome.PS_SECURITY_EXCEPTION);
				return;
			} catch (IllegalAccessException e) {
				log.error("IllegalAccessException",e);
				responseError(myResponse, ResponseOutcome.PS_SECURITY_EXCEPTION);
				return;
			} catch (InvocationTargetException e) {
				log.error("InvocationTargetException",e);
				responseError(myResponse, ResponseOutcome.PS_SECURITY_EXCEPTION);
				return;
			}
			// ------------------------------------------------------------------------------------------
			
			if (output==null) {
				responseError(myResponse, ResponseOutcome.PS_NULL_RESPONSE);
				return;			
			}		
			
			
			if (log.isTraceEnabled()) {
				log.trace("Expected response class is "+statementResponseClazz.getName());
				log.trace("  Output response class is "+output.getClass().getName());
			}
			
			if (!output.getClass().getName().equals(statementResponseClazz.getName())) {
				responseError(myResponse, new CheckerException(ResponseOutcome.PS_RESPONSE_NOT_MATCH, "return "+output.getClass()+", required "+statementResponseClazz.getName()));
				return;
			}
			
			
			// Esito postitivo
			if (log.isDebugEnabled()) log.debug("Create response for class "+statementResponseClazz);	
			URES res = (URES)output;
			if (myConfig.isUseResult()) {				
				// Valorizzazione automatica del valore "result" in caso di assenza
				if (res.takeMyReturn()!=null && res.takeMyReturn().getResult()==null) {
					if (ResponseOutcome.checkOK(res.getReturnCode())) {
						res.takeMyReturn().setResult(ResponseResult.OK);
					} else {
						res.takeMyReturn().setResult(ResponseResult.ERROR);
					}
					if (log.isDebugEnabled()) log.warn("Automatic result value development: "+res.takeMyReturn().getResult());
				} 
			}
			
			MyReturn myReturn = res.takeMyReturn();
			if (myReturn==null) {
				log.error(MyReturn.class+" null!");
				myReturn = new MyReturn(ResponseOutcome.INTERNAL_SERVER_ERROR, ResponseResult.ERROR);
				res.setReturn(myReturn);
			}
			
			
			// Imposto il valore del trace in funzione della configurazione
			if (!log.isDebugEnabled() && 
				!myReturn.isNegative() &&
				myResponse.getInput() instanceof TrackRequestInfo) {
				myReturn.setTrace(((TrackRequestInfo)parameterArgs[0]).isTrackRequestInfo());			
			}
			
			
			// Aggiunta di informazioni accessorie recuperate dalla request
			if (myResponse.getInput() instanceof TrackRequestCustomInfo) {
				// Stampo le info recuperate delle chiamata 
				TrackRequestCustomInfo input = (TrackRequestCustomInfo)myResponse.getInput();
				if (input!=null) {
					myResponse.setInfo(input.takeCustomEntry());
				}
			}
			
			
			// ----
			responseReturn(myResponse, myReturn, res);
				
		} catch (UniversalStatementException e) {
			responseError(myResponse, e);
			return;
		}
		
	}
	

	
	private void customizeRequestInterface(MyHttpServletResponse myresponse, Object input) {
		HttpServletRequest request = myresponse.getHttpServletRequest();
		HttpServletResponse response = myresponse.getHttpServletResponse();
		
		/*
		 * In alcuni casi ho necessit di leggere l'input stream
		 * Posso farlo solo se la request type del manager implementa InputStreamReader
		 * 
		 * ...lo passo all'oggetto request type del manager
		 */
		if (input instanceof InputStreamReaderXML || input instanceof InputStreamReaderJSON) {
			if (log.isDebugEnabled()) log.debug("Put servlet input stream in "+input.getClass());				
			try {
				InputStream instream = null ;				
				
				// Gestione della compressione dello InputStream (GZIP)
				// m.veroni, 2012-10-08 introdotto per compatibilit
				String contentEncoding = request.getHeader("Content-Encoding");
				if ("gzip".equalsIgnoreCase(contentEncoding)) {
					if (log.isDebugEnabled()) log.debug("Compressed input stream: gzip");
				    instream = new GZIPInputStream(request.getInputStream());
				} else {
					instream = request.getInputStream();
				}
								
				if (input instanceof InputStreamReaderXML) {
					((InputStreamReaderXML<?>)(input)).putInputStream(instream);
				} else if (input instanceof InputStreamReaderXML || input instanceof InputStreamReaderJSON) {
					((InputStreamReaderJSON<?>)(input)).putInputStream(instream);
				}
			} catch (IOException e) {
				responseError(myresponse, new CheckerException(ResponseOutcome.PS_INPUT_STREAM_ERROR, e.toString()));
				return;
			}
		}
		
		// Ho una richiesta di tipo MULTIPART-FORM-DATA
		if (input instanceof MultipartFormDataRequestReader) {
			if (log.isDebugEnabled()) log.debug("MultipartFormDataRequest required in http request in "+input.getClass());	
			
			MultipartFormDataRequest mfdrequest = null;
			if (MultipartFormDataRequest.isMultipartFormData(request)) {
				try {
					mfdrequest = new MultipartFormDataRequest(request);
				} catch (IOException e) {
					responseError(myresponse, new CheckerException(ResponseOutcome.MFD_REQUEST_IO_ERROR, e.toString()));
					return;
				} catch (UploadException e) {
					responseError(myresponse, new CheckerException(ResponseOutcome.MDF_REQUEST_UPLOAD_ERROR, e.toString()));
					return;
				} catch (Exception e) {
					responseError(myresponse, new CheckerException(ResponseOutcome.MFD_REQUEST_IO_ERROR, e.toString()));
					return;
				}
				
				((MultipartFormDataRequestReader)(input)).putMultipartFormData(mfdrequest);
			}
		}
		
		// ...o di aggiungere altre informazioni
		if (input instanceof RemoteHostRequestInfo) {
			RemoteHostRequestInfo req = (RemoteHostRequestInfo)input;
			
			// Valorizzazione delle informazioni reltive al "remote host"
			req.putHttpRemote(myresponse.getHttpRemote());
		}
		
		
		// ---------------------------------------------
		
		/*
		 * In alcuni casi ho necessit di scrivere direttamente nell'output stream
		 * Posso farlo solo se la response type del manager implementa OutputStreamWriter
		 * 
		 * ...lo passo all'oggetto response type del manager
		 */
		if (input instanceof OutputStreamWriter) {
			if (log.isDebugEnabled()) log.debug("Put servlet output stream in "+input.getClass());				
			try {
				((OutputStreamWriter)(input)).putOutputStream(response.getOutputStream());
			} catch (IOException e) {
				responseError(myresponse, new CheckerException(ResponseOutcome.PS_OUTPUT_STREAM_ERROR, e.toString()));
				return;
			}			
		}
		
		/*
		 * In alcuni casi ho necessit di scrivere direttamente nel print writer
		 * Posso farlo solo se la response type del manager implementa OutputPrintWriter
		 * 
		 * ...lo passo all'oggetto response type del manager
		 */
		if (input instanceof OutputPrintWriter) {
			if (log.isDebugEnabled()) log.debug("Put servlet print writer in "+input.getClass());				
			try {
				response.setCharacterEncoding("utf-8");	
				response.setContentType("text/plain; charset=\"utf-8\"" );
		        //
				((OutputPrintWriter)(input)).putPrintWriter(response.getWriter());
			} catch (IOException e) {
				responseError(myresponse, new CheckerException(ResponseOutcome.PS_OUTPUT_STREAM_ERROR, e.toString()));
				return;
			}			
		}
				
	}

	private String checkOperation(HttpServletRequest request) throws UniversalStatementException {
		
		// OPERATION
		String operationH = request.getHeader(myConfig.formatHeader(HeaderKey.STATEMENT));
		if (XString.isBlankNullTrim(operationH)) {
			// Provo a cercare l'operazione nella QueryString
			String operationQS = getOperationFromQueryString(request);
			if (XString.isBlankNullTrim(operationQS)) {
				// Provo a cercare l'operazioencome parametro
				String operationP = request.getParameter(HeaderKey.STATEMENT.toString());
				if (XString.isBlankNullTrim(operationP)) {
					throw new UniversalStatementException(ResponseOutcome.PS_STATEMENT_NOT_SPECIFIED);
				}
				return operationP;
			}	
			return operationQS;
		}
		
		String operationQS = getOperationFromQueryString(request);
		if (XString.isNotBlankNull(operationQS)) {
			if (!operationQS.equals(operationH)) {
				throw new UniversalStatementException(ResponseOutcome.PS_STATEMENT_DIFFERENCE);
			}
		}		 
		return operationH; 
	}

	/**
	 * Estra l'operazione dalla queryString della chiamata http
	 * In caso di chiamata in GET la queryString contine tutti i parametri; lo statemet equivale al primo
	 * 
	 * @param request
	 * @return
	 */
	private String getOperationFromQueryString(HttpServletRequest request) {
		String operationQS = request.getQueryString();
		if (operationQS==null || operationQS.length()==0) return null;
		
		int i = operationQS.indexOf("&");
		if (i>0) operationQS = operationQS.substring(0,i);
		i = operationQS.indexOf("=");
		if (i>0) operationQS = operationQS.substring(0,i);
		return operationQS;
	}
	
	
	private void responseError(MyHttpServletResponse response, TraceableWithIdException exception) {
		responseReturn(response, new MyReturn(exception.getId(), exception.getMessage(), null, false, exception.isTraceLog()), null);
	}
	
	private void responseError(MyHttpServletResponse response, ResponseOutcome outcome) {
		responseReturn(response, new MyReturn(outcome, null), null);
	}
	
	
	/**
	 * Resitituisce la risposta valorizzando le variabili di header
	 * @param response
	 * @param return_code
	 * @param return_message
	 * @param res
	 */
	private void responseReturn(MyHttpServletResponse myResponse, MyReturn myReturn, UniversalResponse res) {
		// Tracciamento informazioni della REQUEST nel log applicativo
		if (myReturn.isNegative()) {
			// ERRORE
			log.error(requestInfo(myResponse, myReturn));
		} else if (myReturn.isTrace() && !log.isDebugEnabled()) {
			// POSITIVO CON MESSAGGIO DA TRACCIARE (no debug)
			log.info(requestInfo(myResponse, myReturn));
		} else {
			// DEBUG
			if (log.isDebugEnabled()) {
				HttpRemote remote = myResponse.getHttpRemote();
				HttpServletRequest request = myResponse.getHttpServletRequest();
				//
				String statement = myResponse.getStatement();
				String statement_key = myResponse.getStatementKey();
				//
				String messaggio = "Response >>\n" +
					 		       "-[INFO ]------------------------------------------------------------>> \n" +
							       "Response: "+ myReturn.toString() + "\n" +
								   (myReturn.getResult()!=null?"Result: '"+myReturn.getResult()+"'\n":"") +
								   "Remote host: "+ remote.getTraceHost() + "\n" +
								   "Request URL: " + request.getRequestURL() + (statement_key!=null && statement_key.length()>0?"?"+statement_key:"") + "\n" +
								   "Statement: " + (statement!=null && statement.length()>0?statement:"") + "\n" +
								   "--------------------------------------------------------------------->>";
				log.debug(messaggio);
			}
		}
		
		@SuppressWarnings("unused")
		HttpServletRequest request = myResponse.getHttpServletRequest();
		HttpServletResponse response = myResponse.getHttpServletResponse();
		/*
		 // Setup charset encoding UTF-8
		 response.setCharacterEncoding("UTF-8");
		 response.setContentType("text/xml");
		 //
		 Omesso per evitare problematiche con trasmissione file e flussi non xml
		 //
		 */
		
		// Encoding della response header
		response.setHeader(myConfig.formatHeader(HeaderKey.HEADER_ENCODING), clazzT.getResponseHeaderEncoding().toString());
		
		/* Preset dei valori dell'HEADER per la risposta */
		clazzT.setEncodedHeader(response, myConfig.formatHeader(HeaderKey.RETURN_CODE), myReturn.getCode());
		clazzT.setEncodedHeader(response, myConfig.formatHeader(HeaderKey.RETURN_MESSAGE), myReturn.getMessage());	
		clazzT.setEncodedHeader(response, myConfig.formatHeader(HeaderKey.JARVIS), clusterInfo.getHostName());
		
		// ...gestione dinamica del "Result"
		if (myConfig.isUseResult()) {
			if (myReturn.getResult()==null)	myReturn.setResult(ResponseResult.ERROR);
			// Il valore del result DEVE essere presente
			clazzT.setEncodedHeader(response, myConfig.formatHeader(HeaderKey.RESULT), myReturn.getResult().toString());
		} else {
			// Il valore del result non  obbligatori ma  inserito solo se valorizzato in modo esplicito
			if (myReturn.getResult()!=null) {
				clazzT.setEncodedHeader(response, myConfig.formatHeader(HeaderKey.RESULT),myReturn.getResult().toString());
			}
		}
		// ....
		
		try {		
			MyBody body = null;
			if (res!=null) {
				// Converto la risposta in Header value
				if (log.isDebugEnabled()) log.debug("Setup header value using response class");
				clazzT.setResponseHeaders(res, response);
				if (res.takeBody()!=null) {
					body = res.takeBody();
				}
			} else {
				/*
				 * Non  stata definita alcuna risposta
				 * Sono sicuramente in una codizione di errore
				 */
			}
			
			if (body==null) {
				// In caso di risposta vuota verifico se  presente una risposta di default
				if (getMyConfig().getErrorBody()!=null) {
					body = getMyConfig().getErrorBody();
				} else {
					body = new BodyNull();
				}
			}
			
			// Eseguo prima questa operazione perch pu essere sovrascritta dal bodywrite
			try {
				body.setMyReturn(myReturn);
				body.write(myResponse);						
			} catch (FileNotFoundException e) {
				log.error(e.toString());	
				myReturn.changeHttpStatus(HttpServletResponse.SC_NOT_FOUND);
				// il segno meno (-) nel codice errore  voluto 
				responseError(myResponse, new UniversalStatementException(""+(-HttpServletResponse.SC_NOT_FOUND), e.toString()));		
			}				
			
			// Tracciamento STATEMENT e esito nel log statement (200)
			traceStatementInfo(myResponse, myReturn);
			response.setStatus(myReturn.getHttpStatus());
						
		} catch (UnsupportedEncodingException e) {
			log.error("Unsupported encoding exception", e);	
			myReturn.changeHttpStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			
			// Tracciamento STATEMENT e esito nel log statement
			traceStatementInfo(myResponse, myReturn);
			response.setStatus(myReturn.getHttpStatus());
						
		} catch (IOException e) {
			log.error("Comunication exception", e);	
			myReturn.changeHttpStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			
			// Tracciamento STATEMENT e esito nel log statement
			traceStatementInfo(myResponse, myReturn);
			response.setStatus(myReturn.getHttpStatus());
			
		} catch (BodyResponseException e) {
			log.error("BodyResponseException", e.getCause());	
			myReturn.changeHttpStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			
			// Tracciamento STATEMENT e esito nel log statement
			traceStatementInfo(myResponse, myReturn);
			response.setStatus(myReturn.getHttpStatus());
		
		} catch (UniversalStatementException e) {
			log.error("Exception", e);	
			myReturn.changeHttpStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			
			// Tracciamento STATEMENT e esito nel log statement
			traceStatementInfo(myResponse, myReturn);
			response.setStatus(myReturn.getHttpStatus());
			
		} catch (Exception e) {
			log.error("Unhandled exception", e.getCause());	
			myReturn.changeHttpStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			
			// Tracciamento STATEMENT e esito nel log statement
			traceStatementInfo(myResponse, myReturn);
			response.setStatus(myReturn.getHttpStatus());
		} 		
	}

	
	
	private void traceStatementInfo(MyHttpServletResponse myResponse, MyReturn myReturn) {
		try {
			if (statementLog.isInfoEnabled()) {
				HttpRemote remote = myResponse.getHttpRemote();
				HttpServletRequest request = myResponse.getHttpServletRequest();
				//
				String statement = myResponse.getStatement();
				String statement_key = myResponse.getStatementKey();
				
				EnterpriseCalendar stop = EnterpriseCalendar.now();
				long life = stop.difference(myResponse.getStart(), EnterpriseCalendar.MILLISECOND);
				
				String messaggio = "";
				
				/* JSON */
				String info = "";
				if (myResponse.getInfo()!=null) {
					for (Entry element : myResponse.getInfo()) {
						info += (info.length()==0?"":", ") + element.toJson();
					}
				}
				info = "{"+info+"}";
				
				// ID univoco = 
				String id = Security.toMD5(stop.getTimeInMillis() + getClass().getName() + remote.getTraceHost() + ClusterSynchronizer.getInstance().getPublicHostName() + Ticket.generaTicket());
				/*
				 * Aggiunto al calclol dell'id univoco un valore random
				 * Questo p dovuto al fatto che certi device eseguono la stessa richista pi volte nello stesso secondo 
				 * 
				 */
				
				messaggio += "{" +
							 "\"_id\":\"" + id + "\", " +
							 "\"host\":\"" + remote.getTraceHost() + "\", " +
							 "\"url\":\"" + request.getRequestURL() + (statement_key!=null && statement_key.length()>0?"?"+statement_key:"") + "\", " +
							 "\"statement\":\"" + (statement!=null && statement.length()>0?statement:"?") + "\", " +
							 "\"class\":\"" + getClass().getName()  + "\", " +
							 "\"time\":\"" + myResponse.getStart().toStringUTC() + "\", " +
							 "\"life\":" + life + ", " +
							 "\"response\":" + myReturn.getHttpStatus() + ", " +
							 "\"code\":\"" + myReturn.getCode() + "\", " +
							 //"\"message\":\"" + myReturn.getMessage() + "\", " + 
							 "\"status\":\"" + (myReturn.isNegative()?"ERROR":"VALID") + "\", " +
							 "\"info\":" + info +
							 "}";
				statementLog.info(messaggio);
			}
		} catch (Exception e) {
			log.error("Error in statement tracking: "+e.toString(),e);
		}
	}


	/**
	 * Stampa le informazioni presenti nell'header o nei parametri della richiesta
	 * La stessa chiamta  utilizzata sia in debug che per tracciare gli errori
	 * @param statement_key
	 * @param myResponse
	 * @param myReturn
	 * @return
	 */
	private String requestInfo(MyHttpServletResponse myResponse, MyReturn myReturn) {
		HttpRemote remote = myResponse.getHttpRemote();
		HttpServletRequest request = myResponse.getHttpServletRequest();
		//
		String statement = myResponse.getStatement();
		String statement_key = myResponse.getStatementKey();
		
		String messaggio = "";
		if (myReturn!=null && myReturn.isNegative()) {
			if (!myReturn.isTrace()) {
				// Risposta di errore compatto
				messaggio = "Response: " + myReturn.toString() + 
										  (myReturn.getResult()!=null?"; '"+myReturn.getResult()+"'":"") + 
										  " {remote host: "+ remote.getTraceHost() + ", request URL: " + request.getRequestURL() + (statement_key!=null && statement_key.length()>0?"?"+statement_key:"") +"}"; 
				return messaggio;
			}
						
			messaggio += "Report an error, request info:\n" +
 		     "-[ERROR]------------------------------------------------------------>> \n" +
		     "Negative response: "+ myReturn.toString() + "\n" +
 		     (myReturn.getResult()!=null?"Result: '"+myReturn.getResult()+"'" + "\n":"") +
 		     "Status http: " + myReturn.getHttpStatus() + "\n";
		
		} else if (myReturn!=null && myReturn.isTrace()) {
			messaggio += "Tracking call, request info:\n" +
			 "-[TRACK]----------------------------------------------------------->> \n";
		} else {
			messaggio += "Request info\n" +
			 "-[INFO ]------------------------------------------------------------->> \n";
		}
		messaggio += "Remote host: "+ remote.getTraceHost() + "\n" +
					 "Request URL: " + request.getRequestURL() + (statement_key!=null && statement_key.length()>0?"?"+statement_key:"") + "\n" +
					 "Statement: " + (statement!=null && statement.length()>0?statement:"") + "\n";
					
					 
		// HEADER
		if (myConfig.getRequestMode().equals(ModeRequest.HEADER) ||
			myConfig.getRequestMode().equals(ModeRequest.HEADER_OR_PARAMETER)) {
			// Stampo tutti gli header
			String values = "";
			
			@SuppressWarnings("unchecked")
			List<String> names = Collections.list(request.getHeaderNames());
			Collections.sort(names);
			
			for (String key : names) {
				if (!key.contains("password")) {
					values += "- "+key+" = '"+request.getHeader(key)+"'\n";
				} else {
					try {
						values += "- "+key+" = '"+XString.fillerLeft("", "*", request.getHeader(key).length())+"'\n";
					} catch (NullPointerException en) {
						values += "- "+key+" = ''\n";
					}
					
				}
			}
			if (values.length()>0) {
				messaggio += "HTTP Header values:\n"+
						  	 values;
			}
		}
		
		/*
		// PARAMETERR
		if (myConfig.getRequestMode().equals(ModeRequest.PARAMETER) ||
			myConfig.getRequestMode().equals(ModeRequest.HEADER_OR_PARAMETER)) {
			// Stampo tutti gli header
			
			String values = "";
			for (@SuppressWarnings("unchecked")Enumeration<String> e = request.getParameterNames(); e.hasMoreElements();) {
				String key = e.nextElement();
				if (key.equals(myResponse.getStatementKey())) continue;
				values += "- "+key+" = '"+request.getParameter(key)+"'\n";
			}
			if (values.length()>0) {
				messaggio += "HTTP Parameter values:\n"+
						     values;
			}
			
		}
		*/
		
		// INPUT XML (opzionale)
		if (myResponse.getInput()!=null) {
			if (myResponse.getInput() instanceof InputStreamReaderXML) {			
				try {
					// Stampa il documento XML nel log					
					@SuppressWarnings("unchecked")
					XmlRoot document = ((InputStreamReaderXML<XmlRoot>) myResponse.getInput()).takeBufferedInputDocument();	
					if (document!=null) {
						String xml = document.extractInputXML();
						if (XString.isNotBlankNull(xml)) {
							messaggio += "Request XML input:\n"+
										 "---------------------------------------------------------------------<<\n" +
								          xml;
						} 
					}
				} catch (Exception e) {
					log.warn("Exception printing input XML",e);		
				} catch (Error e) {
					log.warn("Error printing input XML",e);
				}
			} 
			
			if (myResponse.getInput() instanceof InputStreamReaderJSON) {			
				JsonMapper<?> jmapper = ((InputStreamReaderJSON<?>) myResponse.getInput()).takeJsonDocument();	
				if (jmapper!=null) {
					String json = jmapper.print();
					if (XString.isNotBlankNull(json)) {
						messaggio += "Request JSON input:\n"+
									 "---------------------------------------------------------------------<<\n" +
									 json + "\n";
					} 
				}
					
			} 
			
		}
		
		
		messaggio += "--------------------------------------------------------------------->>";
		
		return messaggio;
	}
	
	


	/**
	 * Configurazione della servlet (readonly)
	 * @return
	 */
	public Configuration getMyConfig() {
		return (Configuration)myConfig;
	}
	
	
}
