package it.softecspa.database.dbconnect;

import java.sql.Connection;
import java.sql.SQLException;

/**
 * Fornisce le connessioni al database e ne gestisce la chiusura.
 * 
 * <p>
 * <b><code>ConnectionManager</code> di default</b>: se l'applicazione si
 * connette ad un solo database  conveniente impostare come default il
 * corrispondente <code>ConnectionManager</code>; sar cos possibile utilizzare
 * i metodi che richiedono la connessione al database senza dover fornire
 * esplicitamente un <code>ConnectionManager</code>.
 * 
 * <p>
 * <b><code>DBMS</code> di default</b>: il primo ConnectionManager creato
 * imposta anche il corrispondente <code>DBMS</code> di default; se
 * l'applicazione si connette a database dello stesso tipo  quindi possibile
 * utilizzare i metodi che richiedono la conoscenza del tipo di dbms senza dover
 * fornire esplicitamente un <code>DBMS</code>.
 * 
 * @author Gino Quaglietta
 */

public abstract class ConnectionManager {
	
	private DBMS dbms;
	private static ConnectionManager defaultConnectionManager;

	/**
	 * Costruisce un <code>ConnectionManager</code> per una data tipologia di
	 * dbms. <br>
	 * Il primo <code>ConnectionManager</code> creato imposta anche il
	 * <code>DBMS</code> di default.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}.
	 */
	protected ConnectionManager(DBMS dbms) {
		if (dbms == null) throw new IllegalArgumentException("L'argomento DBMS non pu essere null");
		this.dbms = dbms;

		// se il DBMS di default non  stato impostato, lo imposta
		DBMS defaultDBMS = null;
		try {
			defaultDBMS = DBMS.getDefaultDBMS();
		} catch (RuntimeException e) {
			// Nessuna operazione
		}
		if (defaultDBMS == null) DBMS.setDefaultDBMS(this.dbms);
	}

	/**
	 * Restituisce una connessione al database.
	 * 
	 * @return la connessione al database
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected abstract Connection getConnection() throws SQLException;

	
	
	
	/**
	 * Chiude la connessione.
	 * 
	 * @param con
	 *            la connessione che deve essere chiusa
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected abstract void closeConnection(Connection con) throws SQLException;

	
	/**
	 * Restituisce il tipo di <code>DBMS</code> utilizzato da questo
	 * <code>ConnectionManager</code>. Questo metodo  utile nelle applicazioni
	 * che accedono a pi tipologie di <code>DBMS</code>.
	 * 
	 * @return il <code>DBMS</code> utilizzato da questo
	 *         <code>ConnectionManager</code>
	 */
	public final DBMS getDBMS() {
		return dbms;
	}

	/**
	 * Imposta il <code>ConnectionManager</code> utilizzato per default dai
	 * metodi che implicano la connessione al database. Specificando un
	 * <code>ConnectionManager</code> di default  possibile evitare di fornirlo
	 * ai metodi che accedono al database.
	 * 
	 * @param cm
	 *            il {@link ConnectionManager} che deve essere utilizzato per
	 *            default
	 */
	public static final void setDefault(ConnectionManager cm) {
		if (cm == null)
			throw new IllegalArgumentException("L'argomento ConnectionManager non pu essere null");
		defaultConnectionManager = cm;
	}

	/**
	 * Restituisce il <code>ConnectionManager</code> di default. <br>
	 * Se non  stato impostato il <code>ConnectionManager</code> di default,
	 * lancia una <code>RuntimeException</code>.
	 * 
	 * @return il <code>ConnectionManager</code> di default
	 */
	static final ConnectionManager getDefaultConnectionManager() {
		if (defaultConnectionManager == null)
			throw new RuntimeException("ConnectionManager di default non impostato");
		return defaultConnectionManager;
	}

	/*
	 * @deprecated
	 * Restituisce un <code>ConnectionManager</code> che gestisce le connessioni
	 * al database utilizzando il driver, la url e le credenziali di accesso
	 * specificate.
	 * 
	 * @param dbms
	 *            una delle implementazioni di <code>DBMS</code> fornite da
	 *            {@link DBMS}
	 * @param driverName
	 *            il nome completo della classe driver; esempio:
	 *            <code>"com.mysql.jdbc.Driver"</code>
	 * @param url
	 *            la url del database, nella forma
	 *            <code>jdbc:<i>subprotocol</i>:<code>jdbc:<i>subname</i></code>
	 *            ; esempio:
	 * 
	 *            <pre>
	 * &quot;jdbc:mysql://servername/dbname&quot;
	 * </pre>
	 * @param user
	 *            l'utente del database
	 * @param password
	 *            la password dell'utente del database
	 * @throws ClassNotFoundException
	 *             se non trova la classe driver
	 *
	public static final ConnectionManager get(DBMS dbms, String driverName, String url, String user, String password) throws ClassNotFoundException {
		return new StandardConnectionManager(dbms, driverName, url, user, password);
	}
	*/

}
