package it.softecspa.database.dbconnect;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Implementa le funzioni dipendenti dal DBMS utilizzato.
 * <p>
 * Queste funzioni sono richieste dalla classe di utilit {@link Query}.
 * 
 * @author Gino Quaglietta
 */

public abstract class DBMS {
	/**
	 * Implementazione di <code>DBMS</code> per MySQL.
	 */
	public static final DBMS MYSQL = new DBMS_MySql();

	/**
	 * Implementazione di <code>DBMS</code> per Oracle.
	 */
	public static final DBMS ORACLE = new DBMS_Oracle();

	/**
	 * Implementazione di <code>DBMS</code> per SQLServer.
	 */
	public static final DBMS SQLSERVER = new DBMS_SqlServer();

	/**
	 * Restituisce un DBMS del tipo specificato.
	 */
	public static final DBMS getDBMS(String dbms_type) {
		DBMS dbms = null;
		if ("MYSQL".equals(dbms_type))
			dbms = MYSQL;
		else if ("ORACLE".equals(dbms_type))
			dbms = ORACLE;
		else if ("SQLSERVER".equals(dbms_type))
			dbms = MYSQL;
		return dbms;
	}

	/**
	 * Specifica il formato YYYY-MM-DD per la conversione delle date in SQL.
	 */
	protected static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

	/**
	 * Specifica il formato YYYY-MM-DD HH:mm:ss per la conversione di datetime
	 * in SQL.
	 */
	protected static final SimpleDateFormat DATETIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

	private static DBMS defaultDBMS = null;

	DBMS() {
	}

	/**
	 * Restituisce la conversione in SQL della data usando la sintassi di questo
	 * dbms.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	protected abstract String toSQLDate(Date d);

	/**
	 * Restituisce la conversione in SQL di data e ora usando la sintassi di
	 * questo dbms.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	protected abstract String toSQLDateTime(Calendar d);

	/**
	 * Restituisce la conversione in SQL della data usando la sintassi di questo
	 * dbms.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	protected abstract String toSQLDate(Calendar d);

	/**
	 * Restituisce la conversione in SQL di data e ora usando la sintassi di
	 * questo dbms.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	protected abstract String toSQLDateTime(Date d);

	/**
	 * Restituisce la funzione del dbms che fornisce la data corrente.
	 * 
	 * @return la funzione del dbms che fornisce la data corrente
	 */
	protected abstract String funCurrentDate();

	/**
	 * Restituisce la funzione del dbms che fornisce la data e l'ora corrente.
	 * 
	 * @return la funzione del dbms che fornisce la data e l'ora corrente
	 */
	protected abstract String funCurrentDateTime();

	static final void setDefaultDBMS(DBMS dbms) {
		if (dbms == null)
			throw new IllegalArgumentException("L'argomento DBMS non pu essere null");
		defaultDBMS = dbms;
	}

	static final DBMS getDefaultDBMS() {
		if (defaultDBMS == null)
			throw new RuntimeException("DBMS di default non impostato");
		return defaultDBMS;
	}

	
	public enum DatabaseError {
		  NO_SUCH_TABLE
		, DUPLICATE_ENTRY
		
		, UNHANDLED
		;		
	}
	
	
	protected abstract DatabaseError decodeError(int errorCode);
	
	
	public boolean checkError(int errorCode, DatabaseError check) {
		return check.equals(decodeError(errorCode));
	}
}
