package it.softecspa.database.dbconnect;

import it.softecspa.database.dbconnect.DBMS.DatabaseError;
import it.softecspa.kahuna.sql.QueryFormatter;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * L'oggetto <code>DatabaseStatement</code>  responsabile dell'esecuzione delle
 * query e dell'estrazione dei resultset. <br>
 * Il ciclo di vita di un oggetto <code>DatabaseStatement</code> e di tutte le
 * risorse (connessione, statement, resultset) da esso allocate o fornite 
 * gestito dal framework.
 * 
 * @author Gino Quaglietta
 * @coauthor m.veroni
 */

public class DatabaseStatement {

	static final String SQL_LOG_CATEGORY = "it.softecspa.database.dbconnect.SQL";
	static Logger logSQL = Logger.getLogger(SQL_LOG_CATEGORY);
	
	Logger log = Logger.getLogger(DatabaseStatement.class);
	static boolean autoTO_LOWER = false;
	
	// Nasconde la QUERY al log anche se impostato
	private boolean hideQueryTraceLog = false;
	
	// Nasconde la QUERY andate in errore al log anche se impostato
	private boolean hideQueryErrorLog = false;
	private String notPrintedQuery = null;
	
	
	private ConnectionManager connectionManager;
	private Connection connection;
	private Statement stForExecute;
	private List<ResultSet> openResultset;
	
	
	private boolean transactional;
	

	DatabaseStatement(ConnectionManager cm, boolean forUpdate) throws SQLException {
		try {
			openResultset = new ArrayList<ResultSet>();
			connectionManager = cm;
		
			// Richiesta della connessione
			connection = connectionManager.getConnection();
			
			// Imposto il valore di AUTOCOMMIT			
			connection.setAutoCommit(!forUpdate);
			transactional = forUpdate;
			if (log.isDebugEnabled()) log.debug("AUTOCOMMIT = "+!forUpdate + (transactional?" (transactional)":""));				
			
		} catch (SQLException e) {
			close();
			throw e;
		}
	}

	
	
	
	protected boolean stampaSQL(String query, Throwable e) {
		if (e!=null) {
			if (hideQueryErrorLog) {
				notPrintedQuery = query;
				return false;
			}			
			notPrintedQuery = null;
			
			QueryFormatter.logError(logSQL, query, e);
			return true;
		}
		
		notPrintedQuery = null;
		if (logSQL.isTraceEnabled()) {
			if (hideQueryTraceLog) {
				logSQL.trace("Statemet SQL: is hide for this operation");
				return false;
			}
			
			QueryFormatter.logTrace(logSQL, query);
			return true;
		}
		
		return false;
	}
	
	
	/**
	 * Esegue una query SQL che restituisce un oggetto <code>ResultSet</code>.
	 * 
	 * @param sql
	 *            la query da eseguire
	 * @return l'oggetto <code>ResultSet</code> contenente le righe selezionate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database oppure se la
	 *             query non restituisce un oggetto <code>ResultSet</code>
	 */
	public final ResultSet getResultSet(String sql) throws SQLException {
		return getResultSet(sql, 0);
	}
	
	
	public final ResultSet getResultSet(String sql, int fetchSizeRows) throws SQLException {
		if (autoTO_LOWER) sql = sql.toLowerCase();
		boolean printed = stampaSQL(sql, null);		
		
		try {
			Statement st = connection.createStatement();
			st.setFetchSize(fetchSizeRows);
			ResultSet rs = st.executeQuery(sql);
			openResultset.add(rs);
			return rs;			
		} catch (SQLException e) {
			if (!printed) stampaSQL(sql, e);		
			throw e;
		}
	}
	
	

	/**
	 * Esegue la query specificata. La query pu essere di tipo
	 * <code>INSERT</code>, <code>UPDATE</code> o <code>DELETE</code>; oppure
	 * pu essere un comando SQL che non ha valore di ritorno, ad esempio un
	 * comando DDL.
	 * 
	 * @param sql
	 *            la query da eseguire
	 * @return il numero delle righe inserite, aggiornate o eliminate nel caso
	 *         di una query di tipo <code>INSERT</code>, <code>UPDATE</code> o
	 *         <code>DELETE</code>, oppure <code>0</code> nel caso di un comando
	 *         SQL che non ha valore di ritorno.
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database oppure se la
	 *             query restituisce un oggetto <code>ResultSet</code>
	 */
	public final int execute(String sql) throws SQLException {

		if ((sql == null) || sql.trim().equals(""))	return 0;

		if (autoTO_LOWER) sql = sql.toLowerCase();
		boolean printed = stampaSQL(sql, null);		
		try {
		
			if (stForExecute == null) stForExecute = connection.createStatement();
			return stForExecute.executeUpdate(sql);
						
		} catch (SQLException e) {
			if (!printed) stampaSQL(sql, e);		
			throw e;
		}
	}

	/**
	 * Esegue la chiusura della transazione con un COMMIT o un ROLLBACK
	 * @param commit
	 * @throws SQLException
	 */
	private final void closeTransaction(boolean commit) throws SQLException {
		if (commit) {
			// COMMIT
			if (connection!=null && transactional) {
				if (log.isDebugEnabled()) log.debug("Close transaction: COMMIT");
				connection.commit();
			}
		} else {
			// ROLLBACK
			if (connection!=null && transactional) {
				if (log.isDebugEnabled()) log.debug("Close transaction: ROLLBACK");
				connection.rollback();
			}
		}
	}
	
	final void commit() throws SQLException {
		closeTransaction(true);
	}

	final void rollback() throws SQLException {
		closeTransaction(false);
	}
	
	
	
	
	final void close() {
		if (connection == null) {
			if (log.isDebugEnabled()) log.debug("Close without connection open, nothing to do!");
			return;
		}

		
		if (openResultset != null) {
			for (int i = 0; i < openResultset.size(); i++) {
				try {
					ResultSet rs = (ResultSet) openResultset.get(i);
					rs.close();
					Statement s = rs.getStatement();
					if (s != null) s.close();
				} catch (SQLException e) {
					log.warn("["+ e.getErrorCode() + "] " + e.getMessage());
				}
				
			}
			openResultset = null;
		}

		if (stForExecute != null) {
			try {
				stForExecute.close();
			} catch (SQLException e) {
				log.warn("["+ e.getErrorCode() + "] " + e.getMessage());
			}
		
		}
		stForExecute = null;

		try {
			connectionManager.closeConnection(connection);			
		} catch (SQLException e) {
			log.error("Error closing DatabaseStatement: [" + e.getErrorCode() + "] " + e.toString(), e);
		}
		connection = null;
	}

	/**
	 * Restituisce il numero intero selezionato dalla query specificata.
	 * 
	 * @param sql
	 *            la query da eseguire
	 * @return il numero intero selezionato, oppure <code>0</code> se il
	 *         resultset  vuoto
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database oppure se la
	 *             query non produce un resultset contenente un numero intero
	 */
	public final int getInt(String sql) throws SQLException {
		ResultSet rs = getResultSet(sql);
		if (rs.next()) {
			return rs.getInt(1);
		} else {
			return 0;
		}
	}

	/**
	 * Genera un nuovo valore per una chiave di tipo intero riutilizzando i
	 * codici precedentemente eliminati.
	 * <p>
	 * Esempio: si desidera inserire una nuova occorrenza nella tabella
	 * Studente, che ha chiave (codUniversita, matricola); la chiamata:
	 * 
	 * <pre>
	 * getNewCod(&quot;Studente&quot;, &quot;matricola&quot;, &quot;codUniversita=&quot; + codUniversita)
	 * </pre>
	 * 
	 * restituisce una nuova matricola per l'Universit specificata.
	 * 
	 * @param table
	 *            il nome della tabella nella quale si desidera inserire una
	 *            nuova riga
	 * @param keyField
	 *            il nome del campo chiave per il quale si desidera generare un
	 *            nuovo valore
	 * @param filterWhere
	 *            condizione della clausola <code>WHERE</code> che filtra
	 *            l'identificatore esterno
	 * @return il nuovo valore della chiave
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int getNewCod(String table, String keyField, String filterWhere) throws SQLException {
		String sql = "SELECT " + keyField + " FROM " + table;
		if ((filterWhere != null) && (!filterWhere.trim().equals("")))
			sql += " WHERE " + filterWhere;
		sql += " ORDER BY " + keyField;

		boolean printed = stampaSQL(sql, null);		
		try {
		
			ResultSet rs = getResultSet(sql);
			int cod = 1;
	
			while (rs.next()) {
				int codread = rs.getInt(keyField);
				if (codread == 0)
					cod = 0;
				if (codread == cod)
					cod += 1;
				else
					break;
			}	
			return cod;
				
		} catch (SQLException e) {
			if (!printed) stampaSQL(sql, e);		
			throw e;
		}	
			
	}

	/**
	 * Genera un nuovo valore per una chiave di tipo intero riutilizzando i
	 * codici precedentemente eliminati.
	 * <p>
	 * Esempio: si desidera inserire una nuova occorrenza nella tabella Utente,
	 * che ha chiave codUte; la chiamata:
	 * 
	 * <pre>
	 * getNewCod(&quot;Utente&quot;, &quot;codUte&quot;)
	 * </pre>
	 * 
	 * restituisce una nuovo codice utente.
	 * 
	 * @param table
	 *            il nome della tabella nella quale si desidera inserire una
	 *            nuova riga
	 * @param keyField
	 *            il nome del campo chiave per il quale si desidera generare un
	 *            nuovo valore
	 * @return il nuovo valore della chiave
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int getNewCod(String table, String keyField) throws SQLException {
		return getNewCod(table, keyField, null);
	}

	/**
	 * Restituisce il tipo di <code>DBMS</code> utilizzato da questo
	 * <code>DatabaseStatement</code>. Questo metodo  utile nelle applicazioni
	 * che accedono a pi tipologie di <code>DBMS</code>.
	 * 
	 * @return il <code>DBMS</code> utilizzato da questo
	 *         <code>DatabaseStatement</code>
	 */
	public final DBMS getDBMS() {
		return connectionManager.getDBMS();
	}

	
	
	/**
     * Retrieves a <code>DatabaseMetaData</code> object that contains
     * metadata about the database to which this
     * <code>Connection</code> object represents a connection.
     * The metadata includes information about the database's
     * tables, its supported SQL grammar, its stored
     * procedures, the capabilities of this connection, and so on.
     *
     * @return a <code>DatabaseMetaData</code> object for this 
     *         <code>Connection</code> object
     * @exception SQLException if a database access error occurs
     */
	public DatabaseMetaData getMetaData() throws SQLException {
		return connection.getMetaData();
	}
	
	
	/**
     * Retrieves this <code>Connection</code> object's current catalog name.
     *
     * @return the current catalog name or <code>null</code> if there is none
     * @exception SQLException if a database access error occurs
     * @see #setCatalog
     */
	public String getCatalog() throws SQLException {
		return connection.getCatalog();
	}


	
	protected Connection getConnection() {
		return connection;
	}


	protected ConnectionManager getConnectionManager() {
		return connectionManager;
	}


	
	
	public boolean isHideQueryLog() {
		return hideQueryTraceLog;
	}
	
	public void hideQueryLog() {
		this.hideQueryTraceLog = true;
	}
	
	public void unhideQueryLog() {
		this.hideQueryTraceLog = false;
	}
	
	
	
	
	
	
	/**
	 * Evita di mettere nel log la query in errore in modo automatico
	 * Utilizzato per le sequenze INSERT/UPDATE
	 */
	void doNotTraceQueryInErrorLog() {
		this.hideQueryErrorLog = true;
	}
	
	/**
	 * Ripristina la situazione normale
	 * Utilizzato per le sequenze INSERT/UPDATE
	 */
	void resetTraceQueryInErrorLog() {
		this.hideQueryErrorLog = false;
	}
	

	
	/**
	 * Chiamata per stampare le query in errore, non messe nel log in modo automatico
	 * Utilizzato per le sequenze INSERT/UPDATE
	 * @param e
	 */
	void stampaNotTracedSQL(SQLException e) {
		this.hideQueryErrorLog = false;
		if (notPrintedQuery!=null) {
			stampaSQL(notPrintedQuery, e);
		}
	}


	public static Logger getLoggerSQL() {
		return logSQL;
	}

	
	/**
	 * Verifca se l'eccezione SQL equivale all'errore desiderato
	 * @param errorCode
	 * @param check
	 * @return
	 */
	public boolean checkError(SQLException e, DatabaseError check) {
		return getDBMS().checkError(e.getErrorCode(), check);
	}
	
}
