package it.softecspa.database.dbconnect;

import java.text.ParsePosition;
import java.util.Calendar;
import java.util.Date;

/**
 * Classe di utilit per la costruzione delle query.
 * 
 * Alcuni metodi implicano la conoscenza del tipo di dbms e pertanto richiedono
 * il parametro <code>DBMS</code>; per ciascuno di questi metodi  fornita la
 * versione che non richiede tale parametro ed utilizza invece il
 * <code>DBMS</code> di default.
 * 
 * @author Gino Quaglietta
 */

public class Query {
	private boolean doInsert;
	private String table = null;
	private String fields = null;
	private String values = null;
	private String where = null;

	private static class DBFunction {
		private String function;

		private DBFunction(String function) {
			this.function = function;
		}

	}

	public static final String toSQL(DBFunction dbfunction) {
		return dbfunction != null ? dbfunction.function : null;
	}

	/**
	 * Costruisce una query che inserisce o aggiorna una riga della tabella
	 * specificata.
	 * 
	 * @param table
	 *            il nome della tabella
	 * @param doInsert
	 *            specifica se effettuare un inserimento (<b><code>true</code>
	 *            </b>) o un'aggiornamento (<b><code>false</code></b>) o
	 *            un'aggiornamento della riga
	 */
	public Query(String table, boolean doInsert) {
		this.doInsert = doInsert;
		this.table = table;
		this.fields = "";
		this.values = "";
		this.where = "";
	}

	/**
	 * Restituisce la funzione del dbms specificato che fornisce la data
	 * corrente.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}
	 * @return la funzione del dbms che fornisce la data corrente
	 */
	public static final DBFunction getFunCurrentDate(DBMS dbms) {
		return new DBFunction(dbms.funCurrentDate());
	}

	/**
	 * Restituisce la funzione del dbms di default che fornisce la data
	 * corrente. Scrive un messaggio di errore nel log se non  stato impostato
	 * un dbms di default.
	 * 
	 * @return la funzione del dbms che fornisce la data corrente
	 */
	public static final DBFunction getFunCurrentDate() {
		return getFunCurrentDate(DBMS.getDefaultDBMS());
	}

	/**
	 * Restituisce la funzione del dbms specificato che fornisce la data e l'ora
	 * corrente.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}
	 * @return la funzione del dbms che fornisce la data e l'ora corrente
	 */
	public static final DBFunction getFunCurrentDateTime(DBMS dbms) {
		return new DBFunction(dbms.funCurrentDateTime());
	}

	/**
	 * Restituisce la funzione del dbms di default che fornisce la data e l'ora
	 * corrente. Scrive un messaggio di errore nel log se non  stato impostato
	 * un dbms di default.
	 * 
	 * @return la funzione del dbms che fornisce la data e l'ora corrente
	 */
	public static final DBFunction getFunCurrentDateTime() {
		return getFunCurrentDateTime(DBMS.getDefaultDBMS());
	}

	/**
	 * Restituisce la conversione in SQL della data usando la sintassi del dbms
	 * specificato.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDate(DBMS dbms, Date d) {
		return dbms.toSQLDate(d);
	}

	/**
	 * Restituisce la conversione in SQL di data e ora usando la sintassi del
	 * dbms specificato.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDateTime(DBMS dbms, Date d) {
		return dbms.toSQLDateTime(d);
	}

	/**
	 * Restituisce la conversione in SQL della data usando la sintassi del dbms
	 * specificato.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDate(DBMS dbms, Calendar d) {
		return dbms.toSQLDate(d);
	}

	/**
	 * Restituisce la conversione in SQL di data e ora usando la sintassi del
	 * dbms specificato.
	 * 
	 * @param dbms
	 *            una implementazione di <code>DBMS</code> fornita dalla classe
	 *            {@link DBMS}
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDateTime(DBMS dbms, Calendar d) {
		return dbms.toSQLDateTime(d);
	}

	/**
	 * Restituisce la conversione in SQL della data usando la sintassi del dbms
	 * di default. Scrive un messaggio di errore nel log se non  stato
	 * impostato un dbms di default.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDate(Date d) {
		return toSQLDate(DBMS.getDefaultDBMS(), d);
	}

	/**
	 * Restituisce la conversione in SQL di data e ora usando la sintassi del
	 * dbms di default. Scrive un messaggio di errore nel log se non  stato
	 * impostato un dbms di default.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDateTime(Date d) {
		return toSQLDateTime(DBMS.getDefaultDBMS(), d);
	}

	/**
	 * Restituisce la conversione in SQL della data usando la sintassi del dbms
	 * di default. Scrive un messaggio di errore nel log se non  stato
	 * impostato un dbms di default.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDate(Calendar d) {
		return toSQLDate(DBMS.getDefaultDBMS(), d);
	}

	/**
	 * Restituisce la conversione in SQL di data e ora usando la sintassi del
	 * dbms di default. Scrive un messaggio di errore nel log se non  stato
	 * impostato un dbms di default.
	 * 
	 * @param d
	 *            la data da convertire in SQL
	 * @return la conversione della data in SQL
	 */
	public static final String toSQLDateTime(Calendar d) {
		return toSQLDateTime(DBMS.getDefaultDBMS(), d);
	}

	private void addFieldValue(String field, String value, boolean isKey) {
		if (doInsert || (!isKey)) {
			if (!fields.equals(""))
				fields += ", ";
			fields += field;
		}

		if (doInsert) {
			if (!values.equals(""))
				values += ", ";
			values += value;
		} else if (isKey) {
			if (!where.equals(""))
				where += " AND ";
			where += field + "=" + value;
		} else {
			fields += "=" + value;
		}
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo della tabella.
	 * 
	 * @param field
	 *            il campo della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addFieldValue(String field, int value) {
		addFieldValue(field, value + "", false);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo della tabella. Assume la rappresentazione del boolean
	 * mediante un campo char(1) con valori 'Y' o 'N'.
	 * 
	 * @param field
	 *            il campo della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addFieldValue(String field, boolean value) {
		addFieldValue(field, value ? "Y" : "N");
	}

	/**
	 * Aggiunge a questa query il comando di impostare il campo della tabella a
	 * <code>null</code>.
	 * 
	 * @param field
	 *            il campo della tabella
	 */
	public final void addFieldNull(String field) {
		addFieldValue(field, "NULL", false);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo della tabella.
	 * 
	 * @param field
	 *            il campo della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addFieldValue(String field, double value) {
		addFieldValue(field, value + "", false);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo della tabella.
	 * 
	 * @param field
	 *            il campo della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addFieldValue(String field, String value) {
		addFieldValue(field, toSQL(value), false);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo della tabella.
	 * 
	 * @param field
	 *            il campo della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addFieldValue(String field, Date value) {
		addFieldValue(field, toSQLDateTime(value), false);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo della tabella.
	 * 
	 * @param field
	 *            il campo della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addFieldValue(String field, DBFunction value) {
		if (value == null)
			addFieldNull(field);
		else
			addFieldValue(field, value.function, false);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo chiave della tabella.
	 * 
	 * @param field
	 *            il campo chiave della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addKeyFieldValue(String field, int value) {
		addFieldValue(field, value + "", true);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo chiave della tabella.
	 * 
	 * @param field
	 *            il campo chiave della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addKeyFieldValue(String field, String value) {
		addFieldValue(field, toSQL(value), true);
	}

	/**
	 * Aggiunge a questa query il comando di assegnare un valore del tipo
	 * indicato al campo chiave della tabella.
	 * 
	 * @param field
	 *            il campo chiave della tabella
	 * @param value
	 *            il valore del campo
	 */
	public final void addKeyFieldValue(String field, Date value) {
		addFieldValue(field, toSQLDateTime(value), true);
	}

	/**
	 * Restituisce la query di aggiornamento.
	 * 
	 * @return la query di aggiornamento
	 */
	public final String getQuery() {
		if (doInsert)
			return "INSERT INTO " + table + "(" + fields + ") VALUES (" + values + ")";
		else
			return "UPDATE " + table + " SET " + fields + " WHERE " + where;
	}

	/**
	 * Restituisce la conversione in SQL del valore specificato.
	 * 
	 * @param value
	 *            il valore
	 * @return la conversione in SQL
	 */
	public static final String toSQL(String value) {
		if (value == null)	return "NULL";
		String n = "";
		for (int i = 0; i < value.length(); i++) {
			String c = value.substring(i, i + 1);
			n+=(c.equals("'")?"''":c);
		}
		return "'" + n.trim() + "'";
	}

	/**
	 * Restituisce la conversione in SQL del valore specificato. Assume la
	 * rappresentazione del boolean mediante un campo char(1) con valori 'Y' o
	 * 'N'.
	 * 
	 * @param value
	 *            il valore
	 * @return la conversione in SQL
	 */
	public static final String toSQL(boolean value) {
		return value ? "'Y'" : "'N'";
	}

	/**
	 * Restituisce la <code>java.sql.Date</code> corrispondente al giorno, mese
	 * e anno specificati.
	 * 
	 * @param year
	 *            l'anno
	 * @param month
	 *            il mese
	 * @param day
	 *            il giorno
	 * @return la data corrispondente
	 */
	public static final Date getDate(int year, int month, int day) {
		String strdata = year + "-" + month + "-" + day;
		return new Date(DBMS.DATE_FORMAT.parse(strdata, new ParsePosition(0)).getTime());
	}

}
