package it.softecspa.database.dbconnect;

import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Legge le righe di un <code>ResultSet</code>.
 * <p>
 * Il framework suggerisce di rappresentare ogni tabella mediante una classe che
 * estende <code>Row</code>; poich ciascuna istanza della classe rappresenta
 * una riga della tabella, la lettura di pi righe della tabella comporta la
 * creazione di un oggetto <code>Row</code> per ciascuna riga letta. <br>
 * In generale questa procedura  appropriata, ma vi sono casi nei quali 
 * preferibile elaborare direttamente ogni riga senza dover creare un oggetto
 * per ciascuna riga letta (esempio: export di dati su file, che pu comportare
 * l'estrazione di un gran numero di righe). <br>
 * In casi come questi  opportuno estendere da <code>ResultSetReader</code>
 * anzich da <code>Row</code>. Si noti che lo stesso oggetto <code>Row</code>
 * utilizza un <code>ResultSetReader</code> per restituire l'array
 * <code>Row[]</code> selezionato mediante una query.
 */
public abstract class ResultSetReader {

	/**
	 * Costruisce un nuovo oggetto <code>ResultSetReader</code>.
	 */
	public ResultSetReader() {
	}

	/**
	 * Metodo che occorre implementare per leggere le righe del
	 * <code>ResultSet</code>. Esempio:
	 * 
	 * <pre>
	 * protected void read(ResultSet rs) throws SQLException {
	 * 	while (rs.next()) {
	 * 		//... codice che legge i campi della singola riga
	 * 	}
	 * }
	 * </pre>
	 * 
	 * @param rs
	 *            l'oggetto <code>ResultSet</code> contenente le righe richieste
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected abstract void read(ResultSet rs) throws SQLException;

	/**
	 * Legge le righe selezionate dalla query sql utilizzando l'implementazione
	 * del metodo {@link #read(ResultSet)}.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @param sql
	 *            la query
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final void read(ConnectionManager cm, String sql) throws SQLException {
		DatabaseStatement dbs = new DatabaseStatement(cm, false);
		try {
			ResultSet rs = dbs.getResultSet(sql);
			read(rs);

		} finally {
			dbs.close();
		}
	}

	protected final void read(DatabaseStatement dbs, String sql) throws SQLException {

		ResultSet rs = dbs.getResultSet(sql);
		read(rs);

	}

	/**
	 * Legge le righe selezionate dalla query sql utilizzando l'implementazione
	 * del metodo {@link #read(ResultSet)}. Usa il
	 * <code>ConnectionManager</code> di default.
	 * 
	 * @param sql
	 *            la query
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final void read(String sql) throws SQLException {
		read(ConnectionManager.getDefaultConnectionManager(), sql);
	}
}
