package it.softecspa.database.dbconnect;

import it.softecspa.kahuna.sql.QueryFormatter;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * Rappresenta una tabella o il join di pi tabelle; ogni istanza di
 * <code>Row</code> rappresenta una riga della tabella o del join di tabelle.<br>
 * Ogni tabella (o join di tabelle) del database  mappata in una classe
 * <b>C</b> che estende <code>Row</code> e che ha come attributi d'istanza i
 * campi della tabella (o del join di tabelle) <b>T</b> a cui l'oggetto si
 * riferisce; una istanza della classe <b>C</b> rappresenta una riga della
 * tabella o join <b>T</b>, da cui il nome assegnato alla super classe
 * <code>Row</code>.<br>
 * Si estende direttamente da <code>Row</code> solo quando la tabella che si
 * vuole rappresentare  in sola lettura (ossia l'applicazione non ne prevede
 * l'aggiornamento); una tabella aggiornabile si rappresenta estendendo
 * {@link UpdatableRow}, che a sua volta estende <code>Row</code> aggiungendo le
 * funzionalit di aggiornamento.
 * 
 * <p>
 * L'oggetto <code>Row</code> deve implementare il metodo
 * {@link #readRow(ResultSet)}, che valorizza gli attributi dell'oggetto
 * leggendo i campi della tabella dal <code>Resultset</code> passato come
 * argomento; questo metodo  utilizzato da:
 * <ul>
 * <li> {@link #readRow(ConnectionManager, String)}, che legge una riga della
 * tabella usando il <code>ConnectionManager</code> specificato
 * <li> {@link #readRow(String)}, che legge una riga della tabella usando il
 * <code>ConnectionManager</code> di default
 * <li> {@link #getRows(ConnectionManager, String)}, che restituisce un insieme
 * di righe della tabella usando il <code>ConnectionManager</code> specificato
 * <li> {@link #getRows(String)}, che restituisce un insieme di righe della
 * tabella usando il <code>ConnectionManager</code> di default
 * </ul>
 * 
 * @author Gino Quaglietta
 */

public abstract class Row {

	Logger log = Logger.getLogger(Row.class);
	static Logger logSQL = DatabaseStatement.getLoggerSQL();
	
	
	private static class Reader extends ResultSetReader {
		private Row row;
		private List<Row> rows;

		Reader(Row row) {
			this.row = row;
		}

		private RuntimeException getRuntimeException(Throwable e) {
			return new RuntimeException("Impossibile creare nuove istanze della classe "
										+ row.getClass().getName()
										+ ": la classe e il suo costruttore senza argomenti devono essere public.",e);
		}

		protected final void read(ResultSet res) throws SQLException {
			try {
				while (res.next()) {
					Row r = (Row) row.getClass().newInstance();
					r.readRow(res);
					rows.add(r);
				}

			} catch (IllegalAccessException e) {
				throw getRuntimeException(e);

			} catch (InstantiationException e) {
				throw getRuntimeException(e);
			}
		}

		final Row[] getRows(ConnectionManager cm, String sql) throws SQLException {
			rows = new ArrayList<Row>();
			read(cm, sql);
			Row[] righe = (Row[]) java.lang.reflect.Array.newInstance(row.getClass(), rows.size());
			for (int i = 0; i < rows.size(); i++) {
				righe[i] = (Row) rows.get(i);
			}
			return righe;
		}
		
		final Row[] getRows(DatabaseStatement dbs, String sql) throws SQLException {
			rows = new ArrayList<Row>();
			read(dbs, sql);
			Row[] righe = (Row[]) java.lang.reflect.Array.newInstance(row.getClass(), rows.size());
			for (int i = 0; i < rows.size(); i++) {
				righe[i] = (Row) rows.get(i);
			}
			return righe;
		}
	}

	/**
	 * Metodo che occorre implementare per valorizzare gli attributi
	 * dell'oggetto leggendo una riga dal <code>ResultSet</code>. Esempio:
	 * 
	 * <pre>
	 * protected void readRow(ResultSet res) throws SQLException {
	 * 	myFieldInt = res.getInt(&quot;FIELD_INT&quot;);
	 * 	//...
	 * }
	 * </pre>
	 * 
	 * @param res
	 *            l'oggetto <code>ResultSet</code> contenente campi
	 *            corrispondenti agli attributi di questo oggetto.
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected abstract void readRow(ResultSet res) throws SQLException;

	/**
	 * Costruisce un nuovo oggetto <code>Row</code>.
	 */
	public Row() {
	}

	/**
	 * Legge la riga selezionata dalla query sql e valorizza gli attributi di
	 * questo oggetto utilizzando l'implementazione del metodo
	 * {@link #readRow(ResultSet)}.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @param sql
	 *            la query
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final void readRow(ConnectionManager cm, String sql) throws SQLException {
		DatabaseStatement dbs = null;
		ResultSet res = null;
		boolean doublelog = false;		
		
		try {
			dbs = new DatabaseStatement(cm, false);
			res = dbs.getResultSet(sql);
			if (res.next()) {
				readRow(res);
			} else {
				if (log.isDebugEnabled()) {
					log.debug("Nessuna riga estratta!");
					doublelog=true;
				}
				throw new NoRecordFoundException();
			}
			
		} catch (SQLException e) {
			if (log.isDebugEnabled()) {
				if (!doublelog) log.warn(e);
			}
			throw e;
			
			
		} finally {
			try {
				if (res!=null) res.close();
			} catch (SQLException e) {
				log.warn(e);
			}
			if (dbs!=null) dbs.close();			
		}
	}

	/**
	 * Legge la riga selezionata dalla query sql e valorizza gli attributi di
	 * questo oggetto utilizzando l'implementazione del metodo
	 * {@link #readRow(ResultSet)}. Usa il <code>ConnectionManager</code> di
	 * default.
	 * 
	 * @param sql
	 *            la query
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final void readRow(String sql) throws SQLException {
		readRow(ConnectionManager.getDefaultConnectionManager(), sql);
	}

	/**
	 * Restituisce l'array <code>Row[]</code> selezionato dalla query sql. <br>
	 * L'array <code>Row[]</code> restituito  tale che  possibile effettuarne
	 * il downcasting all'array di oggetti aventi lo stesso tipo di questo
	 * oggetto. In altre parole, se <code>MyTable</code> estende
	 * <code>Row</code> il seguente downcasting:
	 * 
	 * <pre>
	 * (MyTable[]) (new MyTable()).getRows(cm, sql)
	 * </pre>
	 * 
	 *  sempre possibile e fornisce l'array di oggetti <code>MyTable</code>
	 * selezionati dalla query sql. <br>
	 * La valorizzazione degli attributi di questo oggetto usa l'implementazione
	 * del metodo {@link #readRow(ResultSet)}. <br>
	 * <b>Attenzione</b>: se la classe e il suo costruttore senza argomenti non
	 * sono public, questo metodo stampa un messaggio di errore su System.err e
	 * restituisce un array di lunghezza zero.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @param sql
	 *            la query
	 * @return l'array <code>Row[]</code> selezionato dalla query
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final Row[] getRows(ConnectionManager cm, String sql)	throws SQLException {
		Reader r = new Reader(this);
		return r.getRows(cm, sql);
	}

	
	protected final Row[] getRows(DatabaseStatement dbs, String sql)	throws SQLException {
		Reader r = new Reader(this);
		return r.getRows(dbs, sql);
	}
	
	
	
	/**
	 * Restituisce l'array <code>Row[]</code> selezionato dalla query sql. Usa
	 * il <code>ConnectionManager</code> di default. <br>
	 * L'array <code>Row[]</code> restituito  tale che  possibile effettuarne
	 * il downcasting all'array di oggetti aventi lo stesso tipo di questo
	 * oggetto. In altre parole, se <code>MyTable</code> estende
	 * <code>Row</code> il seguente downcasting:
	 * 
	 * <pre>
	 * (MyTable[]) (new MyTable()).getRows(sql)
	 * </pre>
	 * 
	 *  sempre possibile e fornisce l'array di oggetti <code>MyTable</code>
	 * selezionati dalla query sql. <br>
	 * La valorizzazione degli attributi di questo oggetto usa l'implementazione
	 * del metodo {@link #readRow(ResultSet)}. <br>
	 * <b>Attenzione</b>: se la classe e il suo costruttore senza argomenti non
	 * sono public, questo metodo stampa un messaggio di errore su System.err e
	 * restituisce un array di lunghezza zero.
	 * 
	 * @param sql
	 *            la query
	 * @return l'array <code>Row[]</code> selezionato dalla query
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final Row[] getRows(String sql) throws SQLException {
		return getRows(ConnectionManager.getDefaultConnectionManager(), sql);
	}

	/**
	 * Restituisce l'intero selezionato dalla query.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @param sql
	 *            la query che seleziona un intero
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected static final int getInt(ConnectionManager cm, String sql)	throws SQLException {
		DatabaseStatement dbs = new DatabaseStatement(cm, false);
		try {
			if (logSQL.isTraceEnabled()) {
				QueryFormatter.logTrace(logSQL,sql);			
			}
			
			return dbs.getInt(sql);
		} finally {
			dbs.close();
		}
	}

	/**
	 * Restituisce l'intero selezionato dalla query. Usa il
	 * <code>ConnectionManager</code> di default.
	 * 
	 * @param sql
	 *            la query che seleziona un intero
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected static final int getInt(String sql) throws SQLException {
		return getInt(ConnectionManager.getDefaultConnectionManager(), sql);
	}

}
