package it.softecspa.database.dbconnect;

import it.softecspa.database.dbconnect.DBMS.DatabaseError;

import java.sql.SQLException;

/**
 * Estensione di <code>Row</code> con funzionalit di aggiornamento.
 * 
 * <p>
 * L'oggetto <code>UpdatableRow</code> deve implementare il metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)}, che esegue l'operazione di aggiornamento di tipo <code>UpdateType</code> usando il <code>DatabaseStatement</code>
 * passato come parametro; ogni operazione effettuata mediante il <code>DatabaseStatement</code> avviene all'interno della stessa transazione. <br>
 * L'oggetto <code>UpdateType</code> passato come parametro pu essere una delle tipologie di aggiornamento predefinite <code>INSERT</code>, <code>UPDATE</code> o <code>DELETE</code>, oppure una tipologia di aggiornamento prevista dallo sviluppatore.
 * <br>
 * Il metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)}  utilizzato da:
 * <ul>
 * <li> {@link #updateRecord(ConnectionManager, UpdatableRow.UpdateType)}, che esegue l'operazione di aggiornamento di tipo <code>UpdateType</code> usando il <code>ConnectionManager</code> specificato
 * <li> {@link #updateRecord(UpdatableRow.UpdateType)}, che esegue l'operazione di aggiornamento di tipo <code>UpdateType</code> usando il <code>ConnectionManager</code> di default
 * <li> {@link #insert(ConnectionManager)}, {@link #update(ConnectionManager)}, {@link #delete(ConnectionManager)} che eseguono le operazioni di <code>INSERT</code>, <code>UPDATE</code>, <code>DELETE</code> usando il <code>ConnectionManager</code>
 * specificato
 * <li> {@link #insert()}, {@link #update()}, {@link #delete()} che eseguono le operazioni di <code>INSERT</code>, <code>UPDATE</code>, <code>DELETE</code> usando il <code>ConnectionManager</code> di default
 * </ul>
 * 
 * @author Gino Quaglietta
 * @author m.veroni
 */

public abstract class UpdatableRow extends Row {
	
	protected static final UpdateType INSERT = new UpdateType();
	protected static final UpdateType UPDATE = new UpdateType();
	protected static final UpdateType DELETE = new UpdateType();
	
	// Valido su MYSQL 5.0 o superiore
	protected static final UpdateType INSERT_OR_DUPLICATE_KEY_UPDATE = new UpdateType();

	
	
	protected static class UpdateType {
		public UpdateType() {
		}
	}

	/**
	 * Metodo che occorre implementare per eseguire l'operazione di aggiornamento di tipo <code>UpdateType</code> usando il <code>DatabaseStatement</code> passato come parametro. Ogni operazione effettuata mediante il <code>DatabaseStatement</code>
	 * avviene all'interno della stessa transazione.
	 * 
	 * @param updateType
	 *            la tipologia di aggiornamento
	 * @param dbs
	 *            il <code>DatabaseStatement</code> da utilizzare per eseguire l'aggiornamento
	 * @return il numero di righe modificate o altre informazioni custom
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public abstract int updateRecord(UpdateType updateType, DatabaseStatement dbs) throws SQLException;

	
	/**
	 * Esegue l'operazione di aggiornamento di tipo <code>UpdateType</code>. <br>
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @param updateType
	 *            il tipo di aggiornamento che deve essere effettuato
	 * @return il numero di righe modificate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	protected final int updateRecord(ConnectionManager cm, UpdateType updateType) throws SQLException {
		
		DatabaseStatement dbs = null;
		try {
			dbs = new DatabaseStatement(cm, true);			
			
			if (log.isDebugEnabled()) log.debug("Start record activity");
			int ret = updateRecord(updateType, dbs);
			if (log.isDebugEnabled()) log.debug("End record activity (return = " + ret + ")");
			
			// COMMIT
			dbs.commit();
			return ret;

		} catch (SQLException e) {
			if (log.isDebugEnabled()) log.error("Error saving data", e);
			// ROLLBACK
			if (dbs!=null) dbs.rollback();
			
			// ...e rilancio l'eccezione
			throw e;

		} finally {
			if (dbs!=null) dbs.close();
		}
	}

	
	/* Tolto da versione 1.5.3 *
	 * @deprecated 
	 * Esegue l'operazione di aggiornamento di tipo <code>UpdateType</code> utilizzando il <code>ConnectionManager</code> di default.
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param updateType
	 *            il tipo di aggiornamento che deve essere effettuato
	 * @return il numero di righe modificate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 * /
	protected final int update(UpdateType updateType) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with custom action @deprecated");
		return updateRecord(ConnectionManager.getDefaultConnectionManager(), updateType);
	}
	*/

	/**
	 * Esegue l'operazione di <code>INSERT</code>. L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @return il numero di righe inserite o il progressivo della tabella
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int insert(ConnectionManager cm) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT");
		return updateRecord(cm, INSERT);
	}

	/**
	 * Esegue l'operazione di <code>INSERT</code>. L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param dbsu
	 *            il <code>DatabaseStatementUnplugged</code> che deve essere utilizzato
	 * @return il numero di righe inserite o il progressivo della tabella
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int insert(DatabaseStatementUnplugged dbsu) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT");
		return updateRecord(INSERT, dbsu.getDatabaseStatement());
	}

	/* Tolto da versione 1.5.3 *
	 * @deprecated 
	 * Esegue l'operazione di <code>INSERT</code>. Utilizza il <code>ConnectionManager</code> di default. <br>
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @return il numero di righe inserite o il progressivo della tabella
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 * /
	public final int insert() throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT @deprecated");
		return update(INSERT);
	}
	*/

	/**
	 * Esegue l'operazione di <code>INSERT</code> e in caso di record gi presente effetua una <code>UPDATE</code>. 
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * L'operazione &ecaute; atomica.
	 * 
	 * @return il numero di righe inserite o il progressivo della tabella
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int insertOrUpdate(ConnectionManager cm) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT or UPDATE");
		
		DatabaseStatement dbs = null;
		try {
			dbs = new DatabaseStatement(cm, true);
			
			if (log.isDebugEnabled()) log.debug("Start record activity");
			int ret = 0;
			try {
				// Nascondo la query se va in errore, gestisco manualmente l'opeazione
				dbs.doNotTraceQueryInErrorLog();
				
				// Eseguo la INSERT
				ret = updateRecord(INSERT, dbs);
				
			} catch (SQLException e_insert) {
				// ROLLBACK (altrimenti la replica di MySql si blocca)
				dbs.rollback();
				//
				if (!dbs.getDBMS().checkError(e_insert.getErrorCode(), DatabaseError.DUPLICATE_ENTRY)) {
					// Stampo la query nel log non stampata in precedenza
					dbs.stampaNotTracedSQL(e_insert);
					throw e_insert;
				}
				if (logSQL.isTraceEnabled()) {
					logSQL.debug("[WARN] Error #" + e_insert.getErrorCode() + ", catch exception in insert and do update");
				}

				try {
					// ...allora eseguo la UPDATE
					ret = updateRecord(UPDATE, dbs);
				} catch (SQLException e_update) {
					if (log.isDebugEnabled()) {
						log.error("Error saving data", e_insert);
					}
					
					// ROLLBACK
					dbs.rollback();
					throw e_update;
				}
			}
			if (log.isDebugEnabled()) log.debug("End record activity (return = " + ret + ")");
			
			// COMMIT
			dbs.commit();
			return ret;

		} finally {
			if (dbs!=null) {
				// Reset delle condizioni per stampare la query
				dbs.resetTraceQueryInErrorLog();
				//
				dbs.close();		
			}
		}
	}

	
	
	/**
	 * Esegue l'operazione di <code>INSERT</code> e in caso di record gi presente effetua una <code>UPDATE</code>. 
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param dbsu
	 *            il <code>DatabaseStatementUnplugged</code> che deve essere utilizzato
	 * @return il numero di righe inserite o il progressivo della tabella
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int insertOrUpdate(DatabaseStatementUnplugged dbsu) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT or UPDATE");
		
		DatabaseStatement dbs = null;
		try {			
			dbs = dbsu.getDatabaseStatement();
			
			// Nascondo la query se va in errore, gestisco manualmente l'opeazione
			dbs.doNotTraceQueryInErrorLog();
			
			// Eseguo la INSERT
			return updateRecord(INSERT, dbs);
		} catch (SQLException e_insert) {
			
			if (!dbs.getDBMS().checkError(e_insert.getErrorCode(), DatabaseError.DUPLICATE_ENTRY)) {
				// Stampo la query nel log non stampata in precedenza
				dbs.stampaNotTracedSQL(e_insert);
				throw e_insert;
			}
			if (logSQL.isTraceEnabled()) { 
				logSQL.debug("[WARN] Catch exception in insert, do update");
			}
		} finally {
			if (dbs!=null) {
				// Reset delle condizioni per stampare la query
				dbs.resetTraceQueryInErrorLog();
			}
		}
		
		// ...allora eseguo la UPDATE
		return updateRecord(UPDATE, dbs);	
		
	}
	
	/**
	 * @deprecated
	 * Esegue l'operazione di <code>INSERT</code> e in caso di record gi presente effetua una <code>UPDATE</code>. 
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * L'operazione &ecaute; atomica.
	 * 
	 * @param cm
	 * @return
	 * @throws SQLException
	 */
	public final int insertOrDuplicateKeyUpdate(ConnectionManager cm) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT_OR_DUPLICATE_KEY_UPDATE");
		return updateRecord(cm, INSERT_OR_DUPLICATE_KEY_UPDATE);
	}
	
	/**
	 * @deprecated
	 * Esegue l'operazione di <code>INSERT</code> e in caso di record gi presente effettua una <code>UPDATE</code>. 
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * @param dbsu
	 * @return
	 * @throws SQLException
	 */
	public final int insertOrDuplicateKeyUpdate(DatabaseStatementUnplugged dbsu) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key INSERT_OR_DUPLICATE_KEY_UPDATE");
		return updateRecord(INSERT_OR_DUPLICATE_KEY_UPDATE, dbsu.getDatabaseStatement());		
	}
	
	
	
	/**
	 * Esegue l'operazione di <code>UPDATE</code>. L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @return il numero di righe aggiornate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int update(ConnectionManager cm) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key UPDATE");
		return updateRecord(cm, UPDATE);
	}

	/**
	 * Esegue l'operazione di <code>UPDATE</code>. L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param dbsu
	 *            il <code>DatabaseStatementUnplugged</code> che deve essere utilizzato
	 * @return il numero di righe aggiornate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int update(DatabaseStatementUnplugged dbsu) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key UPDATE");
		return updateRecord(UPDATE, dbsu.getDatabaseStatement());
	}

	/* Tolto da versione 1.5.3 *
	 * @deprecated 
	 * Esegue l'operazione di <code>UPDATE</code>. Utilizza il <code>ConnectionManager</code> di default. <br>
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @return il numero di righe aggiornate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 * /
	public final int update() throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key UPDATE @deprecated");
		return update(UPDATE);
	}
	*/

	/**
	 * Esegue l'operazione di <code>DELETE</code>. 
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param cm
	 *            il <code>ConnectionManager</code> che deve essere utilizzato
	 * @return il numero di righe eliminate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int delete(ConnectionManager cm) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key DELETE");
		return updateRecord(cm, DELETE);
	}

	/**
	 * Esegue l'operazione di <code>DELETE</code>. L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @param dbsu
	 *            il <code>DatabaseStatementUnplugged</code> che deve essere utilizzato
	 * @return il numero di righe eliminate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 */
	public final int delete(DatabaseStatementUnplugged dbsu) throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key DELETE");
		return updateRecord(DELETE, dbsu.getDatabaseStatement());
	}

	/* Tolto da versione 1.5.3 *
	 * @deprecated 
	 * Esegue l'operazione di <code>DELETE</code>. Utilizza il <code>ConnectionManager</code> di default. <br>
	 * L'aggiornamento usa l'implementazione del metodo {@link #updateRecord(UpdatableRow.UpdateType, DatabaseStatement)} di questo oggetto.
	 * 
	 * @return il numero di righe eliminate
	 * @throws SQLException
	 *             se si verifica un errore di accesso al database
	 * /
	public final int delete() throws SQLException {
		if (log.isDebugEnabled()) log.debug("Update record with key DELETE @deprecated");
		return update(DELETE);
	}
	*/
}
