package it.softecspa.database.reverser.util;



import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.math.BigDecimal;


import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

public class ColumnInfo {

	/*  
   	TABLE_CAT String => table catalog (may be null) 
	TABLE_SCHEM String => table schema (may be null) 
	TABLE_NAME String => table name 
	COLUMN_NAME String => column name 
	DATA_TYPE int => SQL type from java.sql.Types 
	TYPE_NAME String => Data source dependent type name, for a UDT the type name is fully qualified 
	COLUMN_SIZE int => column size. For char or date types this is the maximum number of characters, for numeric or decimal types this is precision. 
	BUFFER_LENGTH is not used. 
	DECIMAL_DIGITS int => the number of fractional digits 
	NUM_PREC_RADIX int => Radix (typically either 10 or 2) 
	NULLABLE int => is NULL allowed. 
					columnNoNulls - might not allow NULL values 
					columnNullable - definitely allows NULL values 
					columnNullableUnknown - nullability unknown 
	REMARKS String => comment describing column (may be null) 
	COLUMN_DEF String => default value (may be null) 
	SQL_DATA_TYPE int => unused 
	SQL_DATETIME_SUB int => unused 
	CHAR_OCTET_LENGTH int => for char types the maximum number of bytes in the column 
	ORDINAL_POSITION int => index of column in table (starting at 1) 
	IS_NULLABLE String => "NO" means column definitely does not allow NULL values; 
						  "YES" means the column might allow NULL values. 
						  An empty string means nobody knows. 
	SCOPE_CATLOG String => catalog of table that is the scope of a reference attribute (null if DATA_TYPE isn't REF) 
	SCOPE_SCHEMA String => schema of table that is the scope of a reference attribute (null if the DATA_TYPE isn't REF) 
	SCOPE_TABLE String => table name that this the scope of a reference attribure (null if the DATA_TYPE isn't REF) 
	SOURCE_DATA_TYPE short => source type of a distinct type or user-generated Ref type, SQL type from java.sql.Types (null if DATA_TYPE isn't DISTINCT or user-generated REF) 
*/
	
	private String name;	
	private int dataType;	
		
	private Class<?> javaType;
	private String javaFieldName;
	
	private String typeName;	
	private int precision;
	private int scale;
	
	private boolean nullable;
	private int ordinal;
	private boolean autoincrement;
	
	
	private boolean primary;
	private String primaryKeyName;
	
	public ColumnInfo(ResultSet res) throws SQLException {
		
		name = res.getString("COLUMN_NAME");	
		nullable = (1==res.getInt("NULLABLE"));
		dataType = res.getInt("DATA_TYPE");
		typeName = res.getString("TYPE_NAME");	
			
		precision = res.getInt("COLUMN_SIZE");
		scale = res.getInt("DECIMAL_DIGITS");
		
		ordinal = res.getInt("ORDINAL_POSITION");
		
		autoincrement = "YES".equalsIgnoreCase(res.getString("IS_AUTOINCREMENT"));
	}


	public String getName() {
		return name;
	}


	public int getDataType() {
		return dataType;
	}


	public String getTypeName() {
		return typeName;
	}

	
	public Class<?> getJavaType() {
		if (javaType!=null) return javaType;
		
		/*
		 * Vedi link http://db.apache.org/ojb/docu/guides/jdbc-types.html
		 */
		switch (dataType) {
			case Types.TINYINT:	
				javaType = Byte.class;			
				break;
			case Types.BOOLEAN:
			case Types.BIT:
				javaType = Boolean.class;
				break;
			case Types.BLOB:
				javaType = java.sql.Blob.class;
				break;
			case Types.CLOB:
				javaType = java.sql.Clob.class;
				break;
			case Types.BIGINT:	
				javaType = Long.class;
				break;
			case Types.NUMERIC:
			case Types.DECIMAL:
				javaType = BigDecimal.class;				
				break;
			case Types.REAL:
				javaType = Float.class;
				break;
			case Types.FLOAT:
			case Types.DOUBLE:
				javaType = Double.class;
				break;
			case Types.INTEGER:
				javaType = Integer.class;
				break;
			case Types.SMALLINT:
				javaType = Short.class;
				break;
			case Types.OTHER:
				javaType = Object.class;
				break;
			case Types.DATE:
				javaType = java.sql.Date.class;
				break;
			case Types.TIME:
				javaType = java.sql.Time.class;
				break;
			case Types.TIMESTAMP:
				javaType = java.sql.Timestamp.class;
				break;
			case Types.VARBINARY:
			case Types.LONGVARBINARY:
			case Types.BINARY:	
				javaType = byte[].class;
				break;
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.LONGVARCHAR:
				javaType = String.class;
				break;
			case Types.ARRAY:
				javaType = java.sql.Array.class;
				break;
			case Types.STRUCT:
				javaType = java.sql.Struct.class;
				break;
			case Types.REF:
				javaType = java.sql.Ref.class;
				break;
			case Types.DATALINK:
				javaType = java.net.URL.class;
				break;
		}
		return javaType;		
	}
	
	public boolean isJavaTypeNumeric() {
		/*
		 * Vedi link http://db.apache.org/ojb/docu/guides/jdbc-types.html
		 */
		
		switch (dataType) {
			case Types.TINYINT:
			case Types.BOOLEAN:
			case Types.BIT:
			case Types.BIGINT:	
			case Types.NUMERIC:
			case Types.DECIMAL:
			case Types.REAL:
			case Types.FLOAT:
			case Types.DOUBLE:
			case Types.INTEGER:
			case Types.SMALLINT:
				return true;

		default:
			/*	
			case Types.BLOB:
			case Types.CLOB:
			case Types.OTHER:
		
			case Types.DATE:
			case Types.TIME:
			case Types.TIMESTAMP:
			case Types.VARBINARY:
			case Types.LONGVARBINARY:
			case Types.BINARY:	
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.LONGVARCHAR:
			case Types.ARRAY:
			case Types.STRUCT:
			case Types.REF:
			case Types.DATALINK:
			*/
		}
		return false;		
	}
	
	public boolean isJavaTypeCalendar() {
		if (javaType!=null) return javaType==EnterpriseCalendar.class;
		
		/*
		 * Vedi link http://db.apache.org/ojb/docu/guides/jdbc-types.html
		 */
		switch (dataType) {
			case Types.DATE:
			case Types.TIME:
			case Types.TIMESTAMP:
				return true;

		default:
			/*	
			case Types.TINYINT:
			case Types.BOOLEAN:
			case Types.BIT:
			case Types.BIGINT:	
			case Types.NUMERIC:
			case Types.DECIMAL:
			case Types.REAL:
			case Types.FLOAT:
			case Types.DOUBLE:
			case Types.INTEGER:
			case Types.SMALLINT:
			
			case Types.BLOB:
			case Types.CLOB:
			case Types.OTHER:
			case Types.VARBINARY:
			case Types.LONGVARBINARY:
			case Types.BINARY:	
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.LONGVARCHAR:
			case Types.ARRAY:
			case Types.STRUCT:
			case Types.REF:
			case Types.DATALINK:
			*/
		}
		return false;		
	}
	
	public void setJavaType(Class<?> javaType) {
		this.javaType = javaType;		
	}
	
	
	
	public int getPrecision() {
		return precision;
	}


	public int getScale() {
		return scale;
	}


	public boolean isNullable() {
		return nullable;
	}


	public int getOrdinal() {
		return ordinal;
	}


	public boolean isPrimary() {
		return primary;
	}


	void setPrimary(boolean primary) {
		this.primary = primary;
	}


	public String getPrimaryKeyName() {
		return primaryKeyName;
	}

	void setPrimaryKeyName(String primaryKeyName) {
		this.primaryKeyName = primaryKeyName;
	}


	public String getJavaFieldName() {
		if (javaFieldName!=null) return javaFieldName;

		javaFieldName = new String(name);
			
		javaFieldName = XString.replaceAll(javaFieldName, "_", " ");
		javaFieldName = XString.replaceAll(javaFieldName, "-", " ");
		javaFieldName = XString.replaceAll(javaFieldName, ".", " ");
		javaFieldName = javaFieldName.trim();

		// Prima lettera minuscola
		javaFieldName = XString.toUpperCaseFirst(javaFieldName.toLowerCase());

		// Maiuscolo dopo lo spazio/underscore
		int i = 0;
		StringBuffer appoggio = new StringBuffer();

		while ((i = javaFieldName.indexOf(" ")) >= 0) {
			appoggio.append(javaFieldName.substring(0, i));
			i++;
			javaFieldName = XString.toUpperCaseFirst(javaFieldName.substring(i));
		}
		appoggio.append(javaFieldName);
		javaFieldName = XString.toLowerCaseFirst(appoggio.toString());

		
		
		return javaFieldName;
	}

	public void setJavaFieldName(String javaFieldName) {
		this.javaFieldName = javaFieldName;
	}


	public boolean isAutoincrement() {
		return autoincrement;
	}


	
	
	
	
}
