package it.softecspa.kahuna.io;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

@SuppressWarnings("serial")
public class File extends java.io.File {

	public File(java.io.File file) {
		super(file, "");
	}

	public File(String arg0) {
		super(arg0);
	}

	public File(String arg0, String arg1) {
		super(arg0, arg1);
	}

	public File(java.io.File arg0, String arg1) {
		super(arg0, arg1);
	}

	public File(URI arg0) {
		super(arg0);
	}

	public static List<String> flush(String percorso, int giorni, String filtro) {
		if (giorni <= 0) return null;

		FilenameFilter filter = null;
		if (filtro != null) {
			filter = new FilenameFilter(filtro);
			// Debug.livelloDebug("Applicato filtro '"+filtro+"' al flush dei file");
		}
		File dirLettura = new File(percorso);
		java.io.File fileLettura[] = (java.io.File[]) (filter == null ? dirLettura.listFiles() : dirLettura.listFiles(filter));
		if (fileLettura == null)
			return null;

		EnterpriseCalendar ieri = new EnterpriseCalendar();
		ieri.add(EnterpriseCalendar.DATE, -giorni);
		// Debug.livelloDebug("Verifica e cancellazione file anteriori al "+ieri.formatted("YYYY-MM-DD HH:NN.SS"));
		List<String> cancellati = new ArrayList<String>();

		// Ordinamento file --------
		Object sorted[] = fileLettura;
		if (sorted != null) {
			Arrays.sort(sorted);
			for (int i = 0; i < sorted.length; i++) {
				File elementoFile = new File((java.io.File) sorted[i]);
				if (elementoFile.isFile()) {
					java.util.Date data = new java.util.Date(elementoFile.lastModified());
					if (data.before(ieri.getTime())) {
						// Debug.livelloDebug(" - Cancellato file '"+elementoFile.getName()+"'");
						cancellati.add(elementoFile.getName());
						elementoFile.delete();
					}
				}
			}
		}

		return cancellati;
	}

	/**
	 * Esegue una copia del file in quello di destinazione
	 * 
	 * @param file
	 *            di destinazione
	 * @return true se la copia avviene correttamente
	 * @throws IOException
	 */
	public boolean copyTo(File target) throws IOException {
		return copy(this, target);
	}

	/**
	 * Esegue una copia del file in quello di destinazione
	 * 
	 * @param nome
	 *            del file destinazione
	 * @return una istanza al file copiato o null se la copia non avviene
	 * @throws IOException
	 */
	public File copyTo(String target) throws IOException {
		File destinazione = new File(target);
		return (copy(this, destinazione) ? destinazione : null);
	}

	/**
	 * Esegue la copia del del file su un altro
	 * 
	 * @param source
	 * @param target
	 * @return
	 * @throws IOException
	 */
	public static boolean copy(File source, File target) throws IOException {
		if (target.exists()) {
			if (!target.delete()) {
				throw new IOException("Errore in cancellazione file '" + target.getAbsolutePath() + "'");
			}
		}

		FileChannel inChannel = null;
		FileChannel outChannel = null;
		try {
			inChannel = new FileInputStream(source).getChannel();
			outChannel = new FileOutputStream(target).getChannel();
			inChannel.transferTo(0, inChannel.size(), outChannel);
		} finally {
			try {
				if (inChannel!=null) {
					inChannel.close();
				}
			} finally {
				if (outChannel!=null) { 
					outChannel.close();
				}
			}
		}
		return true;
	}

	/**
	 * Estrae l'elenco dei file di una directory, senza utilizzare filtri
	 * 
	 * @param percorso
	 * @param solofile
	 * @return
	 */
	public static List<File> dir(String percorso, boolean solofile) {
		return dir(percorso, null, null, solofile);
	}

	/**
	 * Estrae l'elenco dei file di una directory
	 * 
	 * @param percorso
	 * @param filtro
	 * @param solofile
	 * @return
	 */
	public static List<File> dir(String percorso, java.io.FilenameFilter filtro, boolean solofile) {
		return dir(percorso, null, filtro, solofile);
	}

	/**
	 * Estrae l'elenco dei file di una directory
	 * 
	 * @param percorso
	 * @param filtro
	 * @param solofile
	 * @return
	 */
	public static List<File> dir(String percorso, it.softecspa.kahuna.io.FilenameFilter filtro, boolean solofile) {
		return dir(percorso, filtro, null, solofile);
	}

	public List<File> dir(java.io.FilenameFilter filtro, boolean solofile) {
		return File.dir(this.getAbsolutePath(), null, filtro, solofile);
	}

	public List<File> dir(it.softecspa.kahuna.io.FilenameFilter filtro, boolean solofile) {
		return File.dir(this.getAbsolutePath(), filtro, null, solofile);
	}

	public List<File> dir(boolean solofile) {
		return File.dir(this.getAbsolutePath(), null, null, solofile);
	}

	private static List<File> dir(String percorso, it.softecspa.kahuna.io.FilenameFilter filtroN, java.io.FilenameFilter filtroJ, boolean solofile) {
		File dirLettura = new File(percorso);
		if (!dirLettura.exists())
			return null;

		java.io.File fileLettura[] = null;
		if (filtroN != null) {
			// Estrazione con filtro speciale (@see:kahuna.io.FilenameFilter)
			fileLettura = dirLettura.listFiles(filtroN);
		} else if (filtroJ != null) {
			// Estrazione con filtro standard (@see:kahuna.io.FilenameFilter)
			fileLettura = dirLettura.listFiles(filtroJ);
		} else {
			// Estrazione senza filtro
			fileLettura = dirLettura.listFiles();
		}
		if (fileLettura == null)
			return null;

		List<File> lista = new ArrayList<File>();

		// Prendo la lista ordinata dei file
		java.io.File sorted[] = fileLettura;
		Arrays.sort(sorted);
		for (int i = 0; i < sorted.length; i++) {
			File elemento = new File(sorted[i]);
			if (elemento.isFile() || !solofile) {
				// Eseguo le operazioni richieste
				lista.add(elemento);
			}
		}
		return lista;
	}

	public boolean existsCreate() {
		if (super.exists())
			return true;

		File padre = new File(getParent());
		if (padre.existsCreate()) {
			return super.mkdir();
		}

		return false;

	}

	/**
	 * @deprecated  Use {@link #getExtension(String)} instead
	 */
	public static String getEstenzione(String nomefile) {
		int i = nomefile.lastIndexOf(".");
		if (i >= 0)
			return nomefile.substring(i + 1).toLowerCase();
		return "";
	}

	
	public static String getExtension(String filename) {
		int i = filename.lastIndexOf(".");
		if (i >= 0)
			return filename.substring(i + 1).toLowerCase();
		return "";
	}
	
	public String getExtension() {
		return getExtension(this.getName());
	}
	
	public String getNameAlone() {
		return getNameAlone(this.getName());
	}
	
	public static String getNameAlone(String filename) {
		int i = filename.lastIndexOf(".");
		if (i >= 0)
			return filename.substring(0, i);
		return "";
	}
	
	
	public static File changeExtension(File file, String formato) {
		int l = file.getPath().length() - file.getExtension().length();
		File nuovo = new File(file.getPath().substring(0, l) + formato);
		return nuovo;
	}
	
	
	public static String rationalizePath(String valore) {
		if (valore == null)
			return null;
		valore = valore.trim();
		if (File.separator.equals("\\"))
			valore = XString.replaceAll(valore, "\\", "/");
		valore = XString.replaceAll(valore, "//", "/"); // ...toglie i "doppioni"
		return valore.endsWith("/") ? valore : valore + "/"; // aggiunge la finale
	}

	public static void checkMkdirs(String cartella) {
		File cerca = new File(cartella);
		if (!cerca.exists()) {
			cerca.mkdirs();
		}
	}

	public static List<String> archive(String percorso, int giorni, String filtro) throws Exception {
		if (giorni <= 0) return null;

		FilenameFilter filter = null;
		if (filtro != null) {
			filter = new FilenameFilter(filtro);
			// Debug.livelloDebug("Applicato filtro '"+filtro+"' al flush dei file");
		}
		File dirLettura = new File(percorso);
		java.io.File fileLettura[] = (java.io.File[]) (filter == null ? dirLettura.listFiles() : dirLettura.listFiles(filter));
		if (fileLettura == null)
			return null;

		EnterpriseCalendar giorno = new EnterpriseCalendar();
		giorno.add(EnterpriseCalendar.DATE, -giorni);
		// Debug.livelloDebug("Verifica e cancellazione file anteriori al "+ieri.formatted("YYYY-MM-DD HH:NN.SS"));
		List<String> archiviati = new ArrayList<String>();

		// Ordinamento file --------
		Object sorted[] = fileLettura;
		if (sorted != null) {
			Arrays.sort(sorted);
			for (int i = 0; i < sorted.length; i++) {
				File elementoFile = new File((java.io.File) sorted[i]);
				if (elementoFile.isFile()) {
					java.util.Date data = new java.util.Date(elementoFile.lastModified());
					if (data.before(giorno.getTime())) {
						// Debug.livelloDebug(" - Compresso file '"+elementoFile.getName()+"'");
						archiviati.add(elementoFile.getName());
						elementoFile.zip(true);
					}
				}
			}
		}

		return archiviati;		
	}
	
	
	/**
	 * Esegue la compressione del file
	 * @throws Exception
	 */
	public void zip(boolean deletesource) throws Exception {
		File outputFile = changeExtension(this,"zip");
		
		// Cancellazione di sicurezza
		if (outputFile.exists()) {
			outputFile.delete();
		}
		
		byte[] buf = new byte[1024];

		ZipOutputStream out = null;
		FileInputStream in = null;
		try {
			out = new ZipOutputStream(new FileOutputStream(outputFile));

			// Compress the files
			in = new FileInputStream(this);

			// Add ZIP entry to output stream.
			out.putNextEntry(new ZipEntry(this.getName()));

			// Transfer bytes from the file to the ZIP file
			int len;
			while ((len = in.read(buf)) > 0) {
				out.write(buf, 0, len);
			}

			// Complete the entry
			out.closeEntry();

			
		} catch (Exception e) {
			throw new Exception("Error compacting file '"+outputFile.getAbsolutePath()+"'",e);
		} finally {
			if (out!=null) {
				// Complete the ZIP file
				try {
					out.close();
				} catch (IOException e) {
					throw new Exception("Error closing file '"+outputFile.getAbsolutePath()+"'",e);
				}
			}
			
			if (in!=null) {
				// close input stream
				try {
					in.close();
				} catch (IOException e) {
					throw new Exception("Error closing file '"+this.getAbsolutePath()+"'",e);
				}
			}
		}
		
		if (deletesource) {
			// Cancellazione del file origine
			if (outputFile.exists() && this.exists()) {
				this.delete();
			}
		}
		
	}

}
