package it.softecspa.kahuna.io.ftp;

import it.softecspa.kahuna.io.ftp.remoteFile.Path;
import it.softecspa.kahuna.io.ftp.remoteFile.RemoteFile;
import it.softecspa.kahuna.lang.XString;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;



/**
 * <p>Implements an RFC 959 ftp class. It should be noted that many FTP servers
 * do not provide a complete implementation of this RFC. There for this class
 * does not attempt to support the full RFC either.
 * </p>
 *
 * <p>The most notable feature of this class is that when a '5xx' response is
 * returned by the server the method that recieved this response will return a
 * 0 or a null, instead of throwing an exception.</p>
 *
 * <p>Exceptions will only be thrown when we run into an unexpected error such
 * as socket time out or server disconnection.</p>
 *
 * <p> NOTE: This class is based on the FTPConnection class used in the
 * <a href="http://www.radinks.com/ftp/applet/">Rad FTP applet</a>
 *
 * @author Raditha Dissanayake e il Vera 
 * @version 1.1
 */


public class FTP extends it.softecspa.kahuna.io.ftp.FTPConnection {

  private boolean showHiddenFiles;
  
  
  public boolean openConnection(String host, int port) throws FTPException {
    return openConnection(host,port,null,null);
  }
  
  public boolean openConnection(String host, int port, String user, String password) throws FTPException {
  
    super.open(host,port);
    super.initStream();
    if (!super.login(user,password)) {
      super.close();
      return false;
    }        
    if (!super.testConnection()) {
      super.close();
      return false;
    }
      
    return true;
  }
  
  /**
   * Duplica la connessione con gli stessi parmaetri utilizzati per la principale
   * @return
   * @throws UnknownHostException
   * @throws FTPException
   */
  public FTP duplicateConnection() throws UnknownHostException, FTPException {
    
    if (super.isConnected()) {    
      FTP second = new FTP();
      
      second.identify();
      second.setDebug(this.isDebug());
      
      second.setConnectMode(this.getConnectMode());        
      second.setTimeout(this.getTimeout());      
      second.openConnection(this.getHost(),this.getPort(),this.getLoginUser(),this.getLoginPassword());
      if (second.isConnected()) {
        return second;
      } 
    }
    return null;
  }

  
  
  

  /**
   * Retrieves the path to the current working directory by executing
   * the PWD command.
   * @return current directory.
   * @throws IOException
   */
  public String pwd() throws FTPException {
    send_command("PWD");
    if(checkReply("257")) {
      /*
       * in version 1.00 we split by the space (' ') to separate the
       * path from the status code. But this lead to a situation
       * where folders which contained spaces were not being handled
       * correctly.
       *
       * the current approach is possibly more efficient since it does
       * not split the lastMessage into pieces.
       */
      String diretory = "";
      int strt = lastMessage.indexOf('"');
      if(strt > 0) {
        /* we have a path name that is quoted. */
        int end = lastMessage.lastIndexOf('"');
        diretory = lastMessage.substring(strt-1,end+1).trim();
      } else {
        /* path name is not quoted */
        diretory = lastMessage.substring(4,lastMessage.indexOf("is current directory"));
      }
      return XString.replaceAll(diretory,"\"","");
    } else {
      return null;
    }
  }
  
  
  /**
   * Change Working Directory
   * @param dir to change to
   * @return success or failure.
   * @throws IOException
   */
  public boolean cd(String directory) throws FTPException {
    send_command("CWD",directory);
    return checkReply("250");
  }
  
  
  
  public void showHidden(boolean showHiddenFiles) {
    this.showHiddenFiles = showHiddenFiles;
  }
  
  public boolean isShowHidden() {
      return showHiddenFiles;
  }
  
  
  /**
   * Opens a data connection and retrieves the directory listing. The list
   * is a collection of string objects.
   *
   * @param path The pathname to list.
   * @return file list
   * @throws IOException
   */ 
  public List<RemoteFile> list(String path) throws FTPException {
    List<RemoteFile> list = new ArrayList<RemoteFile>();
    DataConnection data_sock = makeDataConnection();
    if (data_sock!= null) {
      String cmd = (showHiddenFiles)?"LIST -al":"LIST";
      if (path==null || path.equals("")) {
        send_command(cmd);
      } else {
        send_command(cmd, path);
      }
      if (checkReply("150") || lastMessage.startsWith("125")) {
        // Windows ftp server returns 125 at times         
        BufferedReader bin = new BufferedReader(new InputStreamReader(data_sock.getInputStream()));
        
        // Here comes the directory listing
        try {
          while(true) {
            String s = bin.readLine();
            if(s==null) break;
            
            RemoteFile f = RemoteFile.parse(s,getServerType());
            if(f!=null) {
              if (f.getName().indexOf(path)<0) {
                path=path.trim();
                f.setPath(path + (path.endsWith("/")?"":"/") + f.getName()); // aggiungo il path
              }
              list.add(f);
            }
          }
          bin.close();
        } catch (IOException e) {
          throw new FTPException(e);
        }

        // Sock_data.close();
        return (checkReply("226"))?list:null;
      }
    }
    
    return null;
  }
  
  /**
   * List the contents of the current working directory.
   * @return file list.
   * @throws IOException
   */
  public List<RemoteFile> list() throws FTPException {
    String path = pwd();
    return list(path!=null?path:"");
  }
  
  
  
  /**
   * Chmod is implemented via the site command.
   *
   * @param perms the permission to apply
   * @param path the file name to apply the permissions to
   * @return success or failure.
   * @throws IOException
   */
  public boolean chmod(int perms,String path)throws FTPException {
    send_command("SITE CHMOD",perms+" "+path);
    return checkReply("200");
  }
      
  /**
   * Deletes specified directory. Does not glob. failes if empty
   * @param path the directory to delete
   * @return success or failure
   * @throws IOException
   */
  public boolean rmdir(String path) throws FTPException {
    send_command("RMD",path);
    return checkReply("250");
  }

  
  /**
   * Create a new directory on the server, will not attempt to determine if
   * the directory already exists.
   *
   * @param dir - name of the new folder.
   * @return success or failure
   * @throws IOException
   */
  public boolean mkdir(String dir) throws FTPException {
    send_command("MKD",dir);
    return checkReply("257");
  }
  
  
  
  /**
   * Deletes specified path. Does not glob.
   *
   * @param path the file to delete
   * @return success or failure
   * @throws IOException
   */
  public boolean delete(String path) throws FTPException {
    send_command("DELE",path);
    return checkReply("250");
  }
  
  
  
    
  
  /**
   * First step in the rename operation.
   * @param path for the victim
   * @return did it work?
   * @throws IOException
   */
  protected boolean rnfr(String path) throws FTPException {
    send_command("RNFR", path);
    return checkReply("350");
  }

  /**
   * the second part of the rename operation.
   * @param path the new name
   * @return how did it go?
   * @throws IOException
   */
  protected boolean rnto(String path) throws FTPException {
    send_command("RNTO", path);
    return checkReply("250");
  }

    
  public boolean renameTo(String prima, String dopo) throws FTPException {
    if (rnfr(prima)) return rnto(dopo); else return false;
  }
  
  
  /**
   * Creates an OutputStream to the file on the remote server. Does not
   * bother with overwriting existing files. You must call the
   * {@link it.softecspa.kahuna.io.ftp.FTPConnection#isOk isOk()}
   * method after you complete writing to clean up the control connection
   * message que
   *
   * @param path - path name on the server.
   * @return <code>OutputStream</code> to which the contents of the file
   *  should be written.
   * @throws IOException
   */
  public OutputStream put(String path) throws FTPException {    
    DataConnection data_sock = makeDataConnection();
    if(data_sock != null) {
      if(path == null || path.equals("")) {
        return null;
      } else {
        send_command("STOR",path);
      }
      if(checkReply("150") || lastMessage.startsWith("125")) {
        return data_sock.getOutputStream();
      }
    }
    return null;
  }
  
  public boolean put(InputStream fis, String nome) throws FTPException {
    OutputStream fos = null;
    try {
      fos = put(nome);
      if (fos==null) throw new FTPException("OutputStream nullo");
      
      byte[] buf = new byte[1024];
      int i = 0;
      try {
        while((i=fis.read(buf))!=-1) {
          fos.write(buf, 0, i);
        }
        fos.close();
      } catch (IOException e) {
        throw new FTPException(e);
      }
    
      return isOk();
    } finally {
      try {
        if (fos!=null)  fos.close();
      } catch (IOException e) {
        throw new FTPException(e);
      }
    }
  }
  
  
  
  
  /**
   * Returns an inputstream to the file on the remote server. Once the
   * file tranfer has been completed, you need to call is the
   * * {@link it.softecspa.kahuna.io.ftp.FTPConnection#isOk isOk()} method
   * to remove the response from the control connection's message que.
   *
   * @param path the name of the file to retrieve.
   * @return the InputStream for the data connection.
   *
   * @throws IOException
   */
  public InputStream get(String path) throws FTPException {      
    DataConnection data_sock = makeDataConnection();
    if(data_sock != null) {
      if(path == null || path.equals("")) {
        return null;
      } else {
        send_command("RETR",path);
      }
      if(checkReply("150") || lastMessage.startsWith("125")) {
        return data_sock.getInputStream();
      }
    }
    return null;
  }

  
  
  public boolean exists(String path) throws FTPException {
    Path local = new Path(path);
    
    // Estraggo i file della cartella
    boolean ret=false;    
    List<RemoteFile> lista = list(local.getDirectory());
    if (lista==null) ret=false;
    ret=(lista.size()>0);
    
    // Scorro la lista dei file per verificare l'esistenza
    if (ret) {
      ret=false;
      Iterator<RemoteFile> iter = lista.iterator();
      while (iter.hasNext()) {      
        RemoteFile file = (RemoteFile)iter.next();
        ret = (file.compareTo(local.getName())==0);
        if (ret) break;
      }
    }
    
    return ret;
  }
      
}

