package it.softecspa.kahuna.io.ftp;

import it.softecspa.kahuna.log.NLoggerManager;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;

import org.apache.log4j.Level;

/**
 * <p>This is a very minimalistic implemenation of a subset
 * of the FTP protocol. One of the design goals at Rad Inks is to keep the
 * size of our software as small as possible and not to not produce bloatware.
 * Thus this class is suitable when you do not need the full functionality
 * of the file transfer protocol.
 * </p>
 *
 * <p>Subclasses provide a more complete implementation and should be used
 * when you need more complete functionality. Please note that all most all
 * servers in wide use only implement a small subset of the FTP RFC. Thus
 * we see no reason to make this and our subclass a complete implementation
 * either.
 * </p>
 *
 * @author Raditha Dissanayake (modificate da il Vera)
 * @version 1.13
 */

public class FTPConnection {
  public static int ACTIVE_MODE=1;
  public static int PASV_MODE=0;

  public static String ASCII="A";
  public static String BINARY="I";

  public final static int MICROSOFT = 0;
  public final static int UNIX = 1;
  

  private boolean debug = false;

  private final static byte[] CRLF = {0x0d,0x0A};
  //private int contentLength;

  
  private int port = 0;
  private String host = null;
  
  private int type = UNIX;
  private int connectMode=ACTIVE_MODE;  
  private int timeout = 60000;
  
  private String last_user=null;
  private String last_password=null;
  
  private OutputStream out;
  private Writer writer;
  
  private InputStream in;
  private Socket sock_control;
  //private Socket sock_data;
  
  private String welcome="";
  protected String lastMessage;
  
  private int identificativo=0;
  private static int identificativo_univoco = 0;
  
     
  public FTPConnection() {    
    debug = NLoggerManager.getLevel().equals(Level.DEBUG);
  }

  
  public FTPConnection(String host,int port, String user, String password) throws FTPException {
    debug = NLoggerManager.getLevel().equals(Level.DEBUG);
    
    this.host=host;
    this.port=port;
    this.last_user = user;
    this.last_password = password;
  }
  
  protected void open() throws FTPException {
    open(this.host,this.port);
  }
  
  
  /**
   * Apre la connessione all'host
   * @param host
   * @param port
   * @throws FTPException
   * @throws UnknownHostException
   */
  protected void open(String host,int port) throws FTPException {
    sock_control = new Socket();
    port = (port<=0?21:port);
    this.host=host;
    this.port=port;    
    InetSocketAddress addr = new InetSocketAddress(host,port);
    log(": connessione a "+host+":"+port+", timeout "+timeout);
    
    try {
      sock_control.connect(addr);
    } catch (IOException e) {
      throw new FTPException(e);
    }        
    try {
      sock_control.setSoTimeout(timeout);
    } catch (SocketException e) {
      throw new FTPException(e);
    }
    try {
      in = sock_control.getInputStream();
    } catch (IOException e) {
      throw new FTPException(e);    }
    try {
      out = sock_control.getOutputStream();
    } catch (IOException e) {
      throw new FTPException(e);
    }
    writer = new OutputStreamWriter(out);    
  }
  
  
  
  public void close() throws FTPException {
    if (sock_control!=null && sock_control.isConnected()) {
      send_command("QUIT");  
      checkReply("221");
      try {
        sock_control.close();
      } catch (IOException e) {
        throw new FTPException(e);
      }
    }    
  }

  
  public  boolean isClosed() {
    return sock_control.isClosed();
  }

  public  boolean isConnected() {
    return sock_control.isConnected();
  }

  
  
  public int getPort() {
    return port;
  }
  
  public String getHost() {
    return host;
  }
  
  
  /**
   * Scrittura del messaggio di riposta del socket
   * @param messaggio
   */
  private void log(String messaggio) {
    if (debug) {
      NLoggerManager.livelloDebug("FTP"+(identificativo>0?"["+identificativo+"]":"")+messaggio);
    }
  }
  
  /**
   * Reads in a line from the control connection.
   * @return the line that we just read.
   * @throws FTPException
   */
  private String getLine() throws FTPException {
    int iBufLen=4096;
    int i=0;
    byte[] buf = new byte[iBufLen];
    
    try {
      for(i=0 ; i < iBufLen; i++) {
        buf[i] = (byte)in.read();
        if(buf[i] == CRLF[1]) break;
      }
    } catch(IOException e) {
      throw new FTPException(e);
    }
    
    return new String(buf,0,i);
  }
  
  /**
   * Returns true if the server returns the expected code. The exact message
   * is not interesting.
   *
   * @param code the status code that we expect.
   * @return did we recieve the expected code.
   *
   * @throws FTPException
   */
  protected boolean checkReply(String code) throws FTPException {
    if(code.length()==4) {
      /*
       * a specific check is made for a multiline reply (eg '230-')
       */
      lastMessage = getLine();
      log("< "+lastMessage);
      if(lastMessage== null || !lastMessage.startsWith(code)) {
        return false;
      } else {
        return true;
      }
    } else {
      /*
       * the programmer doesn't want to bother with handling multi
       * line replies himself so let's handle it for him.
       */
      lastMessage="";
      
      // Modifica Luigi Mengoni - INIZIO
      String line = getLine();
      
      // TODO: questi "return false" dovrebbero essere eccezioni (errori di formato)
      if (line == null) return false;
      
      lastMessage = line;
      log("< " + line);
      
      if (line.length() < 3) return false; // E` obbligatorio almeno un codice di 3 cifre
      String actualCode = line.substring(0, 3);
      
      boolean replyIsMultiline = line.length() >= 4 && line.charAt(3) == '-';
      if (replyIsMultiline) {
        boolean lastReplyLineReached = false;
        while(!lastReplyLineReached) {
          line = getLine();
          if (line == null) return false; // Troppo rigoroso?
          log("< " + line);
          lastMessage += line;
          if (!line.startsWith(actualCode)) return false; // Troppo rigoroso?
          lastReplyLineReached = !line.startsWith(actualCode + '-');
        }
      }
      
      return actualCode.equals(code);
      // Modifica Luigi Mengoni - FINE
      
      /* Vecchio codice sostituito da quello sopra
      String code2 = code +"-";
      String s;

      while(true) {
        s = getLine();
        log("< "+s);
        if(s != null) {
          lastMessage += s;
          if(s.startsWith(code2)) {
            continue;
          }
        }
        break;
      }
      if(s== null || !s.startsWith(code)) {
        return false;
      } else {
        return true;
      }
      */
    }
  }

  
  /**
   * Error messages are typically 500 status codes. This method returns
   * true if such a status code is not encountered.
   *
   * @return did the server accept the last action.
   *
   * @throws FTPException
   */
  public boolean isOk() throws FTPException {
    /** @todo implement support for other error code */
    return !checkReply("500");
  }
  
  
  /**
   * Are we running on windows or linux?
   * @param st
   */
  public void setServerType(String st) {
    if (st.indexOf("MS")!=-1 || 
        st.indexOf("Windows")!=-1) {
      type=MICROSOFT; 
    } else {
      type=UNIX;
    }
    /*
     * TODO: inserire altre piattaforme
     */
  }
  
  public boolean isTypeMicrosoft() {
    return type==MICROSOFT;
  }
  
  public boolean isTypeUnix() {
    return type==UNIX;
  }
  
  
  public int getServerType() {
    return type;
  }
  
  /**
   * The server usually sends a 220 reply when your first
   * connect to it.
   * @throws FTPException
   */
  public void initStream() throws FTPException {
    while(true) {
      if(checkReply("220-")) {
        // ignore the banner
        continue;
      }

      if(lastMessage != null && lastMessage.startsWith("220")) {
        if(lastMessage.indexOf("Microsoft") != -1) {
          setServerType("MS");
        } else {
          setServerType("*nix");
        }
      }
      break;
    }
  }

  
  /**
   * Effettua la login al server con user e password
   * @param username username
   * @param password password
   * @return succes or failure.
   *
   * @throws FTPException
   */
  public boolean login(String user,String password) throws FTPException {
    this.last_user = user;
    this.last_password = password;
    return (user(user) && password(password));
  }
  
  
  /**
   * effettua la login con user e password predefinito o utilizzando l'accesso anaonimo
   *
   * @return did the server accept you?
   * @throws FTPException
   */
  public boolean login() throws FTPException {    
    if (this.last_user == null) {
      log(": anonymous login");
      return login("anonymous","pass");
    } else {
      return login(last_user,last_password);
    }
  }
  
  
  /**
   * Sends the USER command to the server, you can call this method
   * directory if you want more control than given by the
   * {@link #login login()} method.
   * @param user username
   * @return true if the username is acceptable.
   * @throws FTPException
   */
  private boolean user(String user) throws FTPException {
    String user_cmd ="USER " + user;
    writeln(user_cmd);
    return checkReply("331");
  }

  /**
   * Sends the password as part of the user authentication procedure and reads
   * the welcome message if one is available.
   *
   * @param pass - password for the user
   * @return Is the password accepted?
   *
   * @throws FTPException
   */
  private boolean password(String pass) throws FTPException {
    send_command("PASS", (pass==null) ? "anonymous" : pass);
    if (checkReply("230-")) {
      /* we have a welcome message */
      while(true) {
        welcome += lastMessage;
        if(!checkReply("230-")) {
          return lastMessage.startsWith("230");
        }
      }
    } else {
      return lastMessage.startsWith("230");
    }
  }
  
  /**
   * Send a command to the server over the control connection.
   * @param command the command to excute
   * @param params the parameters for the command.
   * @throws FTPException
   */
  protected void send_command(String command, String params) throws FTPException {
    writeln(command.toUpperCase() + " " + params);
  }
  
  protected void send_command(String command) throws FTPException {
    writeln(command.toUpperCase());
  }
  
  /**
   * Centralize all the write operations for ease of maintenance.
   *
   * @param s the data to be sent over the control connection.
   * @throws FTPException
   */
  private void writeln(String s) throws FTPException {
    if (writer==null) throw new FTPException("You are not connected to host, writer is null");
    
    try {
      writer.write(s);
      writer.write("\r\n");
      writer.flush();
      log("> " + s);
    } catch (IOException e) {
      throw new FTPException(e.toString());        
    }      
  }
  
  /**
   * Return the writer associated with the control socket.
   * @return Writer
   */
  public Writer getWriter() {
    return writer;
  }
    
  
  /**
   * Returns the last response from the server. You may need to call this
   * method if {@link #check_reply check_reply()} returned false, which
   * indicates that the expected response was not recieved. The last
   * message should then be retrieved for closer ispection.
   *
   * @return last response.
   */
  public String getLastMessage() {
    return lastMessage;
  }

  /**
   * Switch between Active and Passive modes
   * @return current mode
   */
  public int switchMode() {
    connectMode=(connectMode==ACTIVE_MODE?PASV_MODE:ACTIVE_MODE);
    log(": switch to "+(connectMode==ACTIVE_MODE?"ACTIVE_MODE":"PASSIV_MODE"));
    return connectMode;
  }

  
  private short makeUnsignedShort(byte b) {
    return (b<0)?(short)(b+256):(short)b;
  }
  
  /**
   * Open a data connection in ACTIVE mode. Active mode requires that the
   * client listens for incoming connections - effectively a reversal of the
   * tradicational client/server relationship.
   *
   * @return a ServerSocket to listen on.
   * @throws FTPException
   */
  protected ServerSocket port() throws FTPException {
    ServerSocket socket;
    try {
      socket = new ServerSocket(0);
    } catch (IOException e) {
      throw new FTPException(e);
    }
    InetAddress localhost = sock_control.getLocalAddress();
    int s_port = socket.getLocalPort();

    byte[] ip = localhost.getAddress();
    byte[] port = new byte[2];

    port[0] =(byte)(s_port >> 8); // most significant babes
    port[1] =(byte)(s_port & 0x00FF);

    String cmd = "PORT " + makeUnsignedShort(ip[0]) + "," +
                  makeUnsignedShort(ip[1]) + "," + makeUnsignedShort(ip[2]) +
                  "," +  makeUnsignedShort(ip[3]) + "," +
                  makeUnsignedShort(port[0]) + "," + makeUnsignedShort(port[1]);
    writeln(cmd);
    if(checkReply("200")) {
      return socket;
    } else {
      return null;
    }
  }

  /**
   * Open a passive mode data connection.
   * @return a client Socket
   * @throws FTPException
   */
  protected Socket pasv() throws FTPException {
    writeln("PASV");
    if (checkReply("227")) {
      int start = lastMessage.indexOf('(');
      int end = lastMessage.indexOf(')');
      String sockaddr = lastMessage.substring(start+1,end);
      String[] parts = sockaddr.split(",");
      /* why loop when it's only a single statement? */
      String s_hostIP = parts[0] + "." + parts[1] + "." + parts[2] + "." + parts[3];
      /* get the port */
      int port = (Integer.parseInt(parts[4]) << 8) + Integer.parseInt(parts[5]);

      /* create a socket and return it */
      //System.out.println(s_hostIP +":" + port);
      try {
        return new Socket(s_hostIP, port);
      } catch (UnknownHostException e) {
        throw new FTPException(e);        
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return null;
  }

  /**
   * Open a new active or passive data connection
   * @return DataConnection
   * @throws FTPException
   */
  public DataConnection makeDataConnection() throws FTPException {

    DataConnection con = new DataConnection();
    if(connectMode == ACTIVE_MODE) {
      con.sock_active = port();
    } else {
      con.sock_pasv = pasv();
    }
    return con;
  }
  
  
  
  /**
   * This method is used to determine if the connection is still active
   * or has been dropped due to time out or some related issue. We make
   * use of the noop call.
   * @return
   */
  public boolean testConnection() {
    if(sock_control.isConnected() &&
      !sock_control.isInputShutdown() &&
      !sock_control.isOutputShutdown()) {
      try {
        writeln("NOOP");
        return checkReply("200");
      }
      catch (FTPException e) {
        return false;
      }
    }
    return false;
  }

  
  /**
   * Switch between ASCII and BINARY modes.
   * @param mode A for ASCII or I for Binary.
   * @return success or failure
   * @throws FTPException
   */
  public boolean type(String mode) throws FTPException {
    send_command("TYPE",mode);    
    return checkReply("200");    
  }
  
  public boolean typeASCII() throws FTPException {
    return type(ASCII);
  }
  
  public boolean typeBINARY() throws FTPException {
    return type(BINARY);
  }
  
  
    
  public InputStream getIn() {
      return in;
  }

  public OutputStream getOut() {
      return out;
  }  
  
  
    
  public boolean isDebug() {
    return debug;
  }
  
  public void setDebug(boolean debug) {
    this.debug = debug;
  }

  
  public int getConnectMode() {
    return connectMode;
  }
  
  public void setConnectMode(int connectMode) {
    this.connectMode = connectMode;
  }
  
  
  public int getTimeout() {
    return timeout;
  }
  
  public void setTimeout(int timeout) {
    this.timeout = timeout;
  }

  
  public String getWelcome() {
    return welcome;
  }

  
  public String getLoginPassword() {
    return last_password;
  }
  
  public String getLoginUser() {
    return last_user;
  }
  
  
  protected void identify() {
    identificativo = (++identificativo_univoco);
  }
    
  
  
  /**
   * <p>
   * Inner class that acts as an abstraction layer for <code>Socket</code>
   * and <code>ServerSocket</code>.
   * </p>
   * <p>
   * Unfortunately Socket and SeverSocket do not share any ancestors.
   * Therefore we need to create our adapter class that encloses both a
   * ServerSocket and a Socket.
   * </p>
   * <p>Though it's possible to obtain a Socket
   * instance from ServerSocket by calling the accept method() it's not
   * suitable for our work because the thread would immidiately become
   * blocked.
   * </p>
   */
  public class DataConnection {
    ServerSocket sock_active;
    Socket sock_pasv;

    /**
     * Follows the adapter pattern, returns the input stream of the
     * server socket or client socket.
     *
     * @return <code>InputStream</code> for the data connection.
     * @throws FTPException
     */
    public InputStream getInputStream() throws FTPException {
      try {
        if(connectMode == ACTIVE_MODE) {
          return sock_active.accept().getInputStream();
        } else {
          return sock_pasv.getInputStream();
        }
      } catch (IOException e) {
        throw new FTPException(e);
      }  
    }
  
    /**
     * returns the output stream of the client or server socket depending
     * on whether the active or passive mode is in effect.
     *
     * @return <code>OutputStream</code> for the data connection.
     * @throws FTPException
     */
    public OutputStream getOutputStream() throws FTPException {
      try {
        if(connectMode == ACTIVE_MODE) {
          return sock_active.accept().getOutputStream();
        } else {
          return sock_pasv.getOutputStream();
        }
      } catch (IOException e) {
        throw new FTPException(e);
      }  
    }
  }
  
}

/**
 * FTP: File Transfer Protocol
 * File Transfer Protocol (FTP) enables file sharing between hosts. FTP uses TCP to create a virtual connection for control information and then creates a separate TCP connection for data transfers. The control connection uses an image of the TELNET protocol to exchange commands and messages between hosts. 
 * 
 * The key functions of FTP are: 
 * to promote sharing of files (computer programs and/or data), 
 * to encourage indirect or implicit (via programs) use of remote computers, 
 * to shield a user from variations in file storage systems among hosts, and 
 * to transfer data reliably and efficiently. FTP, though usable directly by a user at a terminal, is designed mainly for use by programs. 
 * FTP control frames are TELNET exchanges and can contain TELNET commands and option negotiation. However, most FTP control frames are simple ASCII text and can be classified as FTP commands or FTP messages. FTP messages are responses to FTP commands and consist of a response code followed by explanatory text.
 *  
 * 
 * Protocol Structure - FTP (File Transfer Protocol)  


 * Command Description  
 * ABOR Abort data connection process.  
 * ACCT <account> Account for system privileges.  
 * ALLO <bytes> Allocate bytes for file storage on server.  
 * APPE <filename> Append file to file of same name on server.  
 * CDUP <dir path> Change to parent directory on server.  
 * CWD <dir path> Change working directory on server. 
 * DELE <filename> Delete specified file on server. 
 * HELP <command> Return information on specified command.  
 * LIST <name> List information if name is a file or list files if name is a directory.  
 * MODE <mode> Transfer mode (S=stream, B=block, C=compressed).  
 * MKD <directory> Create specified directory on server.  
 * NLST <directory> List contents of specified directory.  
 * NOOP Cause no action other than acknowledgement from server.  
 * PASS <password> Password for system log-in.  
 * PASV Request server wait for data connection.  
 * PORT <address> IP address and two-byte system port ID. 
 * PWD Display current working directory. 
 * QUIT Log off from the FTP server.  
 * REIN Reinitialize connection to log-in status 
 * REST <offset> Restart file transfer from given offset.  
 * RETR <filename> Retrieve (copy) file from server.  
 * RMD <directory> Remove specified directory on server.  
 * RNFR <old path> Rename from old path.  
 * RNTO <new path> Rename to new path.  
 * SITE <params> Site specific parameters provided by server.  
 * SMNT <pathname> Mount the specified file structure.  
 * STAT <directory>  Return information on current process or directory.  
 * STOR <filename> Store (copy) file to server.  
 * STOU <filename> Store file to server name.  
 * STRU <type> Data structure (F=file, R=record, P=page).  
 * SYST Return operating system used by server 
 * TYPE <data type> Data type (A=ASCII, E=EBCDIC, I=binary).  
 * USER <data type> User name for system log-in.  
 * 
 * Standard FTP messages are as follows:  
 * Response Code Explanatory Text 
 * 110 Restart marker at MARK yyyy=mmmm (new file pointers). 
 * 120 Service ready in nnn minutes. 
 * 125 Data connection open, transfer starting.  
 * 150 Open connection.  
 * 200 OK. 
 * 202 Command not implemented. 
 * 211 (System status reply).  
 * 212 (Directory status reply).  
 * 213 (File status reply).  
 * 214 (Help message reply). 
 * 215 (System type reply).  
 * 220 Service ready.  
 * 221 Log off network.  
 * 225 Data connection open.  
 * 226 Close data connection. 
 * 227 Enter passive mode (IP address, port ID).  
 * 230 Log on network.  
 * 250 File action completed. 
 * 257 Path name created. 
 * 331 Password required.  
 * 332 Account name required.  
 * 350 File action pending. 
 * 421 Service shutting down.  
 * 425 Cannot open data connection.  
 * 426 Connection closed.  
 * 450 File unavailable.  
 * 451 Local error encountered.  
 * 452  Insufficient disk space.  
 * 500 Invalid command. 
 * 501 Bad parameter.  
 * 502 Command not implemented.  
 * 503 Bad command sequence. 
 * 504 Parameter invalid for command.  
 * 530 Not logged onto network. 
 * 532  Need account for storing files. 
 * 550  File unavailable.  
 * 551  Page type unknown.  
 * 552 Storage allocation exceeded.  
 * 553 File name not allowed.  
 * 
 */
