package it.softecspa.kahuna.log;

import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.RootLogger;

public class KLogger {

	/*
	 * Level prioprity Hightest->Lowest
	 * 
	 * - OFF
	 * 
	 * - TRACE - DEBUG - INFO
	 * 
	 * - WARN - ERROR - FATAL
	 * 
	 * - ALL
	 */

	// Variabile globale utilizzata per il log
	private Logger logger;
	private boolean ticketable;

	/**
	 * Restituisce uan istanza di NLog a partire da un logger
	 * 
	 * @param logger
	 */
	private KLogger(Logger logger) {
		this(logger, true);
	}

	/**
	 * Restituisce uan istanza di NLog a partire da un logger E' configurato il ticket
	 * 
	 * @param logger
	 */
	private KLogger(Logger logger, boolean ticket) {
		this.logger = logger != null ? logger : Logger.getRootLogger();
		this.ticketable = ticket;
	}

	/**
	 * Restituisce una istanza di NLog a partire da un'altra istanza di NLog
	 * 
	 * @param log
	 */
	private KLogger(KLogger log) {
		this(log != null ? log.getLogger() : null);
	}

	@SuppressWarnings("unchecked")
	public static KLogger getLogger(Class clazz) {
		return new KLogger(Logger.getLogger(clazz));
	}

	public static KLogger getLogger(String name) {
		return new KLogger(Logger.getLogger(name));
	}

	public static KLogger getRootLogger() {
		return new KLogger(Logger.getRootLogger());
	}

	protected static String key(String prefix, String suffix) {
		return prefix == null ? suffix : prefix + "." + suffix;
	}

	/**
	 * Log a message object with the <code>TRACE</code> level.
	 * 
	 * @param commento
	 */
	public void trace(String commento) {
		/* @deprecated if (isValid()) logger.trace(commento); */
		if (isTraceEnabled())
			logger.trace(commento);
	}

	/**
	 * Log a message object with the <code>DEBUG</code> level.
	 * 
	 * @param commento
	 */
	public void debug(String commento) {
		/* @deprecated if (isValid()) logger.debug(commento); */
		if (isDebugEnabled())
			logger.debug(commento);
	}

	/**
	 * Log a message object with the <code>INFO</code> level.
	 * 
	 * @param commento
	 */
	public void info(String commento) {
		/* @deprecated if (isValid()) logger.info(commento); */
		if (isInfoEnabled())
			logger.info(commento);
	}

	/**
	 * Log a message object with the <code>WARN</code> level.
	 * 
	 * @param commento
	 */
	public void warn(String commento) {
		warn(commento, null);
	}

	/**
	 * Log a message object with the <code>WARN</code> level including the stack trace of the {@link Throwable} <code>t</code> passed as parameter.
	 * 
	 * @param commento
	 * @param exception
	 */
	public void warn(String commento, Throwable exception) {
		if (exception != null) {
			if (isValid()) {
				logger.warn(commento, exception);
			} else {
				LogLog.warn(getTimestamp() + " " + commento, exception);
			}

		} else {
			if (isValid()) {
				logger.warn(commento);
			} else {
				LogLog.warn(getTimestamp() + " " + commento);
			}
		}
	}

	/**
	 * Log a message object with the <code>ERROR</code> level including the stack trace of the {@link Throwable} <code>t</code> passed as parameter.
	 * 
	 * @param commento
	 * @param exception
	 * @return
	 */
	public Ticket error(String commento, Throwable exception) {
		return error(commento, exception, ticketable);
	}

	public Ticket error(Throwable exception) {
		return error(exception != null ? exception.toString() : "", exception, ticketable);
	}

	/**
	 * Log a message object with the <code>ERROR</code> level.
	 * 
	 * @param commento
	 */
	public Ticket error(String commento) {
		return error(commento, null, ticketable);
	}

	public Ticket error(String commento, Throwable e, boolean bTicket) {
		Ticket ticket = null;
		if (bTicket) {
			ticket = new Ticket(e, commento, Level.ERROR);
			commento = "[ticket #" + ticket + "] " + commento;
		}

		if (e != null) {
			if (isValid()) {
				logger.error(commento, e);
			} else {
				LogLog.error(getTimestamp() + " " + commento, e);
			}

		} else {
			if (isValid()) {
				logger.error(commento);
			} else {
				LogLog.error(getTimestamp() + " " + commento);
			}
		}

		return ticket;
	}

	/**
	 * Log a message object with the <code>FATAL</code> level.
	 * 
	 * @param commento
	 * @return
	 */
	public Ticket fatal(String commento) {
		return fatal(commento, null, ticketable);
	}

	/**
	 * Log a message object with the <code>FATAL</code> level including the stack trace of the {@link Throwable} <code>t</code> passed as parameter.
	 * 
	 * @param commento
	 * @param exception
	 * @return
	 */
	public Ticket fatal(String commento, Throwable exception) {
		return fatal(commento, exception, ticketable);
	}

	public Ticket fatal(String commento, Throwable e, boolean bTicket) {
		Ticket ticket = null;
		if (bTicket) {
			ticket = new Ticket(e, commento, Level.FATAL);
			commento = "[ticket #" + ticket + "] " + commento;
		}

		if (e != null) {
			if (isValid()) {
				logger.fatal(commento, e);
			} else {
				LogLog.error(getTimestamp() + " " + commento, e);
			}

		} else {
			if (isValid()) {
				logger.fatal(commento);
			} else {
				LogLog.error(getTimestamp() + " " + commento);
			}
		}

		return ticket;
	}

	public boolean isValid() {
		return logger != null;
	}

	public Logger getLogger() {
		return logger;
	}

	public boolean isTicketable() {
		return ticketable;
	}

	public void setTicketable(boolean valore) {
		ticketable = valore;
	}

	public boolean isEnable() {
		return getLevel().isGreaterOrEqual(Level.DEBUG);
	}

	public Level getLevel() {
		if (!isValid())
			return Level.OFF;
		// Recupero il livello dal logger
		if (logger.getLevel() != null)
			return logger.getLevel();
		// Recupero il livello dal parent logger (ROOT)
		if (logger.getParent() != null && logger.getParent().getLevel() != null)
			return logger.getParent().getLevel();
		// Recupero il livello dal logger ROOT
		if (RootLogger.getRootLogger() != null && RootLogger.getRootLogger().getLevel() != null)
			return RootLogger.getRootLogger().getLevel();
		// ....
		return Level.OFF;
	}

	public void remove() {
		if (isValid())
			logger.removeAppender(logger.getName());
	}

	protected void setLogger(Logger logger) {
		this.logger = logger;
	}

	private String getTimestamp() {
		return EnterpriseCalendar.now().format("yyyy-MM-dd HH:mm:ss,SSS");
	}

	/** @deprecated */
	public boolean isInfoLevel() {
		return getLevel().isGreaterOrEqual(Level.INFO);
	}

	/** @deprecated */
	public boolean isDebugLevel() {
		return getLevel().isGreaterOrEqual(Level.DEBUG);
	}

	/** @deprecated */
	public boolean isTraceLevel() {
		return getLevel().isGreaterOrEqual(Level.TRACE);
	}

	public boolean isInfoEnabled() {
		return isValid() ? logger.isInfoEnabled() : false;
	}

	public boolean isDebugEnabled() {
		return isValid() ? logger.isDebugEnabled() : false;
	}

	public boolean isTraceEnabled() {
		return isValid() ? logger.isTraceEnabled() : false;
	}

}
