package it.softecspa.kahuna.log;

import it.softecspa.kahuna.io.File;
import it.softecspa.kahuna.lang.exception.ConfigurationException;
import it.softecspa.kahuna.sql.Sql;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.kahuna.util.Properties;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;
import org.apache.log4j.helpers.Loader;
import org.apache.log4j.helpers.LogLog;

/**
 * @deprecated
 * @author il Vera
 * 
 */
public class NLoggerManager {

	static final int ROLLING_FILE = 1;
	static final int DATED_FILE = 2;

	public final static String LOG_LEVEL 	= "log.level";
	public final static String LOG_QUERY 	= "log.query";
	public final static String LOG_REALM 	= "log.realm";
	public final static String LOG_FILENAME = "log.filename";
	public final static String LOG_EXT 		= "log.ext";
	public final static String LOG_PATH 	= "log.path";
	public final static String LOG_FLUSH 	= "log.flush";
	public final static String LOG_CONSOLE 	= "log.console";
	public final static String LOG_LAYOUT 	= "log.layout";
	public final static String LOG_LAYOUT_CONVERSIONE_PATTERN = "log.layout.conversion-pattern";

	public static final String LOG_CYCLE 	= "log.cycle";
	public static final String LOG_SIZE		= "log.size";

	// Variabile globale utilizzata per il log
	private static Logger logger;
	private static boolean query = false;

	private static String base;
	private static String path;
	private static String filename;
	private static String ext;

	private static String realm = "kahuna-debug";
	private static Layout layout;

	private static boolean bTicket = true;

	public NLoggerManager() {
	}

	public NLoggerManager(boolean valore) {
		if (logger != null) {
			if (valore) {
				logger.setLevel(Level.DEBUG);
			} else {
				logger.setLevel(Level.OFF);
			}
		}
	}

	public NLoggerManager(Logger valore) {
		logger = valore;
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello DEBUG LIVELLO 1 - pi basso
	 * 
	 * @param commento
	 *            da inviare come messaggio di DEBUG
	 */
	public static void livelloDebug(String commento) {
		livelloDebug(logger, commento);
	}

	public static void livelloDebug(Logger logger, String commento) {
		if (logger != null)
			logger.debug(commento);
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello INFO LIVELLO 2
	 * 
	 * @param commento
	 *            da inviare come messaggio di INFO
	 */
	public static void livelloInfo(String commento) {
		livelloInfo(logger, commento);
	}

	public static void livelloInfo(Logger logger, String commento) {
		if (logger != null)
			logger.info(commento);
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello WARNING LIVELLO 3 - intermedio
	 * 
	 * @param commento
	 *            da inviare come messaggio di WARNING
	 */
	public static void livelloWarning(String commento) {
		livelloWarning(logger, commento, null);
	}

	public static void livelloWarning(Logger logger, String commento) {
		livelloWarning(logger, commento, null);
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello WARNING LIVELLO 3 - intermedio
	 * 
	 * @param commento
	 *            da inviare come messaggio di WARNING
	 * @param e
	 *            eccezione da inviare come messaggio di WARNING
	 */
	public static void livelloWarning(String commento, Throwable e) {
		livelloWarning(logger, commento, e);
	}

	public static void livelloWarning(Logger logger, String commento, Throwable e) {
		if (e != null) {
			if (logger != null) {
				logger.warn(commento, e);
			} else {
				LogLog.warn(getTimestamp() + " " + commento, e);
			}

		} else {
			if (logger != null) {
				logger.warn(commento);
			} else {
				LogLog.warn(getTimestamp() + " " + commento);
			}
		}

	}

	/**
	 * Inserisce nel log precelto un messaggio di livello ERROR LIVELLO 4
	 * 
	 * @param commento
	 *            da inviare come messaggio di ERROR
	 */
	public static Ticket livelloError(String commento) {
		return livelloError(logger, commento, null);
	}

	public static Ticket livelloError(Logger logger, String commento) {
		return livelloError(logger, commento, null, bTicket);
	}

	protected static Ticket livelloError(Logger logger, String commento, boolean bTicket) {
		return livelloError(logger, commento, null, bTicket);
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello ERROR LIVELLO 4
	 * 
	 * @param commento
	 *            da inviare come messaggio di ERROR
	 * @param e
	 *            eccezione da inviare come messaggio di ERROR
	 */
	public static Ticket livelloError(String commento, Throwable e) {
		return livelloError(logger, commento, e);
	}

	public static Ticket livelloError(Logger logger, String commento, Throwable e) {
		return livelloError(logger, commento, e, bTicket);
	}

	public static Ticket livelloError(Logger logger, String commento, Throwable e, boolean bTicket) {
		Ticket ticket = null;
		if (bTicket) {
			ticket = new Ticket(e, commento, Level.ERROR);
			commento = "[ticket #" + ticket + "] " + commento;
		}

		if (e != null) {
			if (logger != null) {
				logger.error(commento, e);
			} else {
				LogLog.error(getTimestamp() + " " + commento, e);
			}

		} else {
			if (logger != null) {
				logger.error(commento);
			} else {
				LogLog.error(getTimestamp() + " " + commento);
			}
		}

		return ticket;
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello ERROR per l'errore di tipo SQL LIVELLO 4
	 * 
	 * @param e
	 *            eccezione da inviare come messaggio di ERROR SQL
	 */
	public static Ticket livelloSQLError(SQLException e) {
		return livelloSQLError(logger, e, (String) null);
	}

	public static Ticket livelloSQLError(Logger logger, SQLException e) {
		return livelloSQLError(logger, e, (String) null, bTicket);
	}

	protected static Ticket livelloSQLError(Logger logger, SQLException e, boolean bTicket) {
		return livelloSQLError(logger, e, (String) null, bTicket);
	}

	public static Ticket livelloSQLError(SQLException e, Sql sql) {
		return livelloSQLError(logger, e, sql);
	}

	public static Ticket livelloSQLError(Logger logger, SQLException e, Sql sql) {
		return livelloSQLError(logger, e, sql, bTicket);
	}

	protected static Ticket livelloSQLError(Logger logger, SQLException e, Sql sql, boolean bTicket) {
		return livelloSQLError(logger, e, sql != null ? sql.getSql() : null, bTicket);
	}

	public static Ticket livelloSQLError(SQLException e, String sql) {
		return livelloSQLError(logger, e, sql);
	}

	public static Ticket livelloSQLError(Logger logger, SQLException e, String sql) {
		return livelloSQLError(logger, e, sql, bTicket);
	}

	protected static Ticket livelloSQLError(Logger logger, SQLException e, String sql, boolean bTicket) {
		Ticket ticket = null;

		if (e != null) {
			String commento = "ErrorCode = " + e.getErrorCode();
			if (bTicket) {
				ticket = new Ticket(e, commento, Level.ERROR);
				commento = "[ticket #" + ticket + "] " + commento;
			}

			if (logger != null) {
				logger.error(getTimestamp() + " " + commento, e);
				if (sql != null) {
					logger.error("Statement SQL: \n\n" + sql);
				}

			} else {
				LogLog.error(getTimestamp() + " " + commento, e);
				if (sql != null) {
					LogLog.error("Statement SQL: \n\n" + sql);
				}
			}
		}
		return ticket;
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello FATAL LIVELLO 5 - pi alto, errore bloccante
	 * 
	 * @param commento
	 *            da inviare come messaggio di ERROR
	 */
	public static Ticket livelloFatal(String commento) {
		return livelloFatal(logger, commento, null);
	}

	public static Ticket livelloFatal(Logger logger, String commento) {
		return livelloFatal(logger, commento, bTicket);
	}

	protected static Ticket livelloFatal(Logger logger, String commento, boolean bTicket) {
		return livelloFatal(logger, commento, null, bTicket);
	}

	/**
	 * Inserisce nel log precelto un messaggio di livello FATAL LIVELLO 5 - pi alto, errore bloccante
	 * 
	 * @param commento
	 *            da inviare come messaggio di ERROR
	 * @param e
	 *            eccezione da inviare come messaggio di FATAL
	 */
	public static Ticket livelloFatal(String commento, Throwable e) {
		return livelloFatal(logger, commento, e);
	}

	public static Ticket livelloFatal(Logger logger, String commento, Throwable e) {
		return livelloFatal(logger, commento, e, bTicket);
	}

	protected static Ticket livelloFatal(Logger logger, String commento, Throwable e, boolean bTicket) {
		Ticket ticket = null;
		if (bTicket) {
			ticket = new Ticket(e, commento, Level.FATAL);
			commento = "[ticket #" + ticket + "] " + commento;
		}

		if (e != null) {
			if (logger != null) {
				logger.fatal(commento, e);
			} else {
				LogLog.error("FATAL: " + getTimestamp() + " " + commento, e);
			}

		} else {
			if (logger != null) {
				logger.fatal(commento);
			} else {
				LogLog.error("FATAL: " + getTimestamp() + " " + commento);
			}
		}

		return ticket;
	}

	public static String getTimestamp() {
		return EnterpriseCalendar.now().format("yyyy-MM-dd HH:mm:ss,SSS");
	}

	public static Logger getLogger() {
		return logger;
	}

	public static void setLogger(Logger valore) {
		logger = valore;
	}

	private static void resetRootLogger() {
		Logger.getRootLogger().removeAllAppenders();
		logger = null;
	}

	public static boolean isQuery() {
		return query;
	}

	public static void setQuery(boolean valore) {
		query = valore;
	}

	public static Level getLevel() {
		return (logger != null ? Logger.getRootLogger().getLevel() : Level.OFF);
	}

	public static boolean isEnable() {
		return getLevel().isGreaterOrEqual(Level.DEBUG);
	}

	public static boolean isDebugLevel() {
		return getLevel().equals(Level.DEBUG);
	}

	public static synchronized void autoconfig(Properties prop) throws ConfigurationException {
		// if (prop==null) // TODO null <pointer exception
		List<String> errori = new ArrayList<String>();

		// Livello di debug
		String livello = prop.get(LOG_LEVEL);
		if (livello == null) {
			String msg = "Optional parameter absent: " + LOG_LEVEL + ", default = DEBUG";
			livello = "debug";
			prop.put(LOG_LEVEL, livello);
			LogLog.warn(msg);
		}
		if (livello.equals(Level.OFF)) {
			livello = Level.FATAL.toString();
			prop.put(LOG_LEVEL, livello);
		}

		// Stampa le query in debug
		query = prop.getBoolean(LOG_QUERY, false);

		// Reset del logger
		resetRootLogger();

		// Caricamento nuovo logger
		if (logger == null) {
			// Percorso file di log
			path = prop.get(LOG_PATH);
			if (path == null) {
				String msg = "Mandatory parameter absent: " + LOG_PATH;
				LogLog.error(msg);
				errori.add(msg);
			}

			// Prefisso del nome del file
			filename = prop.get(LOG_FILENAME);
			if (filename == null) {
				String msg = "Mandatory parameter absent: " + LOG_FILENAME;
				LogLog.error(msg);
				errori.add(msg);
			}
			ext = prop.get(LOG_EXT);
			if (ext == null) {
				String msg = "Optional parameter absent: " + LOG_EXT + ", default = '.log'";
				LogLog.warn(msg);
				ext = "log";
				prop.put(LOG_EXT, ext);
			}

			// Formattazione del lyaout del log
			String pattern_string = prop.get(LOG_LAYOUT_CONVERSIONE_PATTERN);
			if (pattern_string == null) {
				String msg = "Mandatory parameter absent: " + LOG_LAYOUT_CONVERSIONE_PATTERN;
				LogLog.error(msg);
				errori.add(msg);
			}

			if (errori.size() > 0) {
				throw new ConfigurationException(errori);
			}

			// Razionalizzazione del path
			path = File.rationalizePath(path);
			prop.put(NLoggerManager.LOG_PATH, path);

			if (!filename.endsWith("."))
				filename += ".";
			if (ext.endsWith("."))
				ext += "log";
			if (ext.indexOf(".") < 0)
				ext = "." + ext;
			if (base != null)
				path = base + path;

			realm = prop.get(LOG_REALM);
			if (realm == null) {
				realm = "kahuna-log";
				String msg = "Optional parameter absent: " + LOG_REALM + ", default = '" + realm + "'";
				LogLog.warn(msg);
				prop.put(LOG_REALM, ext);
			}

			// Caricamento valore di flush del log
			int autoflush = prop.getInt(NLoggerManager.LOG_FLUSH, -1);
			// Caricamento cycle e autosize
			int cycle = prop.getInt(NLoggerManager.LOG_CYCLE, -1);
			String size = prop.get(NLoggerManager.LOG_SIZE);
			int bufferSize = 0;

			int type = NLoggerManager.DATED_FILE;
			if (cycle <= 0) {
				// Considero il log giornaliero {AUTOFLUSH}
				type = NLoggerManager.DATED_FILE;

				if (autoflush < 0) {
					String msg = "Optional parameter absent: " + NLoggerManager.LOG_FLUSH + ", default = 0";
					LogLog.warn(msg);
					autoflush = 0;
					prop.put(NLoggerManager.LOG_FLUSH, autoflush);
				}

			} else {
				// Considero il log ciclico {CYCLE,SIZE}
				type = NLoggerManager.ROLLING_FILE;

				if (size == null) {
					size = "100K";
					prop.put(NLoggerManager.LOG_SIZE, size);
				}
				try {
					bufferSize = Integer.parseInt(size.substring(0, size.length() - 1));
					String b = size.substring(size.length() - 1);
					if (b.equalsIgnoreCase("K")) {
						bufferSize = bufferSize * 1024;
					} else if (b.equalsIgnoreCase("M")) {
						bufferSize = bufferSize * 1024 * 1024;
					} else {
						bufferSize = bufferSize * 1024;
					}
				} catch (Exception e) {
					errori.add("Value is not valid for parameter: " + NLoggerManager.LOG_SIZE + " (K,M)");
				}

			}

			// Caricamento del ROOT logger
			Logger root_logger = Logger.getRootLogger();
			logger = Logger.getLogger(realm); /*
											 * Nota: senza impostare a FALSE il metodo setAdditivity() gli appendere vengono inseriti a livello di root
											 */
			logger.setLevel(Level.toLevel(livello));

			// Creo l'istanza del Layout Log4j e aggiungo l'appender
			layout = new PatternLayout(pattern_string);
			if (type == NLoggerManager.DATED_FILE) {
				// Creo l'istanza di DatedFileAppender e aggiungo l'appender
				DatedFileAppender dfAppender = new DatedFileAppender(path, filename, ext, autoflush);
				dfAppender.setLayout(layout);
				dfAppender.setName(realm);
				root_logger.addAppender(dfAppender);

			} else if (type == NLoggerManager.ROLLING_FILE) {
				// Creo una istanza di RollingFileAppender e aggiungo l'appendere
				RollingFileAppender rfAppender = new RollingFileAppender();
				rfAppender.setFile(path + filename + ext);
				rfAppender.setLayout(layout);
				rfAppender.setName(realm);
				rfAppender.setBufferSize(bufferSize);
				root_logger.addAppender(rfAppender);
			}

			// Se richiesto scrivo sulla console
			if (prop.getBoolean(LOG_CONSOLE)) {
				ConsoleAppender cAppender = new ConsoleAppender((Layout) layout);
				root_logger.addAppender(cAppender);
			}

		}

		return;
	}

	public static String getPath() {
		return path;
	}

	public static String getFilename() {
		return filename;
	}

	public static String getExt() {
		return ext;
	}

	public static String getAppenderFile() {
		if (logger == null)
			return null;
		Appender appender = logger.getAppender(realm);
		if (appender.getClass().isInstance(DatedFileAppender.class)) {
			return ((DatedFileAppender) appender).getFile();
		}
		return null;
	}

	public static void flush(String percorso, int giorni) {
		flush(percorso, giorni, null);
	}

	public static void flush(String percorso, int giorni, String filtro) {
		if (giorni > 0) {
			if (getLevel().equals(Level.DEBUG)) {
				if (filtro != null)
					NLoggerManager.livelloDebug("Applicato filtro '" + filtro + "' al flush dei file di log");

				EnterpriseCalendar ieri = new EnterpriseCalendar();
				ieri.add(EnterpriseCalendar.DATE, -giorni);
				NLoggerManager.livelloInfo("Verifica e cancellazione file log anteriori al " + ieri.format("dd/MM/yyyy HH:mm.ss"));
			}
			List<String> cancellati = File.flush(percorso, giorni, filtro);
			if (cancellati != null && getLevel().equals(Level.DEBUG)) {
				for (int i = 0; i < cancellati.size(); i++) {
					NLoggerManager.livelloDebug(" - Cancellato file '" + cancellati.get(i) + "'");
				}
			}
		}

	}

	/**
	 * Instantiate an object given a class name. Check that the <code>className</code> is a subclass of <code>superClass</code>. If that test fails or the object could not be instantiated, then <code>defaultValue</code> is returned.
	 * 
	 * @param className
	 *            The fully qualified class name of the object to instantiate.
	 * @param superClass
	 *            The class to which the new object should belong.
	 * @param defaultValue
	 *            The object to return in case of non-fulfillment
	 */
	public static Object instantiateByClassName(String className, Class<?> superClass, Object defaultValue) {
		if (className != null) { // Class<?>
			try {
				Class<?> classObj = Loader.loadClass(className);
				if (!superClass.isAssignableFrom(classObj)) {
					LogLog.error("A \"" + className + "\" object is not assignable to a \"" + superClass.getName() + "\" variable.");
					LogLog.error("The class \"" + superClass.getName() + "\" was loaded by ");
					LogLog.error("[" + superClass.getClassLoader() + "] whereas object of type ");
					LogLog.error("\"" + classObj.getName() + "\" was loaded by [" + classObj.getClassLoader() + "].");
					return defaultValue;
				}
				return classObj.newInstance();
			} catch (Exception e) {
				LogLog.error("Could not instantiate class [" + className + "].", e);
			}
		}
		return defaultValue;
	}

	public static Layout getLayout() {
		return layout;
	}

	public static void setTicket(boolean ticket) {
		bTicket = ticket;
	}

	public static void setBase(String valore) {
		if (valore != null)
			valore = valore.trim();
		if (valore.equals(""))
			valore = null;
		if (valore != null)
			valore += "/";
		base = valore;

	}

	public static String getBase() {
		return base;
	}

}
