package it.softecspa.kahuna.mail12;

import it.softecspa.kahuna.lang.XString;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.StringTokenizer;

import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Message;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

public class Lettera extends MailCommon {
	
	public static final String ADDRESS_SEPARATOR = ",";
	private final String SENDER = "Kahuna 1.2.4";

	private final static String XPRIORITY[] = { "Highest", "High", "Normal", "Low", "Lowest" };
	public final static String XPRIORITY_HIGHEST = "1";
	public final static String XPRIORITY_HIGH = "2";
	public final static String XPRIORITY_NORMAL = "3";
	public final static String XPRIORITY_LOW = "4";
	public final static String XPRIORITY_LOWEST = "5";

	private final static String XMSMAIL_PRIORITY_HIGH = "High";
	private final static String XMSMAIL_PRIORITY_NORMAL = "Normal";
	private final static String XMSMAIL_PRIORITY_LOW = "Low";

	private final static String PRIORITY_URGENT = "urgent";
	private final static String PRIORITY_NORMAL = "normal";
	private final static String PRIORITY_NON_URGENT = "non-urgent";

	public final static String CONTENT_TYPE_HTML = "text/html";
	public final static String CONTENT_TYPE_TEXT = "text/plain";

	private String sFrom;
	private String sTo;
	private String sCC;
	private String sBCC;
	private String sReplay;
	private String sSubject;
	private String sText;
	private String sXMailer;
	private String sXPriority;
	private String sContetType;
	private Date sendDate;
	public Flags flags;
	protected List<Allegato> listaAllegati;
	@SuppressWarnings("unused")
	private int nAllegatiValidi;
	private boolean bSpedita;
	private boolean bErrata;
	@SuppressWarnings("unused")
	private String sErrore;

	private Object tag; // Elemento di controllo

	private boolean primoGiro = true;

	public Lettera() {
		super();
		sFrom = "";
		sTo = "";
		sCC = "";
		sBCC = "";
		sReplay = "";
		sSubject = "";
		sText = "";
		sXMailer = "";
		sXPriority = PRIORITY_NORMAL;
		sContetType = CONTENT_TYPE_TEXT;

		nAllegatiValidi = 0;
		sErrore = "nessuno";
		primoGiro = true;
	}

	protected void setSpedita(boolean b) {
		bSpedita = b;
	}

	protected void setErrata(String errore) {
		bErrata = true;
		sErrore = errore;
	}

	public boolean isSpedita() {
		return bSpedita;
	}

	public boolean isErrata() {
		return bErrata;
	}

	public static void controllaIndirizzo(String valore) throws AddressException {
		if (valore.indexOf("\"") > 0)
			throw new AddressException();
		int p1 = valore.indexOf("@");
		if (p1 < 0)
			throw new AddressException();
		int p2 = valore.substring(p1).indexOf(".");
		if (p2 < 0)
			throw new AddressException();

		p1 = valore.indexOf("<");
		p2 = valore.indexOf(">");
		if ((p1 < 0 && p2 >= 0) || (p1 >= 0 && p2 < 0))
			throw new AddressException();
		if (p1 >= 0 && p2 >= 0) {
			if (p1 > p2)
				throw new AddressException();
		}

		new InternetAddress(valore);
	}

	/**
	 * Estrae da una lista separata da delimitatore l'elenco degli indirizzi
	 * email
	 * 
	 * @param indirizzo
	 * @param verifica
	 * @return
	 * @throws IndirizzoException
	 */
	private String[] splitAndCheck(String indirizzo, boolean verifica) throws AddressException {
		indirizzo = XString.replaceAll(indirizzo, "\"", ""); // il carattere " 
		// deleterio
		// nell'indirizzo
		indirizzo = XString.replaceAll(indirizzo, ";", ADDRESS_SEPARATOR);
		String[] indirizzi = null;
		// Attenzione il metodo split  dalla versione 1.4 o superiori
		try {
			indirizzi = indirizzo.split(ADDRESS_SEPARATOR);
		} catch (NoSuchMethodError e) {
			StringTokenizer st = new StringTokenizer(indirizzo, ADDRESS_SEPARATOR);
			indirizzi = new String[st.countTokens()];
			int i = 0;
			while (st.hasMoreTokens()) {
				indirizzi[i++] = st.nextToken();
			}
		}

		// Se richiesta faccio la verifica
		if (verifica) {
			for (int i = 0; i < indirizzi.length; i++) {
				controllaIndirizzo(indirizzi[i]);
			}
		}
		return indirizzi;
	}

	/**
	 * Aggiunge l'indirizzo alla lista
	 * 
	 * @param base
	 * @param indirizzi
	 * @return
	 */
	private String aggiungiIndirizzo(String base, String[] indirizzo) {
		for (int i = 0; i < indirizzo.length; i++) {
			if (base.indexOf(indirizzo[i]) < 0) {
				base += (base.length() > 0 ? ADDRESS_SEPARATOR : "") + indirizzo[i];
			}
		}
		return base;
	}

	/**
	 * Inserisce il mittente della email.
	 * 
	 * @param String
	 *            il mittente.
	 */
	public void setMittente(String indirizzo) throws AddressException {
		setMittente(indirizzo, true);
	}

	public void setMittente(String indirizzo, boolean verifica) throws AddressException {
		if (XString.isBlankNullTrim(indirizzo))
			return;
		if (verifica)
			controllaIndirizzo(indirizzo);
		this.sFrom = indirizzo;
	}

	protected void addMittente(String indirizzo) throws AddressException {
		addMittente(indirizzo, true);
	}

	protected void addMittente(String indirizzo, boolean verifica) throws AddressException {
		if (indirizzo == null || indirizzo.trim().equals(""))
			return;
		this.sFrom = aggiungiIndirizzo(this.sFrom, splitAndCheck(indirizzo, verifica));
	}

	public String getMittente() {
		return this.sFrom;
	}

	/**
	 * Aggiunge un destinatario alla lista
	 * 
	 * @param String
	 *            il nuovo destinatario
	 */
	public void addDestinatario(String indirizzo) throws AddressException {
		addDestinatario(indirizzo, true);
	}

	public void addDestinatario(String indirizzo, boolean verifica) throws AddressException {
		if (XString.isBlankNullTrim(indirizzo))
			return;
		this.sTo = aggiungiIndirizzo(this.sTo, splitAndCheck(indirizzo, verifica));
	}

	/**
	 * Restituisce i destinatari della email.
	 * 
	 * @return I destinatari separati da <B>,</B>.
	 */
	public String getDestinatario() {
		return this.sTo;
	}

	/**
	 * Aggiunge un conoscente alla lista.
	 * 
	 * @param String
	 *            il nuovo conoscente.
	 */
	public void addCC(String indirizzo) throws AddressException {
		addCC(indirizzo, true);
	}

	public void addCC(String indirizzo, boolean verifica) throws AddressException {
		if (XString.isBlankNullTrim(indirizzo))
			return;
		this.sCC = aggiungiIndirizzo(this.sCC, splitAndCheck(indirizzo, verifica));
	}

	/**
	 * Restituisce i conoscenti della email.
	 * 
	 * @return I conoscenti separati da <B>,</B>.
	 */
	public String getCC() {
		return this.sCC;
	}

	/**
	 * Aggiunge un conoscente riservato alla lista.
	 * 
	 * @param String
	 *            il nuovo conoscente.
	 */
	public void addBCC(String indirizzo) throws AddressException {
		addBCC(indirizzo, true);
	}

	public void addBCC(String indirizzo, boolean verifica) throws AddressException {
		if (XString.isBlankNullTrim(indirizzo))
			return;
		this.sBCC = aggiungiIndirizzo(this.sBCC, splitAndCheck(indirizzo, verifica));
	}

	/**
	 * Restituisce i conoscenti riservati della email.
	 * 
	 * @return I conoscenti separati da <B>,</B>.
	 */
	public String getBCC() {
		return this.sBCC;
	}

	/**
	 * Aggiunge l'indirizzo a cui mandare un replay della email.
	 * 
	 * @param String
	 *            l'indirizzo.
	 */
	public void addReply(String indirizzo) throws AddressException {
		addReply(indirizzo, true);
	}

	protected void addReply(String indirizzo, boolean verifica) throws AddressException {
		if (XString.isBlankNullTrim(indirizzo))
			return;
		this.sReplay = aggiungiIndirizzo(this.sReplay, splitAndCheck(indirizzo, verifica));
	}

	/**
	 * Restituisce la lista delgi indirizzi a cui  fatto il replay della email.
	 * 
	 * @return La lista separata da <B>,</B>.
	 */
	public String getReply() {
		return this.sReplay;
	}

	/**
	 * Inserisce il soggetto della email.
	 * 
	 * @param String
	 *            il soggetto.
	 */
	public void setOggetto(String testo) {
		this.sSubject = testo;
	}

	/**
	 * Restituisce il soggetto della email.
	 * 
	 * @return Stringa.
	 */
	public String getOggetto() {
		return this.sSubject != null ? this.sSubject : "";
	}

	/**
	 * Inserisce il contenuto della email.
	 * 
	 * @param String
	 *            il contenuto.
	 */
	public void setTesto(String testo) {
		this.sText = testo;
	}

	public void addTesto(String valore) {
		if (valore == null)
			return;
		this.sText += valore;
	}

	/**
	 * Restituisce il contenuto della email.
	 * 
	 * @return Stringa.
	 */
	public String getTesto() {
		return this.sText != null ? this.sText : "";
	}

	protected void setXMailer(String valore) {
		this.sXMailer = valore;
	}

	public String getXMailer() {
		return this.sXMailer;
	}

	public String getXPriority() {
		return this.sXPriority;
	}

	public void setXPriority(String valore) {
		int i = 3;
		try {
			i = Integer.parseInt(valore);
		} catch (NumberFormatException e) {
			i = 3;
		}
		this.setXPriority(i);
	}

	public void setXPriority(int i) {
		if (i < 1)
			i = 1;
		else if (i > 5)
			i = 5;
		this.sXPriority = "" + i + " (" + XPRIORITY[i - 1] + ")";
	}

	public String getXMSMailPriority() {
		if (this.sXPriority == null)
			return XMSMAIL_PRIORITY_NORMAL;
		String c = this.sXPriority.substring(0, 1);
		if (c.equals(XPRIORITY_NORMAL)) {
			return XMSMAIL_PRIORITY_NORMAL;
		} else if (c.equals(XPRIORITY_HIGH) || c.equals(XPRIORITY_HIGHEST)) {
			return XMSMAIL_PRIORITY_HIGH;
		}
		return XMSMAIL_PRIORITY_LOW;
	}

	public String getPriority() {
		if (this.sXPriority == null)
			return XMSMAIL_PRIORITY_NORMAL;
		// String c = this.sXPriority.substring(0,1);
		if (this.sXPriority.equals(XPRIORITY_NORMAL)) {
			return PRIORITY_NORMAL;
		} else if (this.sXPriority.equals(XPRIORITY_HIGH) || this.sXPriority.equals(XPRIORITY_HIGHEST)) {
			return PRIORITY_URGENT;
		}
		return PRIORITY_NON_URGENT;
	}

	protected void setSendDate(Date valore) {
		this.sendDate = valore;
	}

	public Date getSendDate() {
		return this.sendDate;
	}

	public int numeroAllegati() {
		return (listaAllegati == null ? 0 : listaAllegati.size());
	}

	/**
	 * Aggiunge un allegato alla email dimensioanndo anche il vettore che li
	 * contiene. Questa chiamata funziona solo la prima volta, altrimenti prende
	 * i valori di default (3,2).
	 * 
	 * @param Allegato
	 *            l'allegato.
	 * @return il numero di allegati presenti nella lettera.
	 */
	public int addAllegato(Allegato allegato) {
		if (listaAllegati == null) {
			listaAllegati = new ArrayList<Allegato>();
		}
		listaAllegati.add(allegato);
		if (allegato.getTipo() == Allegato.ATTACHMENT)
			nAllegatiValidi++;
		return this.numeroAllegati();
	}

	/**
	 * Restituisce l'allegato nella posizione indicata. Viene generata una
	 * eccezione se l'allegato non esiste.
	 * 
	 * @param int il numero dell'allegato che si desidera.
	 * @return l'allegato.
	 */
	public Allegato getAllegato(int numero) throws MailException {
		try {
			if ((numero < 0) || (numero >= this.numeroAllegati())) {
				throw new MailException("L'allegato n. " + numero + " ricercato non  presente");
			}
			return (Allegato) listaAllegati.get(numero);

		} catch (Exception e) {
			throw new MailException(e);
		}
	}

	public List<Allegato> getAllegati() {
		return listaAllegati;
	}

	/**
	 * Toglie l'allegato dalla lettera.
	 * 
	 * @param int il numero dell'allegato da togliere.
	 * @return il numero di allegati nella lista.
	 */
	public int removeAllegato(int posizione) throws MailException {
		if ((posizione < 0) || (posizione >= this.numeroAllegati()))
			return 0;
		try {
			listaAllegati.remove(posizione);
			return this.numeroAllegati();
		} catch (Exception e) {
			throw new MailException(e);
		}
	}

	private void estraiDati(Message m) throws Exception {
		Address[] a = null;

		// X-Mailer
		String[] hdrs = m.getHeader("X-Mailer");
		if (hdrs != null)
			this.setXMailer(hdrs[0]);
		else
			this.setXMailer("non disponibile");
		// Data spedizione
		this.setSendDate(m.getSentDate());
		// Mittente
		try {
			a = m.getFrom();
		} catch (AddressException ae) {
			a = null;
		}
		if (a != null) {
			for (int i = 0; i < a.length; i++)
				this.addMittente(a[i].toString(), false);
		}
		// Destinatario
		try {
			a = m.getRecipients(Message.RecipientType.TO);
		} catch (AddressException ae) {
			a = null;
		}
		if (a != null) {
			for (int i = 0; i < a.length; i++)
				this.addDestinatario(a[i].toString(), false);
		}
		// Conoscente
		try {
			a = m.getRecipients(Message.RecipientType.CC);
		} catch (AddressException ae) {
			a = null;
		}
		if (a != null) {
			for (int i = 0; i < a.length; i++)
				this.addCC(a[i].toString(), false);
		}
		// Conoscente riservato
		try {
			a = m.getRecipients(Message.RecipientType.BCC);
		} catch (AddressException ae) {
			a = null;
		}
		if (a != null) {
			for (int i = 0; i < a.length; i++)
				this.addBCC(a[i].toString(), false);
		}
		// Oggetto
		this.setOggetto(m.getSubject());
		// Flag
		this.flags = m.getFlags();
	}

	protected void recuperaMultipart(Part p) throws MailException {
		@SuppressWarnings("unused")
		int level = 0;

		try {
			if (p instanceof Message) {
				this.estraiDati((Message) p);
				primoGiro = true;
			}
			if (log.isTraceEnabled()) log.trace("CONTENT-TYPE: " + p.getContentType());

			// Recupera testo e allegati
			if (p.isMimeType("text/plain")) {
				if (primoGiro) {
					//  il testo della mail
					primoGiro = false; // MASSI - sostituire il null
					this.setTesto((String) p.getContent());

					if (log.isTraceEnabled()) log.trace("Recuperato testo della mail");
				} else {
					//  un allegato

					// if (video)
					// System.out.println(SDEBUG+": Recuperato testo tipo 'text/explain'");
					this.recuperaAllegato(p);
				}
			} else if (p.isMimeType("multipart/*")) {
				if (log.isTraceEnabled()) log.trace("Recuperato messaggio tipo 'multipart/*'");
				Multipart mp = (Multipart) p.getContent();
				level++;
				/* Recupera il sottocorpo */
				int count = 0;
				try {
					count = mp.getCount();
				} catch (Exception e) {
					count = -1;
				}
				// System.out.println("Numero mime: " + count) ;
				for (int i = 0; i < count; i++) {
					this.recuperaMultipart((Part) mp.getBodyPart(i));
				}
				level--;
			} else if (p.isMimeType("message/rfc822")) {
				if (log.isTraceEnabled()) log.trace("Recuperato messaggio tipo 'message/rfc822'");
				level++;
				this.recuperaMultipart((Part) p.getContent());
				level--;
			} else {
				if (log.isTraceEnabled()) log.trace("Recuperato messaggio tipo non conosciuto");
				level++;
				try {
					this.recuperaAllegato(p);
				} catch (Exception e) {
					if (log.getLogger().isTraceEnabled())
						log.warn("Errore - ??????????????????????????????????", e);
					// Salto per Answers
				}
				level--;
			}

		} catch (Exception e) {
			throw new MailException(e);
		}
	}

	private void recuperaAllegato(Part p) throws Exception {

		Allegato allegato = new Allegato(p.getDataHandler(), p.getFileName());
		// Attualmente sono gestite queste 2 sole categorie
		if ((p.getDisposition() != null) && (Part.ATTACHMENT.equals(p.getDisposition()))) {
			allegato.setTipo(Allegato.ATTACHMENT);
		} else {
			// Questa categoria comprende anche i CONTENT-TYPE text/html
			allegato.setTipo(Allegato.ALTRO);
		}

		this.addAllegato(allegato);
		/*
		 * if (video) { InputStream is =
		 * allegato.getDataHandler().getInputStream(); int c; while ((c =
		 * is.read()) != -1) System.out.write(c); System.out.println(); }
		 */
	}

	protected Message getMessage(Session session) throws MailException {
		Message msg = new MimeMessage(session);
		// Gli indirizzi sono sicuramente corretti perch il controllo viene
		// fatto a livello di Lettera
		String stato = "?";
		try {
			// Inserisco il mittente
			stato = "FROM";
			if (XString.isBlankNull(this.getMittente())) {
				throw new MailException("Manca un parametro fondamentale: " + stato);
			}
			msg.setFrom(new InternetAddress(this.getMittente()));

			// Inserisco il destinatario
			stato = "TO";
			if (XString.isBlankNull(this.getDestinatario())) {
				throw new MailException("Manca un parametro fondamentale: " + stato);
			}
			msg.setRecipients(Message.RecipientType.TO, InternetAddress.parse(this.getDestinatario(), false));

			// Aggiungo il conoscente
			stato = "CC";
			if (XString.isNotBlankNull(this.getCC()))
				msg.setRecipients(Message.RecipientType.CC, InternetAddress.parse(this.getCC(), false));

			// Aggiungo il conoscente riservato
			stato = "BCC";
			if (XString.isNotBlankNull(this.getBCC()))
				msg.setRecipients(Message.RecipientType.BCC, InternetAddress.parse(this.getBCC(), false));

			// Aggiungo l'indirizzo per il replay
			if (XString.isNotBlankNull(this.getReply()))
				msg.setReplyTo(InternetAddress.parse(this.getReply(), false));

			// Imposta l'oggetto
			stato = "SUBJECT";
			if (XString.isBlankNull(this.getOggetto())) {
				throw new MailException("Manca un parametro fondamentale: " + stato);
			}
			msg.setSubject(this.getOggetto());
		} catch (AddressException e) {
			// deve essere gestita...
			throw new MailException("Indirizzo non valido per " + stato, e);

		} catch (Exception e) {
			// deve essere gestita...
			throw new MailException("Eccezione non gestita per " + stato, e);
		}

		// Imposta il contenuto della mail
		stato = "TEXT";
		if (XString.isBlankNull(this.getTesto())) {
			throw new MailException("Manca un parametro fondamentale: " + stato);
		}

		try {
			this.setXMailer(SENDER);
			msg.setHeader("X-Mailer", this.getXMailer());

			msg.addHeader("X-Priority", this.getXPriority());
			msg.addHeader("X-MSMail-Priority", this.getXMSMailPriority());
			// msg.addHeader("Priority", this.getPriority());

			this.setSendDate(new Date());
			msg.setSentDate(this.getSendDate());

			if (this.numeroAllegati() > 0) {
				Multipart mp = new MimeMultipart();

				// Ho degli allegati - MULTIPART
				MimeBodyPart mpTesto = new MimeBodyPart();
				mpTesto.setText(this.getTesto());
				/*
				 * TODO da testare, sotto if (this.getContetType()!=null &&
				 * !this.getContetType().equals("")) {
				 * mpTesto.setContent(this.getTesto(),this.getContetType()); }
				 * else { mpTesto.setText(this.getTesto()); }
				 */

				mp.addBodyPart(mpTesto);
				msg.setText(this.getTesto());

				// ciclo su tutti gli allegati
				for (int i = 0; i < this.numeroAllegati(); i++) {
					MimeBodyPart mbAllegato = new MimeBodyPart();
					Allegato allegato = (Allegato) this.listaAllegati.get(i);
					mbAllegato.setDataHandler(allegato.getDataHandler());
					mbAllegato.setFileName(allegato.getNome());
					mp.addBodyPart(mbAllegato);
				}

				msg.setContent(mp);
			} else {
				// Non ho allegati
				if (XString.isNotBlankNull(this.getContetType())) {
					msg.setContent(this.getTesto(), this.getContetType());
				} else {
					msg.setText(this.getTesto());
				}
			}

			return msg;
		} catch (Exception e) {
			throw new MailException(e);
		}
	}

	public String toString() {
		String out = "from: " + this.getMittente() + "\n";
		out += "to: " + this.getDestinatario() + "\n";
		out += "cc: " + this.getCC() + "\n";
		out += "bcc: " + this.getBCC() + "\n";
		out += "data: " + (this.getSendDate() != null ? this.getSendDate().toString() : "sconosciuta") + "\n";
		out += "flag: ";
		boolean first = true;
		Flags.Flag[] sf = this.flags.getSystemFlags();
		for (int i = 0; i < sf.length; i++) {
			if (sf[i] == Flags.Flag.ANSWERED)
				out += "Answered";
			else if (sf[i] == Flags.Flag.DELETED)
				out += "Deleted";
			else if (sf[i] == Flags.Flag.DRAFT)
				out += "Draft";
			else if (sf[i] == Flags.Flag.FLAGGED)
				out += "Flagged";
			else if (sf[i] == Flags.Flag.RECENT)
				out += "Recent";
			else if (sf[i] == Flags.Flag.SEEN)
				out += "Seen";
			else
				continue; // skip it
			if (first)
				first = false;
			else
				out += ",";
		}
		out += "\n";
		out += "n. allegati: " + this.numeroAllegati() + "\n";
		out += "oggetto: " + this.getOggetto() + "\n";
		out += XString.fillerRight("-", "-", 60) + "\n";
		out += this.getTesto() + "\n";
		out += XString.fillerRight("-", "-", 60) + "\n";
		out += "x-mailer: " + this.getXMailer();

		return out;
	}

	public String getContetType() {
		return sContetType;
	}

	public void setContetType(String valore) {
		sContetType = valore;
	}

	public Object getTAG() {
		return tag;
	}

	public void setTAG(Object tag) {
		this.tag = tag;
	}

}