package it.softecspa.kahuna.mail12;

import it.softecspa.kahuna.lang.XString;

import java.util.Properties;

import javax.mail.FetchProfile;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.URLName;

public class Ricezione extends Servente {

	private String inbox = "INBOX";

	public Ricezione() {
		this("Ricezione");
	}

	public Ricezione(String nome) {
		super(nome);
	}

	public void setINBOX(String inbox) {
		this.inbox = inbox;
	}

	public String getINBOX() {
		return this.inbox;
	}

	/**
	 * Imposta i parametri per la connessione al server di posta
	 * 
	 * @param host
	 *             il nome del server
	 * @param user
	 *             l'utenza per la connessione
	 * @param password
	 *             la password
	 */
	public void setConfigurazione(String host, String user, String password) {
		super.setHost(host);
		super.setUser(user);
		super.setPassword(password);
	}

	private String verificaParametriFondamentali() {
		// Controllo della esistenza dei paramtri fondamentali

		if (this.getHost() == null && this.getUrl() == null) {
			return "Manca un parametro fondamentale: HOST o URL";
		}

		if (this.getProtocollo() == null) {
			return "Manca un parametro fondamentale: PROTOCOLLO";
		}

		if (this.getUser() == null || this.getPassword() == null) {
			return "Manca un parametro fondamentale: USER e PASSWORD";
		}

		return (String) null;
	}

	/**
	 * Riceve le email. Non viene generata alcuna eccezione. Eventuali errori di
	 * ricezione devono essere verificati con i metodi <I>errorState()</I> e
	 * <I>printErrorStack()</I>
	 */
	public void esegui(FlagEsegui flagRicezione) throws MailException {
		esegui(flagRicezione, false);
	}

	/**
	 * Riceve le email. Non viene generata alcuna eccezione. Eventuali errori di
	 * ricezione devono essere verificati con i metodi <I>errorState()</I> e
	 * <I>printErrorStack()</I>
	 * 
	 * @param boolean se settato a <B>true</B> ripulisce il buffer delle email
	 *        precedentemente scaricate
	 */
	public void esegui(FlagEsegui flagRicezione, boolean ripulisci) throws MailException {
		log.debug("Casella '" + super.getNome() + "', " + this.getProtocollo() + ", in ricezione...");

		String messaggio = verificaParametriFondamentali();
		if (messaggio != null) {
			throw new MailException(messaggio);
		}

		// Se richiesto, svuota la lista precedentemente caricata
		if (ripulisci) {
			try {
				if (log.isTraceEnabled()) log.trace("Svuota cartella 'Posta in Arrivo' precedentemente scaricata");
				this.removeAllLettera();

			} catch (Exception e) {
				throw new MailException(e);
			}
		}

		try {
			
			// Instaura la connessione verso il server di posta
			Store store = connectSession();
			
			// Apre la cartella di default del server di posta
			Folder folder = openDefaultFolder(store);
			
			
			int totalMessages = folder.getMessageCount();
			if (totalMessages == 0) {
				if (log.isTraceEnabled()) log.trace("Nessun messaggio da scaricare");
				folder.close(false);
				store.close();
				// ...interrompo il ciclo
			} else {

				if (log.isTraceEnabled()) {
					log.debug("Numero messaggi da scaricare: " + totalMessages);
					int newMessages = folder.getNewMessageCount();
					log.debug("Numero nuovi messaggi: " + newMessages);
				}
				
				Message[] msgs = recuperaMail(folder);
				for (int n = 0; n < msgs.length; n++) {
					log.debug(XString.fillerLeft("-", "-", 20) + " Messaggio n. " + (n + 1));

					try {
						// Marca le lettere per essere cancellate dal server */
						if (flagRicezione.isRemoveFromServer()) {
							msgs[n].setFlag(Flags.Flag.DELETED, true);
						}

						Lettera lettera = new Lettera();
						lettera.recuperaMultipart(msgs[n]);
						super.lista.add(lettera);

						if (log.isTraceEnabled()) {
							log.debug("*-----------------------------------------------------------*");
							log.debug(lettera.toString());
							log.debug("*-----------------------------------------------------------*");
						}
					} catch (Exception e) {
						throw new MailException(e);

					}
				}

				if (flagRicezione.isRemoveFromServer()) {
					/*
					 * Q: I want to delete messages on a POP3 server. I set the
					 * DELETED flag on those messages. Then I call the expunge()
					 * method, but I get a MethodNotSupportedException. How do I
					 * delete messages when I use Sun's POP3 provider? A: The
					 * expunge() method is not supported by the POP3 provider.
					 * Instead, close the folder with the expunge flag set to
					 * true. That is, invoke folder.close(true).
					 * folder.expunge();
					 */

					// Cancello la posta dal server
					folder.close(true);
					log.debug("Email eliminate dal server");
				} else {
					// Non cancello la posta dal server
					log.debug("Email non cancellate dal server");
					folder.close(false);
				}

				if (log.isTraceEnabled()) log.trace("Chiusura della connessione al server!");
				store.close();
			}

		} catch (Exception e) {
			// Per qualunque tipo di eccezione interrompo l'elaborazione e passo
			// alla buca lettere seguente
			throw new MailException(e);
		}
	}

	private Folder openDefaultFolder(Store store) throws MessagingException, MailException {
		// Apre le cartelle
		Folder folder = store.getDefaultFolder();

		if (folder == null) {
			throw new MailException("Non esiste alcuna cartella di default sul server");
		}

		folder = folder.getFolder(this.getINBOX());
		if (folder == null) {
			throw new MailException("La cartella '" + this.getINBOX() + " non esiste sul server");
		}

		try {
			folder.open(Folder.READ_WRITE);
			if (log.isTraceEnabled()) log.trace("Cartella aperta in LETTURA/SCRITTURA: " + this.getINBOX());

		} catch (MessagingException ex) {
			folder.open(Folder.READ_ONLY);
			log.warn("Cartella aperta in SOLA LETTURA: " + this.getINBOX());
		}
		
		return folder;
	}

	private Store connectSession() throws MessagingException {
		/*
		 * ...definisco la sessione per collegarmi al server e scaricare la
		 * posta
		 */
		Properties props = System.getProperties();
		Session session = Session.getDefaultInstance(props, null);
		setSessionDebugLevel(session);
		
		Store store = null;
		if (this.getUrl() != null) {
			URLName urln = new URLName(this.getUrl());
			store = session.getStore(urln);
			store.connect();
		} else {
			store = session.getStore(this.getProtocollo().toString());
		}

		// Effettua la connessione al server di posta
		store.connect(this.getHost(), this.getPorta(), this.getUser(), this.getPassword());

		return store;
	}

	
	private Message[] recuperaMail(Folder folder) throws MessagingException {
		// Attributes & Flags for all messages ...
		Message[] msgs = folder.getMessages();
		// Use a suitable FetchProfile
		FetchProfile fp = new FetchProfile();
		fp.add(FetchProfile.Item.ENVELOPE);
		fp.add(FetchProfile.Item.FLAGS);
		fp.add("X-Mailer");
		folder.fetch(msgs, fp);
		log.debug("Email recuperate: " + (msgs.length));
		
		return msgs;

	}
	
	
	/**
	 * Cancella tutte le vecchie mail dalla postale sul server senza scaricare
	 * l'eventuale posta.
	 * 
	 * @throws MailException
	 *             se viene riscontrato un errore.
	 */
	public void rimuoviPostaDalServer() throws MailException {
		this.rimuoviPostaDalServer(FlagRimuovi.all_email());
	}

	/**
	 * Svuota la casella postale sul server senza scaricare l'eventuale posta.
	 * 
	 * @param boolean se settato a <B>true</B> ripulisce l'intera cartella;
	 *        boolean se settato a <B>false</B> vengono eliminate soltanto le
	 *        email marcate DELETED.
	 * @throws MailException
	 *             se viene riscontrato un errore.
	 */
	public void rimuoviPostaDalServer(FlagRimuovi flagRimuovi) throws MailException {
		log.debug("Casella '" + super.getNome() + "', " + this.getProtocollo() + ", cancellazione...");

		String messaggio = verificaParametriFondamentali();
		if (messaggio != null) {
			throw new MailException(messaggio);
		}

		try {
			
			// Instaura la connessione verso il server di posta
			Store store = connectSession();
			
			// Apre la cartella di default del server di posta
			Folder folder = openDefaultFolder(store);
			
			
			int totalMessages = folder.getMessageCount();
			if (totalMessages == 0) {
				if (log.isTraceEnabled()) log.trace("Nessun messaggio da cancellare");
				folder.close(false);
				store.close();
				// ...interrompo il ciclo
			} else {

				if (log.isTraceEnabled()) {
					log.debug("Numero messaggi presenti: " + totalMessages);
					int newMessages = folder.getNewMessageCount();
					log.debug("Numero nuovi messaggi: " + newMessages);
				}
				
				Message[] msgs = recuperaMail(folder);
				
				int numero = msgs.length;
				if (flagRimuovi.isAllEmail()) {
					numero = msgs.length;
				} else if (flagRimuovi.isFirstEmail()) {

					// Cancello soltanto i primi n messaggi
					numero = flagRimuovi.getNumeroEmailDaCancellare();
					if (numero > msgs.length)
						numero = msgs.length;
				}
				log.debug("Email da cancellare: " + numero);

				for (int n = 0; n < numero; n++) {
					if (log.isTraceEnabled()) log.trace(XString.fillerLeft("-", "-", 20) + " Messaggio n. " + (n + 1));
					try {
						// Marca le lettere per essere cancellate dal server
						msgs[n].setFlag(Flags.Flag.DELETED, true);

					} catch (Exception e) {
						throw new MailException(e);
					}
				}

				/*
				 * Q: I want to delete messages on a POP3 server. I set the
				 * DELETED flag on those messages. Then I call the expunge()
				 * method, but I get a MethodNotSupportedException. How do I
				 * delete messages when I use Sun's POP3 provider? A: The
				 * expunge() method is not supported by the POP3 provider.
				 * Instead, close the folder with the expunge flag set to true.
				 * That is, invoke folder.close(true). folder.expunge();
				 */

				// Cancello la posta dal server
				folder.close(true);
				log.debug("Email eliminate dal server");

				if (log.isTraceEnabled()) log.trace("Chiusura della connessione al server!");
				store.close();
			}

		} catch (Exception e) {
			// Per qualunque tipo di eccezione interrompo l'elaborazione e passo
			// alla buca lettere seguente
			throw new MailException(e);
		}

	}

}