package it.softecspa.kahuna.net;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

public class SafeHttpURLConnection {

	private Logger log = Logger.getLogger(getClass());
	private int step_counter;
	
	private HttpURLConnection connection;
	private Map<String,String> buffer_properties;
	
	private boolean connected;
	private int httpCode;
	private String httpMessage;
	
	
	public SafeHttpURLConnection(URL url) throws IOException {
		this.connection = (HttpURLConnection) url.openConnection();
		buffer_properties = new HashMap<String, String>();
		this.connected = false;
	}
	
	private SafeHttpURLConnection(HttpURLConnection connection, Map<String,String> properties, int step) throws IOException {
		this.connection = connection;
		this.buffer_properties = properties;
		
		// Controllo anti loop
		this.step_counter = step+1;
		if (step_counter>5) throw new IOException("Too many redirect");
		this.connected = false;
	}
	
	

	public int write(byte[] stream) throws IOException {
		// Gestione manuale del redirect
		connection.setInstanceFollowRedirects(false);
		
		OutputStream output = null;
		try {
			if (stream!=null) {
				// Comunico il mio XML a server remoto
				output = connection.getOutputStream(); 
				output.write(stream);
				output.flush();
			}
			
			httpCode = connection.getResponseCode();
			httpMessage = connection.getResponseMessage();
			if (log.isDebugEnabled()) {
				log.debug("Remote call to url '"+connection.getURL()+"', response code "+httpCode);
			}
			
			if (httpCode==HttpURLConnection.HTTP_MOVED_PERM ||
				httpCode==HttpURLConnection.HTTP_MOVED_TEMP	) {
				URL new_url = new URL(connection.getHeaderField("Location"));
				log.warn("Http status code "+httpCode+", new url '"+new_url+"'");
				
				// Eseguo nuovamente la connessione
				try {
					if (output != null) output.close();
				} catch (IOException e) {
					log.warn("Exception closing output stream", e);
				}
				
				SafeHttpURLConnection redirect = clone(new_url);
				connection = redirect.getConnection();
				httpCode = redirect.write(stream);
			}
		
			connected = true;
			return httpCode;
		} finally {				
			try {
				if (output != null) output.close();
			} catch (IOException e) {
				log.warn("Exception closing output stream", e);
			}			
		}  		
	}
	
	
	public int connect() throws IOException {
		return write(null);
	}

	/**
     * Gets the status code from an HTTP response message.
     * For example, in the case of the following status lines:
     * <PRE>
     * HTTP/1.0 200 OK
     * HTTP/1.0 401 Unauthorized
     * </PRE>
     * It will return 200 and 401 respectively.
     * Returns -1 if no code can be discerned
     * from the response (i.e., the response is not valid HTTP).
     * @throws IOException if an error occurred connecting to the server.
     * @return the HTTP Status-Code, or -1
     */
	public int getResponseCode() throws IOException {
		if (httpCode==0) return connect();
		return httpCode;
	}
	
	/**
     * Gets the HTTP response message, if any, returned along with the
     * response code from a server.  From responses like:
     * <PRE>
     * HTTP/1.0 200 OK
     * HTTP/1.0 404 Not Found
     * </PRE>
     * Extracts the Strings "OK" and "Not Found" respectively.
     * Returns null if none could be discerned from the responses 
     * (the result was not valid HTTP).
     * @throws IOException if an error occurred connecting to the server.
     * @return the HTTP response message, or <code>null</code>
     */
	public String getResponseMessage() throws IOException {
		if (httpCode==0) connect();
		return httpMessage;
	}
	
	
	/**
	 * Clonazione della connessione e riapiertura verso il nuovo URL
	 * @param new_url
	 * @return
	 * @throws IOException
	 */
	private SafeHttpURLConnection clone(URL new_url) throws IOException {
		HttpURLConnection redirect = (HttpURLConnection) new_url.openConnection();
	
		// Eseguo la disconnessione della vecchia connessione (altrimenti non posso eseguire getRequestProperty)
		connection.disconnect();
				
		// Copia di tutti i parametri
		redirect.setRequestMethod(connection.getRequestMethod());
		redirect.setDoInput(connection.getDoInput());
		redirect.setDoOutput(connection.getDoOutput());		
		redirect.setUseCaches(connection.getUseCaches());	
		redirect.setConnectTimeout(connection.getConnectTimeout());
		redirect.setAllowUserInteraction(connection.getAllowUserInteraction());
		// ...e le properties
		for (String key : buffer_properties.keySet()) {
			redirect.setRequestProperty(key, buffer_properties.get(key));
		}
		
		// ...
		return new SafeHttpURLConnection(redirect, buffer_properties, step_counter);
	}


	public HttpURLConnection getConnection() {
		return connection;
	}

	public InputStream getInputStream() throws IOException {
		return connection.getInputStream();
	}
	
	public InputStream getErrorStream() {
		return connection.getErrorStream();
	}


	public void setRequestProperty(String key, String value) {
		connection.setRequestProperty(key, value);
		buffer_properties.put(key, value);
	}

	public void setRequestMethod(String method) throws ProtocolException {
		connection.setRequestMethod(method);		
	}

	public void setDoOutput(boolean dooutput) {
		connection.setDoOutput(dooutput);
	}
	
	public void setAllowUserInteraction(boolean allowuserinteraction) {
		connection.setAllowUserInteraction(allowuserinteraction);
	}
	
	public void setDoInput(boolean doinput) {
		connection.setDoInput(doinput);
	}

	public void setUseCaches(boolean usecache) {
		connection.setUseCaches(usecache);		
	}

	public String getHeaderField(String name) {
		return connection.getHeaderField(name);
	}
	
	public void disconnect() {
		connection.disconnect();
	}

	public boolean isConnected() {
		return connected;
	}

	public URL getURL() {
		return connection.getURL();
	}

	

	
	
	
}
