package it.softecspa.kahuna.services;

import it.softecspa.kahuna.log.KLogger;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.Socket;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

/**
 * Check to see if Tomcat is UP/DOWN.
 * 
 * This parses the server.xml file for the Tomcat admin port and see if we can
 * connect to it. If we can, then the Tomcat is UP otherwise it is DOWN
 * 
 * It is invoked as follows: java -Dcatalina.base=c:/tomcat-6.0.10
 * CatalinaShutdown
 * 
 * It can also (optionally) shutdown the Tomcat by adding the shutdown command
 * line parameter as follows:
 * 
 * java -Dcatalina.base=c:/tomcat-6.0.10 CatalinaShutdown shutdown
 * 
 * @author Shiraz Kanga <skanga at yahoo.com>
 */
public class CatalinaShutdown {
	
	KLogger log = KLogger.getLogger(getClass());
	
	/**
	 * Pathname to the server configuration file.
	 */
	protected String configFile = "conf/server.xml";
	protected String serverShutdown;
	protected int serverPort;

	
	public CatalinaShutdown() {
		Document configDom = getXmlDom(configFile());
		parseDocument(configDom);
	}
	
	
	public CatalinaShutdown(int porta, String shutdown) {
		serverShutdown = shutdown;
		serverPort = porta;
	}


	/**
	 * The application main program.
	 * 
	 * @param args
	 *            Command line arguments
	 */
	public static void main(String args[]) {
		
		CatalinaShutdown status = new CatalinaShutdown();
		// System.out.println ("Catalina.serverPort: " + serverPort);
		// System.out.println ("Catalina.serverShutdown: " + serverShutdown);

		status.doShutdown();
	}

	/**
	 * Return a File object representing our configuration file.
	 */
	protected File configFile() {
		File confFile = new File(configFile);
		if (!confFile.isAbsolute())
			confFile = new File(System.getProperty("catalina.base"), configFile);
		return (confFile);
	}

	/**
	 * Parses an XML file and returns a DOM document.
	 */
	public Document getXmlDom(File fileName) {
		try {
			// Create a builder factory
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

			// Create the builder and parse the file
			Document doc = factory.newDocumentBuilder().parse(fileName);
			return doc;
		} catch (SAXException e) {
			// A parsing error occurred; the xml input is not valid
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Extract the server port & shutdown command from the DOM
	 */
	private void parseDocument(Document configDom) {
		// get the root element which is Server Eg: <Server port="8005"
		// shutdown="SHUTDOWN">

		Element docEle = configDom.getDocumentElement();
		serverPort = new Integer(docEle.getAttribute("port")).intValue();
		serverShutdown = docEle.getAttribute("shutdown");
	}

	/**
	 * Send the shutdown command to the server
	 */
	public boolean doShutdown() {
		log.info("Catalina: port= " + serverPort +", shutdown = '" + serverShutdown+"'");		
	    
		// Stop the existing server
		Socket localSocket = null;
		try {
			localSocket = new Socket("localhost", serverPort);
			log.debug("Server status: UP");
			log.info("Tomcat shutdown initiated...");
			
			OutputStream outStream = null;
			try {
				outStream = localSocket.getOutputStream();
				for (int i = 0; i < serverShutdown.length(); i++) {
					outStream.write(serverShutdown.charAt(i));
				}
				
			} catch (IOException e) {
				log.error("I/O Exception during server shutdown.",e);
			} finally {
				if (outStream!=null) {
					outStream.flush();
					outStream.close();
				}
			}
			log.info("Tomcat shutdown completed!");
			return true;
			
		} catch (IOException e) {
			log.error("Error during server shutdown", e);			
		
		} finally {
			if (localSocket!=null) {
				try {
					localSocket.close();
				} catch (IOException e) {
					log.error("Error closing socket during server shutdown",e);
				}
			}
		}
		
		return false;
	}


	public String getServerShutdown() {
		return serverShutdown;
	}


	public int getServerPort() {
		return serverPort;
	}
}
