package it.softecspa.kahuna.springframework;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.log.KLogger;
import it.softecspa.kahuna.log.Ticket;
import it.softecspa.kahuna.util.Properties;

import javax.naming.NamingException;

import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.jmx.export.annotation.ManagedAttribute;
import org.springframework.jmx.export.annotation.ManagedOperation;
import org.springframework.jmx.export.annotation.ManagedResource;

@ManagedResource(objectName="KahunaJMX:name=SpringEngine",description="Gestione KSpringEngine")
public class KSpringEngine extends SpringContext {	
	
	// logger standard
	KLogger log = KLogger.getLogger(getClass().getSimpleName());;
	
	private static String location = "classpath*:kahuna.spring.xml";
	private static String bootmanager = "bootmanager";
	
	private AbstractApplicationContext appContext;	
	private static KSpringEngine instance;
	private static String uniqueId;
	
	private boolean bNoCommit;
	
	
	// Hibernate session factory
	private SessionFactory sessionFactory;
	
	
	// Kahuna properties file (XML)
	private Properties config;
	
	
	private KSpringEngine() {
		super();	
	}
	
	
	public static KSpringEngine initInstance() {
		if (instance == null) {
		 	synchronized (KSpringEngine.class) {
				// Utilizza il file di configurazione standard di SPRING
		 		KSpringEngine engine = new KSpringEngine();  
		 		engine.loadSpring();
		 		engine = null;
			}
		}
		return instance;
	}
	
	
	
	
	public static KSpringEngine initInstance(String confFile) {
		return initInstance(confFile, Integer.toString(Ticket.generaTicket()));
	}
	
	
	public static KSpringEngine initInstance(String confFile, String uniqueId) {
		if (instance == null) {
			synchronized (KSpringEngine.class) {
				KSpringEngine.setConfFile(confFile);
				KSpringEngine.setUniqueId(uniqueId);
				KSpringEngine engine = new KSpringEngine();  
				engine.loadSpring();
				engine = null;
			}
		}
		return instance;
	}
	
	
	
	public static KSpringEngine getInstance() {
		/* <MASSI>
		 * La sincronizzazione si  resa necessaria in ambiante multithread
		 * in quanto il porcesso di inizializzazione dura abbastanza a lungo
		 * e la variabile "instance" non risulta subito consitente per tutti i thread
		 * In questo modo la variabile  disponibile soltanto al termine del ciclo di load
		 */
		synchronized (KSpringEngine.class) {
			return instance;
		}
	}
	
	
	
	
	
	public static SpringContext getInitialContext() throws NamingException {
		if (instance == null)
			throw new NamingException("No Context Exist");
		
		return instance;
	}
	
	
	
	
	private synchronized void loadSpring() {
	    try {
	    	log.info("* Boostrapping application and loading Spring Context\n");
	    	if (appContext!=null){
	  			//TODO da sistemare
	  			
	  			appContext.refresh();
	  		} else {
	  			//URL urlo =ClassLoader.getSystemResource("kserver.spring.xml");
	  			// Sono una classe esterna quindi fo il BOOT e popolo la mia ref del context
	  			log.info("Loading Spring Context using file '"+location+"'");
	  			
	  			/*
	  			 * Devo utilizzare il metodo di ricerca dei file assoluto
	  			 * altrimenti non riesco a referenziare il file di configurazione di LOG4J
	  			 * ed il file di properties dell'appliczione
	  			 * 
	  			 * http://note19.com/2007/08/01/spring-configuration-with-property-files/
	  			 */
	  			
	  			/*
	  			 * Verificare dove vanno a finire i file di LOG
	  			 * Rendere relativi a cartella
	  			 */
	  			
	  			appContext = new ClassPathXmlApplicationContext(new String[] {location});
	  			
	  			log.info("Getting bean instances of "+this.getClass().getSimpleName());
	  			// Popolo l'appContext della istanza di Spring
	  			instance = (KSpringEngine)appContext.getBean(bootmanager);
	  			instance.setAppContext(appContext);
	  				  			
	  			log.info("* Engine started!");	  						
	  		}
	    	
	    	
	    } catch (Throwable e) {
	    	appContext=null;
	    	log.error(e.toString(), e);
	    }
    }


	
	public static String getConfFile() {
		return location;
	}

	public static void setConfFile(String location) {
		KSpringEngine.location = location;
	}
	
	
	
	public static String getBeanId() {
		return bootmanager;
	}

	public static void setBeanId(String bootmanager) {
		KSpringEngine.bootmanager = bootmanager;
	}
	

	@Override
	public AbstractApplicationContext getAppContext() {
		return appContext;
	}
	
	public void setAppContext(AbstractApplicationContext appContext) {
		this.appContext = appContext;
	}

	
	/** @deprecated */
	public SessionFactory getSessionFactory() {
		return sessionFactory;
	}
	
	
	/**
	 * Restituisce una sessione valida al database
	 * @return
	 */
	public Session openSession() {
		if (sessionFactory==null) return null;
		Session session = sessionFactory.openSession();
		
		if (session==null) {
			log.warn("Acquired null factory session, retry later!");
			return null;
		}
		
		// Applico il FLUSH MODE ad ALWAYS per le sessioni
		session.setFlushMode(FlushMode.ALWAYS);
		return session;
	}
	

	/**
	 * Metodo utilizzato da Spring
	 * @param sessionFactory
	 */
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}

	//ManagedAttribute(description="Parametri di configurazione")
	public Properties getConfig() {
		return config;
	}


	public void setConfig(Properties config) {
		this.config = config;
		
		
		// Setup delle variabili di sistema....
		log.info("Setup of configuration inner system variables");
		
		// NO-COMMIT
		bNoCommit = getConfig().getBoolean("no-commit",false);
		
		
		
	}


	/**
	 * Controlla la variabile di configureazione "no-commit"
	 * @return
	 */	
	@ManagedOperation(description="Restituisce True se il sistema  in modali NO-COMMIT")
	public boolean checkNoCommit() {
		return bNoCommit;
	}
	
	
	@ManagedOperation(description="Mette il sistema in modalit COMMIT")
	public void attivaCommit() {
		bNoCommit = false;
	}
	
	@ManagedOperation(description="Mette il sistema in modalit NO-COMMIT")
	public void noCommitMode() {
		bNoCommit = true;
	}
	
	
	@ManagedAttribute(description="Restituisce True se il sistema  in modali NO-COMMIT")
	public boolean isNoCommit() {
		return bNoCommit;
	}


	public static String getUniqueId() {
		return uniqueId;
	}


	public static void setUniqueId(String instanceId) {
		KSpringEngine.uniqueId = (XString.isNotBlankNullTrim(instanceId)?instanceId:null);
	}
	
	/*
	try {
	System.out.println("Starting spring init");
	ClassLoader classLoader = service.getClassLoader();
	ClassPathXmlApplicationContext appCtx = new
	ClassPathXmlApplicationContext(new String[] {
	"classpath:/framework/cache/baseCacheContext.xml",
	"applicationContext.xml",
	"cacheContext.xml"
	}, false);
	appCtx.setClassLoader(classLoader);
	appCtx.refresh();
	System.out.println("spring loaded");
	} catch (Exception ex) {
	ex.printStackTrace();
	}
*/
	
	
}
