package it.softecspa.kahuna.sql;

/**
 * Title:        Package lib<p>
 * Description:  Classe di ausilio alla scrittura di comandi SQL<P>
 * Copyright:    Copyright (c) .<p>
 * Company:      infogroup
 * @author Massimiliano Veroni
 * @version 1.0
 */

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.util.List;
import java.util.StringTokenizer;

public abstract class AbstractSql {

	String sSQL;

	String sTables;
	String sWhere;
	String sValues;
	String sColumns;
	String sOptions;

	protected String sDateDelimitator = "'";
	private String sStringDelimitator = "'";

	private final String CR = "\r\n";

	AbstractSql() {
		sSQL = "";

		sTables = "";
		sWhere = "";
		sValues = "";
		sColumns = "";
		sOptions = "";
	}

	AbstractSql(AbstractSql origine) {
		sSQL = new String(origine.sSQL);

		sTables = new String(origine.sTables);
		sWhere = new String(origine.sWhere);
		sValues = new String(origine.sValues);
		sColumns = new String(origine.sColumns);
		sOptions = new String(origine.sOptions);
	}

	/*
	 * public void reset() { sSQL="";
	 * 
	 * sTables=""; sWhere=""; sValues=""; sColumns=""; sOptions=""; }
	 */
	public abstract void reset();

	public void resetTable(String elenco) {
		sTables = "";
	}

	/**
	 * Valorizza la lista delle tabelle con con cui comporre la query.
	 * 
	 * @param elenco
	 *            lista delle tabelle.
	 * @see AbstractSql#getTables() getTables
	 */
	public void setTables(String elenco) {
		sTables = elenco.trim();
	}

	public void addTable(String tabella) {
		if (XString.isBlankNullTrim(tabella))
			return;
		tabella = tabella.trim();
		sTables += (sTables != null && !sTables.equals("") ? (tabella.toUpperCase().indexOf("JOIN ") >= 0 ? " " : ", ") : "") + tabella;
	}

	/**
	 * Restituisce l'elenco delle tabelle utilizzate nella composizione della query.
	 * 
	 * @return una stringa con l'elenco delle tabelle utilizzate.
	 * @see AbstractSql#setTables(String) setTables
	 */
	public String getTables() {
		return sTables;
	}

	/**
	 * Valorizza la lista delle colonne con con cui comporre l'UPDATE.
	 * 
	 * @param colonna
	 *            lista delle colonne.
	 * @param valore
	 *            valore da asseganre al SET. Questi valori sono utilizzati per comporre la <B>SET</B>.
	 * @see AbstractSql#getColumns() getColumns
	 */
	public void addColumnSET(String colonna, String valore) {
		if (XString.isBlankNullTrim(colonna))
			return;
		sColumns += (XString.isNotBlankNull(sColumns) ? ", " : "") + colonna + " = " + valore;
	}

	public void setColumnSET(String colonna, String valore) {
		resetColumnsSET();
		addColumnSET(colonna, valore);
	}

	public void resetColumnsSET() {
		sColumns = "";
		sValues = "";
	}

	/**
	 * Aggiunge un valore alla lista delle colonne con con cui comporre la SELECT o l'UPDATE.
	 * 
	 * @param elenco
	 *            lista dei valori. In caso di UPDATE questi valori sono utilizzati per comporre la <B>SET</B>.
	 * @see AbstractSql#getColumns() getColumns
	 */
	public void setColumns(String elenco) {
		sColumns = elenco.trim();
	}

	/**
	 * Valorizza la lista delle colonne con con cui comporre la SELECT.
	 * 
	 * @param colonna
	 *            lista delle colonne. In caso di UPDATE questi valori sono utilizzati per comporre la <B>SET</B>.
	 * @see AbstractSql#getColumns() getColumns
	 */
	public void addColumn(String colonna) {
		if (XString.isBlankNullTrim(colonna))
			return;
		sColumns += (XString.isNotBlankNull(sColumns) ? ", " : "") + colonna;
	}

	public void setColumn(String colonna) {
		sColumns = "";
		addColumn(colonna);
	}

	public void resetColumns() {
		sColumns = "";
	}

	/**
	 * Restituisce l'elenco delle colonne utilizzate nella composizione della SELECT o dell'UPDATE.
	 * 
	 * @return una stringa con l'elenco delle colonne utilizzate.
	 * @see AbstractSql#setColumns(String) setColumns
	 */
	public String getColumns() {
		return sColumns;
	}

	/**
	 * Valorizza la lista dei valori passati al comando di INSERT.
	 * 
	 * @param elenco
	 *            lista dei valori. I valori devono essere separati dalla <I>virgola</I>.
	 * @see AbstractSql#getValues() getValues
	 */
	public void setValues(String elenco) {
		sValues = elenco.trim();
	}

	/**
	 * Aggiunge un valore alla lista dei valori passati al comando di INSERT.
	 * 
	 * @param valore
	 *            il valore passato.
	 * @see AbstractSql#getValues() getValues
	 */
	public void addValue(String valore) {
		sValues += (XString.isNotBlankNull(sValues) ? ", " : "") + (valore != null ? valore.trim() : "null");
	}

	public void addValue(String colonna, String valore) {
		if (XString.isBlankNullTrim(colonna))
			return;
		sColumns += (XString.isNotBlankNull(sColumns) ? ", " : "") + colonna.trim();
		sValues += (XString.isNotBlankNull(sValues) ? ", " : "") + (valore != null ? valore.trim() : "null");
	}

	public void setValue(String valore) {
		resetValues();
		addValue(valore);
	}

	public void setValue(String colonna, String valore) {
		resetColumns();
		resetValues();
		addValue(colonna, valore);
	}

	public void resetValues() {
		sValues = "";
	}

	/**
	 * Restituisce l'elenco dei valori passati al comando di INSERT.
	 * 
	 * @return una stringa con l'elenco delle colonne utilizzate. I valori ritornati sono separati dalla <I>virgola</I>.
	 * @see AbstractSql#setValues(String) setValues
	 */
	public String getValues() {
		return sValues.trim();
	}

	/**
	 * Valorizza la condizioni di WHERE della query.
	 * 
	 * @param elenco
	 *            lista delle condizioni
	 * @see AbstractSql#getWhere() getWhere.
	 */
	public void setWhere(String elenco) {
		sWhere = elenco.trim();
		// ...per evitare problemi
		if (sWhere.toLowerCase().startsWith("and"))
			sWhere = sWhere.substring(3);
	}

	/**
	 * Aggiunge una condizione WHERE alla query
	 * 
	 * @param valore
	 */
	public void addWhere(String valore) {
		if (valore == null)
			return;
		sWhere += (sWhere != null && !sWhere.equals("") ? " and " : "") + (valore != null ? valore.trim() : "null");
	}

	public void resetWhere() {
		sWhere = "";
	}

	/**
	 * Restituisce l'elenco delle condizioni di WHERE della query.
	 * 
	 * @return una stringa con l'intera condizione inserita.
	 * @see AbstractSql#setWhere(String) setWhere
	 */
	public String getWhere() {
		return sWhere;
	}

	/**
	 * Valorizza l'elenco delle eventuali opzioni da inserire nella query.
	 * 
	 * @param elenco
	 *            elenco dei valori
	 * @see AbstractSql#getOptions() getOptions
	 */
	public void setOptions(String elenco) {
		sOptions = elenco.trim();
	}

	/**
	 * Aggiunge una condizione di option alla query
	 * 
	 * @param valore
	 */
	public void addOption(String valore) {
		if (XString.isBlankNullTrim(valore))
			return;
		if (sOptions == null)
			sOptions = "";
		sOptions += (sOptions.length()>0?" ":"") + valore;
	}

	public void resetOptions() {
		sOptions = "";
	}

	/**
	 * Restituisce l'elenco delle opzioni inserite nella query.
	 * 
	 * @return una stringa con l'intera condizione inserita.
	 * @see AbstractSql#setOptions(String) setOptions
	 */
	public String getOptions() {
		return sOptions;
	}

	/**
	 * Imposta il delimitatore utilizzazto nella formattazione delle stringhe nelle query.
	 * 
	 * @param valore
	 *            carettere delimitatore delle stringhe. Per default  impostato il carattere <B>'</B>
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public void setStringDelimitator(String valore) {
		sStringDelimitator = valore;
	}

	/**
	 * Restituisce il delimitatore utilizzato nella formattazzione delle stringhe nella query.
	 * 
	 * @return il carattere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 */
	public String getStringDelimitator() {
		return sStringDelimitator;
	}

	/**
	 * Imposta il delimitatore utilizzazto nella formattazione delle date nelle query.
	 * 
	 * @param valore
	 *            carettere delimitatore delle date. Per default  impostato il carattere <B>'</B>
	 * @see AbstractSql#getDateDelimitator() getDateDelimitator
	 */
	public void setDateDelimitator(String valore) {
		sDateDelimitator = valore;
	}

	/**
	 * Restituisce il delimitatore utilizzato nella formattazzione delle date nella query.
	 * 
	 * @return il carattere di default.
	 * @see AbstractSql#setDateDelimitator(String) setDateDelimitator
	 */
	public String getDateDelimitator() {
		return sDateDelimitator;
	}

	/**
	 * Restituisce l'ultima query costruita.
	 * 
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#setTables(String) setTables
	 * @see AbstractSql#setColumns(String) setColumns
	 * @see AbstractSql#setValues(String) setValues
	 * @see AbstractSql#setWhere(String) setWhere
	 * @see AbstractSql#setOptions(String) setOptions
	 * @see AbstractSql#costruisciSelect() costruisciSelect
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String getSql() {
		return sSQL;
	}

	protected void setSql(String sSQL) {
		this.sSQL = sSQL;
	}

	/**
	 * Costruisce la SELECT a partire dai dati inseriti in precedenza.
	 * 
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#setTables(String) setTables
	 * @see AbstractSql#setColumns(String) setColumns
	 * @see AbstractSql#setWhere(String) setWhere
	 * @see AbstractSql#setOptions(String) setOptions
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciSelect() {
		String s;

		s = "select " + sColumns + CR;
		s = s + "from " + sTables + CR;
		if (!sWhere.equals(""))
			s = s + "where " + sWhere + CR;
		if (!sOptions.equals(""))
			s = s + sOptions + CR;
		sSQL = s;
		return s;
	}

	/**
	 * Costruisce la SELECT impostando tutti i parametri.
	 * 
	 * @param colonne
	 *            lista delle colonne.
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param where
	 *            condizione.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciSelect(String colonne, String tabelle, String where, String opzioni) {
		this.setValues("");
		this.setColumns(colonne);
		this.setTables(tabelle);
		this.setWhere(where);
		this.setOptions(opzioni);
		return this.costruisciSelect();
	}

	/**
	 * Costruisce la INSERT a partire dai dati inseriti in precedenza.
	 * 
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#setTables(String) setTables
	 * @see AbstractSql#setColumns(String) setColumns
	 * @see AbstractSql#setValues(String) setValues
	 * @see AbstractSql#setOptions(String) setOptions
	 * @see AbstractSql#costruisciSelect() costruisciSelect
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciInsert() {
		String s;

		s = "insert into " + sTables + CR;
		if (!sColumns.equals(""))
			s += "(" + sColumns + ")" + CR;
		if (!sValues.equals(""))
			s += "values(" + sValues + ")" + CR;
		if (!sOptions.equals(""))
			s = s + sOptions + CR;
		sSQL = s;
		return s;
	}

	/**
	 * Costruisce la INSERT impostando tutti i parametri.
	 * 
	 * @param colonne
	 *            lista delle colonne.
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param valori
	 *            lista dei valori da inserire.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciInsert(String tabelle, String colonne, String valori, String opzioni) {
		this.setWhere("");
		this.setColumns(colonne);
		this.setValues(valori);
		this.setTables(tabelle);
		this.setOptions(opzioni);
		return this.costruisciInsert();
	}

	/**
	 * Costruisce la UPDATE a partire dai dati inseriti in precedenza.
	 * 
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#setTables(String) setTables
	 * @see AbstractSql#setColumns(String) setColumns
	 * @see AbstractSql#setWhere(String) setWhere
	 * @see AbstractSql#setOptions(String) setOptions
	 * @see AbstractSql#costruisciSelect() costruisciSelect
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciUpdate() {
		String s;

		s = "update " + sTables + CR;
		s = s + "set " + sColumns + CR;
		if (!sWhere.equals("")) {
			s = s + "where " + sWhere + CR;
		}
		if (!sOptions.equals("")) {
			s = s + sOptions + CR;
		}
		sSQL = s;
		return s;
	}

	/**
	 * Costruisce la UPDATE impostando tutti i parametri.
	 * 
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param colonne
	 *            lista delle colonne.
	 * @param where
	 *            condizione.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciUpdate(String tabelle, String colonne, String where, String opzioni) {
		this.setValues("");
		this.setColumns(colonne);
		this.setTables(tabelle);
		this.setWhere(where);
		this.setOptions(opzioni);
		return this.costruisciUpdate();
	}

	/**
	 * Costruisce la DELETE a partire dai dati inseriti in precedenza.
	 * 
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#setTables(String) setTables
	 * @see AbstractSql#setWhere(String) setWhere
	 * @see AbstractSql#setOptions(String) setOptions
	 * @see AbstractSql#costruisciSelect() costruisciSelect
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 */
	public String costruisciDelete() {
		String s;

		s = "delete" + CR;
		s += "from " + sTables + CR;
		if (!sWhere.equals("")) {
			s += "where " + sWhere + CR;
		}
		if (!sOptions.equals("")) {
			s += sOptions + CR;
		}
		sSQL = s;
		return s;
	}

	/**
	 * Costruisce la DELETE impostando tutti i parametri.
	 * 
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param where
	 *            condizione.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return una stringa contenente la query costruita.
	 * @see AbstractSql#getSql() getSql
	 * @see AbstractSql#costruisciInsert() costruisciInsert
	 * @see AbstractSql#costruisciUpdate() costruisciUpdate
	 * @see AbstractSql#costruisciDelete() costruisciDelete
	 */
	public String costruisciDelete(String tabelle, String where, String opzioni) {
		this.setColumns("");
		this.setValues("");
		this.setTables(tabelle);
		this.setWhere(where);
		this.setOptions(opzioni);
		return this.costruisciDelete();
	}

	/**
	 * Sostituisce il carattere apice (') con ('') per l'inserimento nel database
	 */
	protected String apice(String s) {
		int i = 0;
		while (s.indexOf("'", i) >= 0) {
			// System.out.println(s + " - " + i + " - " + s.length());
			i = s.indexOf("'", i) + 2;
			s = s.substring(0, i - 2) + "'" + s.substring(i - 2);
			if (i >= s.length())
				return s;
		}
		return s;
	}

	/**
	 * Sostituisce il carattere slash (\) con (\\) per l'inserimento nel database
	 */
	protected String slash(String s) {
		int i = 0;
		while (s.indexOf("\\", i) >= 0) {
			// System.out.println(s + " - " + i + " - " + s.length());
			i = s.indexOf("\\", i) + 2;
			s = s.substring(0, i - 2) + "\\" + s.substring(i - 2);
			if (i >= s.length())
				return s;
		}
		return s;
	}

	/**
	 * Formatta la stringa per essere inserita nella query utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             una <I>stringa</I>.
	 * @return la stringa formattata e delimitata dal carettere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public String fStr(String valore) {
		return (valore == null ? "null" : sStringDelimitator + slash(apice(valore)) + sStringDelimitator);
	}
	

	/**
	 * Trasforma l'array di elementi in una lista divisa da , (virgola)
	 * @param valori
	 * @return
	 */
	public String fStr(String[] valori) {
		String condizione = "";
		for (int i = 0; i < valori.length; i++) {
			if (valori[i] != null) {
				condizione += (condizione != "" ? "," : "") + sStringDelimitator + slash(apice(valori[i])) + sStringDelimitator;
			}
		}
		return condizione;
	}

	
	/**
	 * Trasforma la List di elementi in una lista divisa da , (virgola)
	 * @param valori
	 * @return
	 */
	public String fStr(List<String> valori) {
		if (valori.isEmpty()) return "null";
		String condizione = "";
		for (int i = 0; i < valori.size(); i++) {
			if (valori.get(i) != null) {
				condizione += (condizione != "" ? "," : "") + sStringDelimitator + slash(apice(valori.get(i))) + sStringDelimitator;
			}
		}
		return condizione;
	}
	
	public String fStr(String valore, String separator) {
		String s = "";
		for (StringTokenizer t = new StringTokenizer(valore, separator); t.hasMoreElements();) {
			s += (s != "" ? separator : "") + (valore == null ? "null" : sStringDelimitator + slash(apice((String) t.nextElement())) + sStringDelimitator);
		}
		return s;
	}

	/**
	 * Formatta il valore per essere inserito nella query come stringa utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             un <I>intero</I>.
	 * @return la stringa formattata e delimitata dal carettere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public String fStr(int valore) {
		return sStringDelimitator + Integer.toString(valore) + sStringDelimitator;
	}

	/**
	 * Formatta il valore per essere inserito nella query come stringa utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             un <I>long</I>.
	 * @return la stringa formattata e delimitata dal carettere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public String fStr(long valore) {
		return sStringDelimitator + Long.toString(valore) + sStringDelimitator;
	}

	/**
	 * Formatta il valore per essere inserito nella query come stringa utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             un <I>double</I>.
	 * @return la stringa formattata e delimitata dal carettere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public String fStr(double valore) {
		return sStringDelimitator + Double.toString(valore) + sStringDelimitator;
	}

	/**
	 * Formatta il valore per essere inserito nella query come stringa utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             un <I>float</I>.
	 * @return la stringa formattata e delimitata dal carettere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public String fStr(float valore) {
		return sStringDelimitator + Float.toString(valore) + sStringDelimitator;
	}

	/**
	 * Formatta il valore per essere inserito nella query come stringa utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             un <I>byte</I>.
	 * @return la stringa formattata e delimitata dal carettere di default.
	 * @see AbstractSql#setStringDelimitator(String) setStringDelimitator
	 * @see AbstractSql#getStringDelimitator() getStringDelimitator
	 */
	public String fStr(byte valore) {
		return sStringDelimitator + Byte.toString(valore) + sStringDelimitator;
	}

	public String fStr(XString valore) {
		return fStr(valore.toString());
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             una <I>stringa</I>.
	 * @return una stringa.
	 */
	public String fNum(String valore) {
		return (valore == null ? "null" : valore);
	}

	
	/**
	 * Trasforma l'array di elementi in una lista divisa da , (virgola)
	 * @param valori
	 * @return
	 */
	public String fNum(String[] valori) {
		String condizione = "";
		for (int i = 0; i < valori.length; i++) {
			if (XString.isNotBlankNullTrim(valori[i])) {
				condizione += (condizione != "" ? "," : "") + valori[i];
			}
		}
		return condizione;
	}
	
	public String fNum(List<Integer> valori) {
		if (valori.isEmpty()) return "null";
		String condizione = "";
		for (int i = 0; i < valori.size(); i++) {
			if (valori.get(i) != null) {
				condizione += (condizione != "" ? "," : "") + valori.get(i);
			}
		}
		return condizione;
	}

	
	/**
	 * Trasforma l'array di elementi in una lista divisa da , (virgola)
	 * @param valori
	 * @return
	 */
	public String fNum(Integer[] valori) {
		String condizione = "";
		for (int i = 0; i < valori.length; i++) {
			if (valori[i] != null) {
				condizione += (condizione != "" ? "," : "") + valori[i];
			}
		}
		return condizione;
	}
	
	
	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>BigDecimal</I>.
	 * @return una stringa.
	 */
	public String fNum(java.math.BigDecimal valore) {
		return (valore == null ? "null" : valore.toString());
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>intero</I>.
	 * @return una stringa.
	 */
	public String fNum(int valore) {
		return Integer.toString(valore);
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>long</I>.
	 * @return una stringa.
	 */
	public String fNum(long valore) {
		return Long.toString(valore);
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>double</I>.
	 * @return una stringa.
	 */
	public String fNum(double valore) {
		return Double.toString(valore);
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>float</I>.
	 * @return una stringa.
	 */
	public String fNum(float valore) {
		return Float.toString(valore);
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>byte</I>.
	 * @return una stringa.
	 */
	public String fNum(byte valore) {
		return Byte.toString(valore);
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>byte</I>.
	 * @return una stringa.
	 */
	public String fNum(short valore) {
		return Short.toString(valore);
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>Integer</I>.
	 * @return una stringa.
	 */
	public String fNum(Integer valore) {
		return (valore != null ? valore.toString() : "null");
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>Long</I>.
	 * @return una stringa.
	 */
	public String fNum(Long valore) {
		return (valore != null ? valore.toString() : "null");
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>Byte</I>.
	 * @return una stringa.
	 */
	public String fNum(Byte valore) {
		return (valore != null ? valore.toString() : "null");
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero.
	 * 
	 * @param valore
	 *             un <I>Short</I>.
	 * @return una stringa.
	 */
	public String fNum(Short valore) {
		return (valore != null ? valore.toString() : "null");
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero decimale
	 * 
	 * @param valore
	 *             un <I>Double</I>.
	 * @return una stringa.
	 */
	public String fNum(Double valore) {
		return (valore != null ? "" + valore.toString() : "null");
	}

	/**
	 * Formatta il valore per essere inserito nella query come numero
	 * 
	 * @param valore
	 *             un <I>Object</I>.
	 * @return una stringa.
	 */
	public String fNum(Object valore) {
		return (valore != null ? "" + valore.toString() : "null");
	}

	public String fNum(XString valore) {
		return fNum(valore.toString());
	}

	
	
	
	
	
	
	
	
	
	/**
	 * Formatta la stringa per essere inserita nella query utilizzando il delimitatore impostato.
	 * 
	 * @param valore
	 *             una variabile di tipo <I>EnterpriseCalendar</I>.
	 * @param formato
	 *             una variabile di tipo <I>String</I>.
	 * @return la stringa formattata e delimitata dal carattere di default.
	 * @see AbstractSql#setDateDelimitator(String) setDateDelimitator
	 * @see AbstractSql#getDateDelimitator() getDateDelimitator
	 */
	public String fData(EnterpriseCalendar valore, String formato) {
		return (valore == null ? "null" : sDateDelimitator + valore.format(formato) + sDateDelimitator);
	}

	public String fData(String valore) {
		return (valore == null ? "null" : sDateDelimitator + valore + sDateDelimitator);
	}

	public String fData(XString valore) {
		return fData(valore.toString());
	}

}
