package it.softecspa.kahuna.sql;
/**                     
 * DbConnectionBroker.   
 * @version 1.0.13 3/12/02 
 * @author Marc A. Mnich 
 */                     
 
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.log.NLog;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.log4j.Logger;
 
 
/** 
 * DbConnectionBroker 
 * A servlet-based broker for database connections. 
 * Creates and manages a pool of database connections. 
 * @version 1.0.13 3/12/02 
 * @author Marc A. Mnich 
 */ 
public class DbConnectionPool implements Runnable { 
  private Thread runner; 
     
  private Connection[] connPool; 
  private int[] connStatus; 
 
  private long[] connLockTime;
  private long[] connCreateDate; 
  private String[] connID; 
  private String dbDriver;
  private String dbServer;
  private String dbLogin;
  private String dbPassword; 
  @SuppressWarnings("unused")
  private int currConnections, connLast, minConns, maxConns, maxConnMSec, maxCheckoutSeconds; 
 
  //available: set to false on destroy, checked by getConnection() 
  private boolean available=true; 
 
  private SQLWarning currSQLWarning; 
  private String pid; 
  private String pidName;
 
  private NLog logDB;  
  
  @SuppressWarnings("unused")
  private final int DEFAULTMAXCHECKOUTSECONDS=60; 
  @SuppressWarnings("unused")
  private final int DEFAULTDEBUGLEVEL=2; 
       
  /** 
   * Creates a new Connection Broker<br> 
   * dbDriver:            JDBC driver. e.g. 'oracle.jdbc.driver.OracleDriver'<br> 
   * dbServer:            JDBC connect string. e.g. 'jdbc:oracle:thin:@203.92.21.109:1526:orcl'<br> 
   * dbLogin:             Database login name.  e.g. 'Scott'<br> 
   * dbPassword:          Database password.    e.g. 'Tiger'<br> 
   * minConns:            Minimum number of connections to start with.<br> 
   * maxConns:            Maximum number of connections in dynamic pool.<br> 
   * maxConnTime:         Time in days between connection resets. (Reset does a basic cleanup)<br> 
   * maxCheckoutSeconds:  Max time a connection can be checked out before being recycled. Zero value turns option off, default is 60 seconds. 
   * debugLevel:          Level of debug messages output to the log file.  0 -> no messages, 1 -> Errors, 2 -> Warnings, 3 -> Information
   * logger:              Logger of Log4j<br>     
   */  
  public DbConnectionPool(String dbDriver, String dbServer, String dbLogin, String dbPassword, int minConns, int maxConns, double maxConnTime, int maxCheckoutSeconds, String pidPath, String pidName, Logger logger) throws Exception { 
    setupBroker(dbDriver, dbServer, dbLogin, dbPassword, minConns, maxConns, maxConnTime, maxCheckoutSeconds, pidPath, pidName, logger); 
  } 
 
  
  private synchronized void setupBroker(String dbDriver, String dbServer, String dbLogin, String dbPassword, int minConns, int maxConns, double maxConnTime, int maxCheckoutSeconds, String pidPath, String pidName, Logger logger) throws Exception { 
    this.available=true; // aggiunto in data 11/07/2008
    this.logDB = new NLog(logger);
    
    this.connPool = new Connection[maxConns]; 
    this.connStatus = new int[maxConns]; 
    this.connLockTime = new long[maxConns]; 
    this.connCreateDate = new long[maxConns]; 
    this.connID = new String[maxConns]; 
    this.currConnections = minConns; 
   
    this.maxConns = maxConns; 
    this.dbDriver = dbDriver; 
    this.dbServer = dbServer; 
    this.dbLogin = dbLogin; 
    this.dbPassword = dbPassword;
     
    this.maxCheckoutSeconds = maxCheckoutSeconds;
    this.maxConnMSec = (int)(maxConnTime * 86400000.0);  //86400 sec/day 
    if (this.maxConnMSec < 30000) {  // Recycle no less than 30 seconds. 
      this.maxConnMSec = 30000; 
    } 
 
    // Write the pid file (used to clean up dead/broken connection)    
    pidPath = (pidPath!=null?it.softecspa.kahuna.io.File.rationalizePath(pidPath):"./");
    pidName = pidPath + (XString.isNotBlankNull(pidName)?pidName:this.getClass().getName());
    this.pidName = pidName + (pidName.endsWith(".pid")?"":".pid");    
    creaPID();

    
    if (logDB==null) {      
      logDB.info("Nessun debugger configurato, utilizzato System.out");
    }
    logDB.info("-----------------------------------------"); 
    logDB.info("Starting DbConnectionPool 1.0.1 (derived from DbConnectionBroker 1.0.13):"); 
    logDB.info("driver = " + dbDriver); 
    logDB.info("server = " + dbServer); 
    logDB.info("login = " + dbLogin);
    logDB.info("min connections = " + minConns); 
    logDB.info("max connections = " + maxConns); 
    logDB.info("total refresh interval = " + maxConnTime + " days");
    logDB.info("max checkout seconds = " + maxCheckoutSeconds);
    logDB.info("-----------------------------------------"); 
    
    
    // Initialize the pool of connections with the mininum connections: 
    // Problems creating connections may be caused during reboot when the 
    //    servlet is started before the database is ready.  Handle this 
    //    by waiting and trying again.  The loop allows 5 minutes for  
    //    db reboot. 
    boolean connectionsSucceeded=false; 
    int dbLoop=20; 
    
    try { 
      for(int i=1; i < dbLoop; i++) { 
        try { 
          for(int j=0; j < currConnections; j++) {  
            createConn(j); 
          } 
          connectionsSucceeded=true; 
          break; 
        } catch (SQLException e) {
          logDB.error("--->Attempt (" + String.valueOf(i) + " of " + String.valueOf(dbLoop) +") failed to create new connections set at startup: "); 
          logDB.error("    Will try again in 15 seconds...",e); 
          try { 
            Thread.sleep(15000);
          } catch(InterruptedException e1) {} 
        } 
      } 
      
      if (!connectionsSucceeded) { // All attempts at connecting to db exhausted 
        logDB.fatal("All attempts at connecting to Database exhausted", null);        
        throw new Exception("All attempts at connecting to Database exhausted"); 
      } 
    } catch (Exception e) {  
      throw e; 
    } 
      
    // Fire up the background housekeeping thread
    runner = new Thread(this); 
    runner.start();
  } //End DbConnectionBroker() 

  
  
  private void creaPID (){
    BufferedWriter pidout = null;
    try {
      SimpleDateFormat formatter = new SimpleDateFormat ("dd/MM/yyyy G 'at' hh:mm:ss a zzz"); 
      pid = formatter.format(new Date());
      pidout = new BufferedWriter(new FileWriter(pidName)); 
      pidout.write(pid); 
      
      logDB.info("File for pid info is created: " + pidName);
    } catch (IOException e) {
      logDB.error("Can't write the file for pid info: " + pidName, e);
    } finally {
      try {
        if (pidout!=null) pidout.close();
      } catch(Exception e) {
        logDB.error("Error closing file for pid info: " + pidName, e);
      }
    }
  }
  
  
  
  /** 
   * Housekeeping thread.  Runs in the background with low CPU overhead. 
   * Connections are checked for warnings and closure and are periodically 
   * restarted. 
   * This thread is a catchall for corrupted 
   * connections and prevents the buildup of open cursors. (Open cursors 
   * result when the application fails to close a Statement). 
   * This method acts as fault tolerance for bad connection/statement programming. 
   */ 
  public void run() { 
    boolean forever = true; 
    Statement stmt=null; 
    //String currCatalog=null; 
    long maxCheckoutMillis = maxCheckoutSeconds * 1000; 
  
  
    while(forever) {               
      // Make sure the log file is the one this instance opened 
      // If not, clean it up! 
      BufferedReader in = null;
      try { 
        in = new BufferedReader(new FileReader(pidName)); 
        String curr_pid = in.readLine(); 
        if (curr_pid!=null && curr_pid.equalsIgnoreCase(pid)) { 
          //log.println("They match = " + curr_pid); 
        } else { 
          if (logDB!=null) logDB.error("No match pid: " + curr_pid); 
                  
          // Close all connections silently - they are definitely dead. 
          for(int i=0; i < currConnections; i++) { 
            try { 
              connPool[i].close(); 
            } catch (SQLException e) {} // ignore 
          } 
          // Returning from the run() method kills the thread 
          return; 
        }               
        if (in!=null) in.close(); 
              
      } catch (IOException e) { 
        logDB.error("Can't read the file for pid info: " + pidName, e);
        
        // ...provo a ricreare il file PID
        creaPID();
      } finally {
        try {
          if (in!=null) in.close();          
        } catch (IOException e) {
          logDB.error("Error closing file for pid info: " + pidName, e);
        }          
      }
          
          
      // Get any Warnings on connections and print to event file 
      for(int i=0; i < currConnections; i++) {             
        try {  
          currSQLWarning = connPool[i].getWarnings();  
          if (currSQLWarning != null) { 
            logDB.warn("Warnings on connection " + String.valueOf(i) + " " + currSQLWarning); 
            connPool[i].clearWarnings(); 
          } 
        } catch(SQLException e) { 
          logDB.error("Cannot access Warnings",e);
        }     
      } 
          
      for(int i=0; i < currConnections; i++) { // Do for each connection 
        long age = System.currentTimeMillis() - connCreateDate[i]; 
              
        try {  // Test the connection with createStatement call 
          synchronized(connStatus) { 
            if(connStatus[i] > 0) { // In use, catch it next time! 
                          
              // Check the time it's been checked out and recycle 
              long timeInUse = System.currentTimeMillis() - connLockTime[i];                         
              logDB.debug("Connection " + i + " in use for " + timeInUse + " ms");              
              if(maxCheckoutMillis != 0) { 
                if(timeInUse > maxCheckoutMillis) { 
                  logDB.warn("Connection " + i + " failed to be returned in time.  Recycling...");
                  throw new SQLException(); 
                } 
              }                          
              continue; 
            } 
          
            connStatus[i] = 2; // Take offline (2 indicates housekeeping lock) 
          } 
        
          if(age > maxConnMSec) {  // Force a reset at the max conn time 
            throw new SQLException(); 
          } 
                  
          stmt = connPool[i].createStatement(); 
          connStatus[i] = 0;  // Connection is O.K. 
          //log.println("Connection confirmed for conn = " + String.valueOf(i)); 
                  
          // Some DBs return an object even if DB is shut down 
          if(connPool[i].isClosed()) { 
            throw new SQLException(); 
          } 

                  
          // Connection has a problem, restart it 
        } catch(SQLException e) { 

          logDB.info("Recycling connection " + String.valueOf(i)); 
          try {                        
            connPool[i].close();  
          } catch(SQLException eSql) { 
            logDB.error("Can't close connection! Might have been closed already.  Trying to recycle anyway...",eSql);             
          } 
  
          try { 
            createConn(i); 
          } catch(SQLException eSql) { 
            logDB.error("Failed to create connection",eSql);
            connStatus[i] = 0;  // Can't open, try again next time 
          } 
        } finally { 
          try{if(stmt != null) {stmt.close();}} catch(SQLException e1){}; 
        } 
      } 
          
      try { 
        Thread.sleep(20000);  // Wait 20 seconds for next cycle 
      } catch(InterruptedException e) { 
        // Returning from the run method sets the internal  
        // flag referenced by Thread.isAlive() to false. 
        // This is required because we don't use stop() to  
        // shutdown this thread. 
        return; 
      }   
      
    } 
      
  } // End run 
      
  /** 
   * This method hands out the connections in round-robin order. 
   * This prevents a faulty connection from locking 
   * up an application entirely.  A browser 'refresh' will 
   * get the next connection while the faulty 
   * connection is cleaned up by the housekeeping thread. 
   *  
   * If the min number of threads are ever exhausted, new 
   * threads are added up the the max thread count. 
   * Finally, if all threads are in use, this method waits 
   * 2 seconds and tries again, up to ten times.  After that, it 
   * returns a null. 
   */ 
  public Connection getConnection() {
    Connection conn=null; 
  
    if (available) { 
      boolean gotOne = false; 
              
      for(int outerloop=1; outerloop<=10; outerloop++) { 
        try { 
          int loop=0; 
          int roundRobin = connLast + 1; 
          if(roundRobin >= currConnections) roundRobin=0; 
               
          do { 
            synchronized(connStatus) { 
              if((connStatus[roundRobin] < 1) && 
                 (! connPool[roundRobin].isClosed())) { 
                conn = connPool[roundRobin]; 
                connStatus[roundRobin]=1; 
                connLockTime[roundRobin] = System.currentTimeMillis(); 
                connLast = roundRobin; 
                gotOne = true; 
                break; 
              } else { 
                loop++; 
                roundRobin++; 
                if(roundRobin >= currConnections) roundRobin=0; 
              } 
            } 
          } while((gotOne==false)&&(loop < currConnections)); 
                
        } catch (SQLException e) { 
          logDB.error("getConnection()",e); 
        } 
        
        if(gotOne) { 
          break; 
        } else { 
          synchronized(this) {  // Add new connections to the pool 
            if(currConnections < maxConns) { 
              try { 
                createConn(currConnections); 
                currConnections++; 
              } catch(SQLException e) { 
                logDB.error("Unable to create new connection",e);
              } 
            } 
          } 
                    
          try { 
            Thread.sleep(2000); 
          } catch(InterruptedException e) {} 
          logDB.fatal("Connections Exhausted!  Will wait and try again in loop " + String.valueOf(outerloop),null);
        } 
            
      } // End of try 10 times loop 
        
    } else { 
      logDB.error("Unsuccessful getConnection() request during destroy()",null);
    } // End if(available)     
    
    logDB.debug("Handing out connection " + idOfConnection(conn));
    return conn;
  } 
      
  /** 
   * Returns the local JDBC ID for a connection. 
   */ 
  public int idOfConnection(Connection conn) { 
    int match; 
    String tag; 
      
    try { 
      tag = conn.toString(); 
    } catch (NullPointerException e1) { 
      tag = "none"; 
    } 
        
    match=-1; 
        
    for(int i=0; i< currConnections; i++) { 
      if(connID[i].equals(tag)) { 
        match = i; 
        break; 
      } 
    } 
    return match; 
  } 
      
  /** 
   * Frees a connection.  Replaces connection back into the main pool for 
   * reuse. 
   */ 
  public String freeConnection(Connection conn) { 
    String res=""; 
    
    int thisconn = idOfConnection(conn); 
    if (thisconn >= 0) { 
      connStatus[thisconn]=0;
      logDB.debug("Freed connection " + String.valueOf(thisconn) + " normal exit");       
    } else { 
      logDB.error("Could not free connection!");
    } 
      
    return res;
  } 
      
  
  /** 
   * Returns the age of a connection -- the time since it was handed out to 
   * an application. 
   */ 
  public long getAge(Connection conn) { // Returns the age of the connection in millisec. 
    int thisconn = idOfConnection(conn); 
    return System.currentTimeMillis() - connLockTime[thisconn]; 
  } 
  
  
  private void createConn(int i) throws SQLException {
    try { 
       Class.forName (dbDriver); 
       connPool[i] = DriverManager.getConnection(dbServer,dbLogin,dbPassword); 
       
       /* Massi - aggiunto per cambio SESSIOn su Oracle */ 
       if (connPool[i]!=null) { 
         // ...imposto il formato della data di default, per la sessione, ma solo per ORACLE
         if (SqlDB.isOracle(connPool[i])) {
           SqlDB.alterSessionORACLE_DATE_FORMAT(connPool[i]);
           //db_logger.livelloDebug("Execute alter session: NLS_DATE_FORMAT = '"+SqlDB.ORACLE_DATE_FORMAT+"'");      
         }
       }
       /* ...questo mi fa fare anche un controllo sulla validit della query */
       
       connStatus[i]=0; 
       connID[i]=connPool[i].toString(); 
       connLockTime[i]=0; 
       connCreateDate[i] =  (new Date()).getTime();
       
       
    } catch (ClassNotFoundException e) { 
      logDB.error("Error creating connection",e);
    }
    
    logDB.info("Opening connection " + String.valueOf(i) + " " + connPool[i].toString()); 
  } 
   
  
  /** 
   * Shuts down the housekeeping thread and closes all connections  
   * in the pool. Call this method from the destroy() method of the servlet. 
   */ 
  
  /** 
   * Multi-phase shutdown.  having following sequence: 
   * <OL> 
   * <LI><code>getConnection()</code> will refuse to return connections. 
   * <LI>The housekeeping thread is shut down.<br> 
   *    Up to the time of <code>millis</code> milliseconds after shutdown of 
   *    the housekeeping thread, <code>freeConnection()</code> can still be 
   *    called to return used connections. 
   * <LI>After <code>millis</code> milliseconds after the shutdown of the 
   *    housekeeping thread, all connections in the pool are closed. 
   * <LI>If any connections were in use while being closed then a 
   *    <code>SQLException</code> is thrown. 
   * <LI>The log is closed. 
   * </OL><br> 
   * Call this method from a servlet destroy() method. 
   * 
   * @param      millis   the time to wait in milliseconds. 
   * @exception  SQLException if connections were in use after  
   * <code>millis</code>. 
   */ 
  public void destroy(int millis) throws SQLException { 
    // Checking for invalid negative arguments is not necessary, 
    // Thread.join() does this already in runner.join(). 

    // Stop issuing connections 
    available=false; 

    // Shut down the background housekeeping thread 
    runner.interrupt(); 

    // Wait until the housekeeping thread has died. 
    try { runner.join(millis); }  
    catch(InterruptedException e){} // ignore  
    
    // The housekeeping thread could still be running 
    // (e.g. if millis is too small). This case is ignored. 
    // At worst, this method will throw an exception with the  
    // clear indication that the timeout was too short. 

    long startTime=System.currentTimeMillis(); 

    // Wait for freeConnection() to return any connections 
    // that are still used at this time. 
    int useCount; 
    while((useCount=getUseCount())>0 && System.currentTimeMillis() - startTime <=  millis) { 
      try { Thread.sleep(500); } 
      catch(InterruptedException e) {} // ignore  
    } 

    // Close all connections, whether safe or not 
    for(int i=0; i < currConnections; i++) { 
      try { 
        connPool[i].close(); 
      } catch (SQLException e) { 
        logDB.error("Cannot close connections["+i+"] on Destroy", e);
      } 
    } 

    if(useCount > 0) { 
      //bt-test successful 
      String msg="Unsafe shutdown: Had to close "+useCount+" active DB connections after "+millis+"ms"; 
      logDB.info(msg);
      // Throwing following Exception is essential because servlet authors 
      // are likely to have their own error logging requirements. 
      throw new SQLException(msg); 
    }

  } //End destroy() 
  
  
  /** 
   * Less safe shutdown.  Uses default timeout value. 
   * This method simply calls the <code>destroy()</code> method  
   * with a <code>millis</code> 
   * value of 10000 (10 seconds) and ignores <code>SQLException</code>  
   * thrown by that method. 
   * @see     #destroy(int) 
   */ 
  public void destroy() {  
    try { 
      destroy(10000); 
    } 
    catch(SQLException e) {} 
  } 
  
  
  
  /** 
   * Returns the number of connections in use. 
   */ 
  // This method could be reduced to return a counter that is 
  // maintained by all methods that update connStatus. 
  // However, it is more efficient to do it this way because: 
  // Updating the counter would put an additional burden on the most 
  // frequently used methods; in comparison, this method is 
  // rarely used (although essential). 
  public int getUseCount() { 
    int useCount=0; 
    synchronized(connStatus) { 
      for(int i=0; i < currConnections; i++) { 
        if(connStatus[i] > 0) { // In use 
          useCount++; 
        } 
      } 
    } 
    return useCount;
    
  }//End getUseCount() 
  
  /** 
   * Returns the number of connections in the dynamic pool. 
   */ 
  public int getSize() { 
    return currConnections; 
  }//End getSize() 


  
  
  
  public String getUrl() {
    return dbServer;
  }

  
  public String getDriver() {
    return dbDriver;
  }
  
  
  
  
  
  
}// End class 
