package it.softecspa.kahuna.sql;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

public class Sql extends AbstractSql {

	private static Logger log = Logger.getLogger("it.softecspa.kahuna.SQL");

	// Connessione al database
	private Connection connection;

	// Statement creato per le query sql
	private Statement res_stmt;
	private Integer resultSetType;
	private Integer resultSetConcurrency;
	private Integer resultSetHoldability;

	private ResultSet res;

	// Numero di record aggiornati nell'ultima operazione di executeUpdate()
	private int updateCount;

	// Elenco delle variabili da passare come Preparedstatement
	private List<Object> bind_list;

	
	

	private Sql() {
		super();
	}

	/**
	 * Crea l'oggetto e lo associa ad una istanza <I>Connection</I>.
	 */
	public Sql(Connection connection) throws SQLException {
		this();
		this.connection = connection;
	}

	/**
	 * Crea l'oggetto e lo associa ad una istanza <I>Connection</I> Creates a
	 * <code>Statement</code> object that will generate <code>ResultSet</code>
	 * objects with the given type, concurrency, and holdability. This method is
	 * the same as the <code>createStatement</code> method above, but it allows
	 * the default result set type, concurrency, and holdability to be
	 * overridden.
	 * 
	 * @param resultSetType
	 *            one of the following <code>ResultSet</code> constants:
	 *            <code>ResultSet.TYPE_FORWARD_ONLY</code>,
	 *            <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or
	 *            <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
	 * @param resultSetConcurrency
	 *            one of the following <code>ResultSet</code> constants:
	 *            <code>ResultSet.CONCUR_READ_ONLY</code> or
	 *            <code>ResultSet.CONCUR_UPDATABLE</code>
	 * @param resultSetHoldability
	 *            one of the following <code>ResultSet</code> constants:
	 *            <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> or
	 *            <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code>
	 * @exception SQLException
	 *                if a database access error occurs or the given parameters
	 *                are not <code>ResultSet</code> constants indicating type,
	 *                concurrency, and holdability
	 * @see ResultSet
	 * @since 1.4
	 */
	public Sql(Connection connection, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
		this();
		this.connection = connection;
		// ...
		this.resultSetType = new Integer(resultSetType);
		this.resultSetConcurrency = new Integer(resultSetConcurrency);
		this.resultSetHoldability = new Integer(resultSetHoldability);

	}

	/**
	 * Crea l'oggetto a partire da un'altra istanza SqlPlus.
	 */
	public Sql(Sql sql) {
		this();
		this.connection = sql.getConnection();
	}

	public void reset() {
		sSQL = "";

		sTables = "";
		sWhere = "";
		sValues = "";
		sColumns = "";
		sOptions = "";

		bind_list = null;
	}

	protected void finalize() throws Throwable {
		try {
			if (connection != null && !connection.isClosed() && res_stmt != null) {
				res_stmt.close();
			}
		} catch (Exception e) {
			log.warn("Problema in chiusura statement", e);
		}
		super.finalize();
	}

	/**
	 * Compone ed esegue la SELECT utilizzando i parametri impostati in
	 * precedenza. Il puntatore dei risultati  inizialmente uguale a
	 * <I>null</I>.
	 * 
	 * @return l'istanza ResulSet.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeQuery(String)
	 *      java.sql.Statement.executeQuery(String)
	 */
	public ResultSet executeSelect() throws SQLException {
		super.costruisciSelect();
		return executeSelect(super.getSql());
	}

	public ResultSet executeSelect(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
		super.costruisciSelect();
		return executeSelect(super.getSql(), resultSetType, resultSetConcurrency, resultSetHoldability);
	}

	/**
	 * Compone ed esegue la SELECT utilizzando i parametri inseriti. Il
	 * puntatore dei risultati  inizialmente uguale a <I>null</I>.
	 * 
	 * @param colonne
	 *            lista delle colonne.
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param where
	 *            condizione.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return l'istanza ResulRet.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeQuery(String)
	 *      java.sql.Statement.executeQuery(String)
	 */
	public ResultSet executeSelect(String colonne, String tabelle, String where, String opzioni) throws SQLException {
		super.costruisciSelect(colonne, tabelle, where, opzioni);
		return executeSelect(super.getSql());
	}

	/**
	 * Esegue la SELECT utilizzando la stringa SQL. Il puntatore dei risultati 
	 * inizialmente uguale a <I>null</I>. Creates a <code>Statement</code>
	 * object that will generate <code>ResultSet</code> objects with the given
	 * type, concurrency, and holdability. This method is the same as the
	 * <code>createStatement</code> method above, but it allows the default
	 * result set type, concurrency, and holdability to be overridden.
	 * 
	 * @param resultSetType
	 *            one of the following <code>ResultSet</code> constants:
	 *            <code>ResultSet.TYPE_FORWARD_ONLY</code>,
	 *            <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or
	 *            <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
	 * @param resultSetConcurrency
	 *            one of the following <code>ResultSet</code> constants:
	 *            <code>ResultSet.CONCUR_READ_ONLY</code> or
	 *            <code>ResultSet.CONCUR_UPDATABLE</code>
	 * @param resultSetHoldability
	 *            one of the following <code>ResultSet</code> constants:
	 *            <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> or
	 *            <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code>
	 * @exception SQLException
	 *                if a database access error occurs or the given parameters
	 *                are not <code>ResultSet</code> constants indicating type,
	 *                concurrency, and holdability
	 * @see ResultSet
	 * @since 1.4
	 * @param sql
	 *            la query string da eseguire.
	 * @return l'istanza ResulSet.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeQuery(String)
	 *      java.sql.Statement.executeQuery(String)
	 */
	public ResultSet executeSelect(String query) throws SQLException {
		super.setSql(query);
		boolean printed = stampaSQL(query, null);

		try {
			if (res != null)
				res.close();
		} catch (SQLException e) {
			log.warn("close()", e);
		}

		try {
			// ...utilizzo i valori di default per la connessione
			if (res_stmt == null) res_stmt = createStatement();
			res = res_stmt.executeQuery(query);

		} catch (SQLException e) {
			if (!printed) stampaSQL(query, e);
			throw e;
		}

		return res;
	}

	public ResultSet executeSelect(String query, int resultSetType) throws SQLException {
		return executeSelect(query, resultSetType, getDefaultResultSetConcurrency());
	}

	public ResultSet executeSelect(String query, int resultSetType, int resultSetConcurrency) throws SQLException {
		return executeSelect(query, resultSetType, resultSetConcurrency, getDefaultResultSetHoldability());
	}

	public ResultSet executeSelect(String query, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
		super.setSql(query);
		boolean printed = stampaSQL(query, null);

		try {
			if (res != null)
				res.close();
		} catch (SQLException e) {
			log.warn("close()", e);
		}

		try {
			if (res_stmt != null) {
				// ..forzo la creazione di un nuovo statemnt con le
				// caratterisiche diverse dal default
				res_stmt.close();
				res_stmt = null;
			}
			res_stmt = createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
			res = res_stmt.executeQuery(query);

		} catch (SQLException e) {
			if (!printed) stampaSQL(query, e);
			throw e;
		}

		return res;
	}

	/**
	 * Funzione privata per l'apertura dello statemnt E' fatta una gestione
	 * dell'errore per la retrocompatibilit con alcune versione dei driver
	 * Oracle Vengono utilizzati i valori di default
	 * 
	 * @return
	 * @throws SQLException
	 */
	private Statement createStatement() throws SQLException {
		if (resultSetType == null)
			resultSetType = new Integer(getDefaultResultSetType());
		if (resultSetConcurrency == null)
			resultSetConcurrency = new Integer(getDefaultResultSetConcurrency());
		if (resultSetHoldability == null)
			resultSetHoldability = new Integer(getDefaultResultSetHoldability());

		return createStatement(resultSetType.intValue(), resultSetConcurrency.intValue(), resultSetHoldability.intValue());
	}

	/**
	 * Funzione privata per l'apertura dello statemnt E' fatta una gestione
	 * dell'errore per la retrocompatibilit con alcune versione dei driver
	 * Oracle
	 * 
	 * @param resultSetType
	 * @param resultSetConcurrency
	 * @param resultSetHoldability
	 * @return Statement
	 * @throws SQLException
	 */
	private Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
		try {
			return connection.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);

		} catch (SQLException e) {
			/*
			 * Nota: gestione particolare legata alla versione dei driver JDBC
			 * per Oracle riscontrata in alcune installazioni Weblogic 8.1 sp 2
			 * Vedi EDS per Eniservizi (Progetto Vipra/DWit)
			 * 
			 * Non  supportata la creazione di uno statement specificando
			 * "resultSetHoldability"
			 */
			if (isOracle()) {
				// ORA-17023: Unsupported feature
				if (e.getErrorCode() == 17023) {
					return connection.createStatement(resultSetType, resultSetConcurrency);
				}
			}
			throw e;
		}
	}

	/**
	 * Ritorna il risultato corrente di un oggetto <I>ResultSet</I>.
	 * 
	 * @return il risultato corrente, <I>null</I> se non ci sono risultati o
	 *         l'operazione non  una SELECT.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#getResultSet() java.sql.Statement.getResultSet()
	 */
	public ResultSet getResultSet() throws SQLException {
		if (res == null)
			return null;
		return res;
	}

	public int getResultSetType() throws SQLException {
		if (res_stmt == null)
			return getDefaultResultSetType();
		return res_stmt.getResultSetType();
	}

	protected static int getDefaultResultSetType() throws SQLException {
		return ResultSet.TYPE_FORWARD_ONLY;
	}

	public int getResultSetConcurrency() throws SQLException {
		if (res_stmt == null)
			return getDefaultResultSetConcurrency();
		return res_stmt.getResultSetConcurrency();
	}

	protected static int getDefaultResultSetConcurrency() throws SQLException {
		return ResultSet.CONCUR_READ_ONLY;
	}

	protected int getResultSetHoldability() throws SQLException {
		if (res_stmt == null)
			return getDefaultResultSetHoldability();
		return res_stmt.getResultSetHoldability();
	}

	protected static int getDefaultResultSetHoldability() throws SQLException {
		return ResultSet.HOLD_CURSORS_OVER_COMMIT;
	}

	/**
	 * Compone ed esegue un comando UPDATE utilizzando i parametri impostati in
	 * precedenza.
	 * 
	 * @return il numero di righe modificate.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeUpdate() throws SQLException {
		super.costruisciUpdate();
		return executeUpdate(super.getSql());
	}

	/**
	 * Compone ed esegue un comando UPDATE utilizzando i parametri inseriti.
	 * 
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param colonne
	 *            lista delle colonne.
	 * @param where
	 *            condizione.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return il numero di righe inserite.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeUpdate(String tabelle, String colonne, String where, String opzioni) throws SQLException {
		super.costruisciUpdate(tabelle, colonne, where, opzioni);
		return executeUpdate(super.getSql());
	}

	/**
	 * Compone ed esegue un comando UPDATE utilizzando i parametri inseriti.
	 * 
	 * @param sql
	 *            istruzione UPDATE.
	 * @return il numero di righe inserite.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeUpdate(String update) throws SQLException {
		super.setSql(update);
		return executeSTATEMENT(super.getSql());
	}

	/**
	 * Compone ed esegue un comando INSERT utilizzando i parametri impostati in
	 * precedenza.
	 * 
	 * @return il numero di righe inserite.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeInsert() throws SQLException {
		super.costruisciInsert();
		return executeInsert(super.getSql());
	}

	/**
	 * Compone ed esegue un comando INSERT utilizzando i parametri inseriti.
	 * 
	 * @param colonne
	 *            lista delle colonne.
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param valori
	 *            lista dei valori da inserire.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return il numero di righe inserite.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeInsert(String tabelle, String colonne, String valori, String opzioni) throws SQLException {
		super.costruisciInsert(tabelle, colonne, valori, opzioni);
		return executeInsert(super.getSql());
	}

	/**
	 * Esegue un comando INSERT utilizzando i parametri impostati in precedenza.
	 * 
	 * @return il numero di righe inserite.
	 * @param sql
	 *            istruzione INSERT.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeInsert(String insert) throws SQLException {
		super.setSql(insert);
		return executeSTATEMENT(super.getSql());
	}

	/**
	 * Compone ed esegue un comando DELETE utilizzando i parametri impostati in
	 * precedenza.
	 * 
	 * @return il numero di righe cancellate.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeDelete() throws SQLException {
		super.costruisciDelete();
		return executeDelete(super.getSql());
	}

	/**
	 * Compone ed esegue un comando DELETE utilizzando i parametri inseriti.
	 * 
	 * @param tabelle
	 *            lista delle tabelle.
	 * @param where
	 *            condizione.
	 * @param opzioni
	 *            elenco di condizioni aggiuntive.
	 * @return il numero di righe inserite.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeDelete(String tabelle, String where, String opzioni) throws SQLException {
		super.costruisciDelete(tabelle, where, opzioni);
		return executeDelete(super.getSql());
	}

	/**
	 * Compone ed esegue un comando DELETE utilizzando i parametri inseriti.
	 * 
	 * @param sql
	 *            istruzione DELETE.
	 * @return il numero di righe inserite.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public int executeDelete(String delete) throws SQLException {
		super.setSql(delete);
		return executeSTATEMENT(super.getSql());
	}

	private int executeSTATEMENT(String sql) throws SQLException {
		return executeSTATEMENT(sql, false);
	}

	private int executeSTATEMENT(String sql, boolean forcePreparedStatement) throws SQLException {
		boolean usePreparedStatement = (this.getBindList().size() > 0) || forcePreparedStatement;
		boolean printed = stampaSQL(sql, null);

		Statement st = null;
		PreparedStatement pst = null;
		updateCount = 0;
		try {
			if (usePreparedStatement) {
				pst = connection.prepareStatement(sql);

				// ...ciclo di valorizzazione di parametri
				List<?> lista = this.getBindList();
				for (int i = 0; i < lista.size();) {
					Object bind = lista.get(i);
					i++;

					if (String.class.isInstance(bind)) {
						pst.setString(i, (String) bind);

					} else if (IClob.class.isInstance(bind)) {
						IClob clob = (IClob) bind;
						pst.setCharacterStream(i, clob.getReader(), clob.length());

					} else if (IBlob.class.isInstance(bind)) {
						IBlob blob = (IBlob) bind;
						try {
							FileInputStream fis = blob.getReader();
							pst.setBinaryStream(i, fis, fis.available());
							// pst.setBinaryStream(i, blob.getReader(),
							// blob.length());
						} catch (FileNotFoundException e) {
							new SQLException(e.toString());
						} catch (IOException e) {
							new SQLException(e.toString());
						}

					} else {
						pst.setObject(i, bind);
						log.warn("Tipologia di bind non gestita: " + bind.getClass().getName(), null);
					}
				}

				updateCount = pst.executeUpdate();

			} else {
				// Utilizzo un "createStatement"
				st = createStatement();
				updateCount = st.executeUpdate(sql);
			}

			/*
			 * -----------------------------------------------------------------
			 * // Operazioni speciali nascoste (vedi BLOB) for (int i=0;
			 * i<this.getHiddenList().size(); i++) { Object hidden =
			 * getHiddenList().get(i);
			 * 
			 * if (hidden instanceof IBlobDeprecated) { IBlobDeprecated blob =
			 * (IBlobDeprecated)hidden; blob.updateEmptyBlob(this,
			 * this.getTables());
			 * 
			 * } else {
			 * livelloWarning("Tipologia di 'hidden' non gestita: "+hidden
			 * .getClass().getName(),null); } } //
			 * -----------------------------------------------------------------
			 */

		} catch (SQLException e) {
			if (!printed) stampaSQL(sql, e);
			throw e;

		} finally {
			bind_list = null;

			if (st != null)
				st.close();
			if (pst != null)
				pst.close();
		}

		return updateCount;
	}

	private List<Object> getBindList() {
		if (bind_list == null)
			bind_list = new ArrayList<Object>();
		return bind_list;
	}

	/**
	 * Ritorna il numero di righe modificate da una INSERT, UPDATE, DELETE.
	 * 
	 * @return il numero di righe modificate.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#getResultSet() java.sql.Statement.getResultSet()
	 */
	public int getUpdateCount() {
		return updateCount;
		// return this.getStatement().getUpdateCount();
	}

	/**
	 * Ritorna la connessione associata la controllo.
	 * 
	 * @return una istanza di tipo <I>Connection</I>.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#getConnection()
	 *      java.sql.Statement.getConnection()
	 */
	public Connection getConnection() {
		return connection;
	}

	public void setConnection(Connection conn) {
		connection = conn;
	}

	/**
	 * Esegue un comando specifico.
	 * 
	 * @return l'esecuzione del comando.
	 * @param String
	 *            il comando da eseguire.
	 * @throws SQLException
	 *             se viene riscontrato un errore.
	 * @see java.sql.Statement#executeUpdate(String)
	 *      java.sql.Statement.executeUpdate(String)
	 */
	public boolean execute(String query) throws SQLException {
		boolean printed = stampaSQL(query, null);

		Statement st = null;
		boolean ret = false;
		try {
			st = createStatement();
			ret = st.execute(query);
		} catch (SQLException e) {
			if (!printed) stampaSQL(query, e);
			throw e;
		
		} finally {
			if (st != null)
				st.close();
		}
		return ret;
	}

	/**
	 * Creaa una vista a partire da uan query
	 * 
	 * @param NAME
	 * @param query
	 * @return
	 * @throws SQLException
	 */
	public boolean createView(String name, String query) throws SQLException {
		query = "create or replace view " + name + " as\n" + query;
		boolean printed = stampaSQL(query, null);

		Statement st = null;
		boolean ret = false;
		try {
			st = createStatement();
			ret = st.execute(query);
		} catch (SQLException e) {
			if (!printed) stampaSQL(query, e);
			throw e;
		
		} finally {
			if (st != null)
				st.close();
		}
		return ret;
	}


	public boolean stampaSQL(String query, Throwable e) {
		return stampaSQL(log, query, e);
	}
	
	public static boolean stampaSQL(Logger log, String query, Throwable e) {
		if (e!=null) {
			String code = "";
			if (e instanceof SQLException) {
				code = " ["+((SQLException)e).getErrorCode()+"]";
			}
			log.error("Error statement SQL"+code+": "+e.toString()+"\n" +
					  "------------------------------------------------------------\n" +
					  query + (!query.endsWith("\n")?"\n":"") +
					  "------------------------------------------------------------",e);
			return true;
			
		} else if (log.isTraceEnabled()) {
			log.trace("Statement SQL:\n" +
					  "------------------------------------------------------------\n" +
					  query + (!query.endsWith("\n")?"\n":"") +
					  "------------------------------------------------------------");
			return true;
		}
		return false;
	}

	
	/**
	 * Chiude lo Statement aperto e legato all'oggetto
	 * 
	 * @throws SQLException
	 */
	public void close() throws SQLException {
		// System.out.println("<-- CHIUSO STATEMENT -------------->");
		try {
			if (res != null)
				res.close();
		} catch (SQLException e) {
			log.warn("close()", e);
		}
		if (res_stmt != null)
			res_stmt.close();
	}

	public boolean isOracle() {
		return SqlDB.isOracle(this.getConnection());
	}

	public boolean isMySQL() {
		return SqlDB.isMySQL(this.getConnection());
	}

	public boolean isSqlServer() {
		return SqlDB.isSqlServer(this.getConnection());
	}

	/**
	 * Funzione di database per la data corrente
	 * 
	 * @return String
	 */
	public String adesso() {
		if (isOracle())
			return "sysdate";
		else if (isMySQL())
			return "NOW()";
		else if (isSqlServer())
			return "NOW()";
		else
			return "---"; // Errore
	}

	/**
	 * Funzione di database per timestamp corrente
	 * 
	 * @return String
	 */
	public String timestamp() {
		if (isOracle())
			return "systimestamp";
		else if (isMySQL())
			return "NOW()";
		else if (isSqlServer())
			return "NOW()";
		else
			return "---"; // Errore
	}

	/*
	 * @deprecated public String fData(java.sql.Date valore, String
	 * formatoORALCE) { return fData(valore==null?"null":sDateDelimitator + new
	 * EnterpriseCalendar(valore).format4ORACLE(formatoORALCE) + sDateDelimitator); }
	 */

	public String fData(java.sql.Date valore, String formato) {
		return fData(EnterpriseCalendar.newInstance(valore), formato);
	}

	/*
	 * @deprecated public String fData(java.util.Date valore, String
	 * formatoORALCE) { return (valore==null?"null":sDateDelimitator + new
	 * EnterpriseCalendar(valore).format4ORACLE(formatoORALCE) + sDateDelimitator); }
	 */
	public String fData(java.util.Date valore, String formato) {
		return fData(EnterpriseCalendar.newInstance(valore), formato);
	}

	public String fData(EnterpriseCalendar valore) {
		if (valore == null)
			return "null";

		// Becca in modo automatico il formato della data
		if (isOracle()) {
			return sDateDelimitator + valore.format("yyyy-MM-dd HH:mm:ss") + sDateDelimitator;
		} else if (isMySQL()) {
			return sDateDelimitator + valore.format("yyyy-MM-dd HH:mm:ss") + sDateDelimitator;
		} else if (isSqlServer()) {
			return "CONVERT(DATETIME, '" + valore.format("dd/MM/yyyy HH:mm:ss") + "', 105)";
		}
		return "?";
	}

	public IClob fClob(String valore) {
		return new IClob(valore);
	}

	public IClob fClob(XString valore) {
		return new IClob(valore);
	}

	/**
	 * Inserisce un oggetto clob nel database
	 * 
	 * @param colonna
	 * @param clob
	 */
	public void addValue(String colonna, IClob clob) {
		if (XString.isBlankNullTrim(colonna))
			return;
		sColumns += (XString.isNotBlankNull(sColumns) ? ", " : "") + colonna.trim();
		if (XString.isNotBlankNull(sValues))
			sValues += ", ";
		if (clob != null) {
			this.getBindList().add(clob);
			sValues += "?";
		} else {
			sValues += "null";
		}
	}

	public void setValue(String colonna, IClob clob) {
		resetValues();
		addValue(colonna, clob);
	}

	/**
	 * Inserisce un oggetto blob nel database
	 * 
	 * @param colonna
	 * @param blob
	 */
	public void addValue(String colonna, IBlob blob) {
		if (XString.isBlankNullTrim(colonna))
			return;
		sColumns += (XString.isNotBlankNull(sColumns) ? ", " : "") + colonna.trim();
		if (XString.isNotBlankNull(sValues))
			sValues += ", ";
		if (blob != null) {
			this.getBindList().add(blob);
			sValues += "?";
		} else {
			sValues += "null";
		}
	}

	public void setValue(String colonna, IBlob blob) {
		resetValues();
		addValue(colonna, blob);
	}

	public void addColumnSET(String colonna, IClob clob) {
		if (XString.isBlankNullTrim(colonna))
			return;
		sColumns += (XString.isNotBlankNull(sColumns) ? ", " : "") + colonna.trim() + " = ";
		if (clob != null) {
			this.getBindList().add(clob);
			sColumns += "?";
		} else {
			sColumns += "null";
		}
	}

	public void setColumnSET(String colonna, IClob clob) {
		resetColumnsSET();
		addColumnSET(colonna, clob);
	}

	/**
	 * Restituisce il valore impostato come deafult se il resultser restituisce
	 * un valore null
	 * 
	 * @param res
	 *            ResultSet
	 * @param campo
	 *            Valore stringa rappresentante la COLUMN della tabella
	 * @param valore
	 *            Valore di deafult
	 * @return String
	 * @throws SQLException
	 */
	public static String getStringCatchNull(ResultSet res, String campo, String valore) throws SQLException {
		return (res.getString(campo) != null ? res.getString(campo) : valore);
	}

	public static String getStringCatchNull(ResultSet res, int campo, String valore) throws SQLException {
		return (res.getString(campo) != null ? res.getString(campo) : valore);
	}

	/**
	 * Restituisce il valore impostato come deafult se il Resultser restituisce
	 * un valore null
	 * 
	 * @param res
	 *            ResultSet
	 * @param campo
	 *            Valore stringa rappresentante la COLUMN della tabella
	 * @param valore
	 *            Valore di deafult
	 * @return Integer
	 * @throws SQLException
	 */
	public static Integer getIntegerCatchNull(ResultSet res, String campo, Integer valore) throws SQLException {
		return (res.getString(campo) != null ? new Integer(res.getInt(campo)) : valore);
	}

	public static Integer getIntegerCatchNull(ResultSet res, int campo, Integer valore) throws SQLException {
		return (res.getString(campo) != null ? new Integer(res.getInt(campo)) : valore);
	}

	public static Integer getInteger(ResultSet res, String campo) throws SQLException {
		return (res.getObject(campo) != null ? new Integer(res.getInt(campo)) : null);
	}

	/**
	 * Restituisce il valore impostato come deafult se il Resultser restituisce
	 * un valore null
	 * 
	 * @param res
	 * @param campo
	 * @param valore
	 * @return
	 * @throws SQLException
	 */
	public static Long getLongCatchNull(ResultSet res, String campo, Long valore) throws SQLException {
		return (res.getString(campo) != null ? new Long(res.getInt(campo)) : valore);
	}

	public static Long getLongCatchNull(ResultSet res, int campo, Long valore) throws SQLException {
		return (res.getString(campo) != null ? new Long(res.getInt(campo)) : valore);
	}

	public static Long getLong(ResultSet res, String campo) throws SQLException {
		return (res.getObject(campo) != null ? new Long(res.getInt(campo)) : null);
	}

	/**
	 * Restituisce il valore impostato come deafult se il Resultser restituisce
	 * un valore null
	 * 
	 * @param res
	 *            ResultSet
	 * @param campo
	 *            Valore stringa rappresentante la COLUMN della tabella
	 * @param valore
	 *            Valore di deafult
	 * @return Double
	 * @throws SQLException
	 */
	public static Double getDoubleCatchNull(ResultSet res, String campo, Double valore) throws SQLException {
		return (res.getString(campo) != null ? new Double(res.getDouble(campo)) : valore);
	}

	public static Double getDoubleCatchNull(ResultSet res, int campo, Double valore) throws SQLException {
		return (res.getString(campo) != null ? new Double(res.getDouble(campo)) : valore);
	}

	public static Double getDouble(ResultSet res, String campo) throws SQLException {
		return (res.getObject(campo) != null ? new Double(res.getDouble(campo)) : null);
	}

	/**
	 * Restituisce il valore impostato come deafult se il Resultser restituisce
	 * un valore null
	 * 
	 * @param res
	 *            ResultSet
	 * @param campo
	 *            Valore stringa rappresentante la COLUMN della tabella
	 * @param valore
	 *            Valore di deafult
	 * @return Float
	 * @throws SQLException
	 */
	public static Float getFloatCatchNull(ResultSet res, String campo, Float valore) throws SQLException {
		return (res.getString(campo) != null ? new Float(res.getFloat(campo)) : valore);
	}

	public static Float getFloatCatchNull(ResultSet res, int campo, Float valore) throws SQLException {
		return (res.getString(campo) != null ? new Float(res.getFloat(campo)) : valore);
	}

	public static Float getFloat(ResultSet res, String campo) throws SQLException {
		return (res.getObject(campo) != null ? new Float(res.getFloat(campo)) : null);
	}

	public Sql duplicate() throws SQLException {
		Sql nuovo = new Sql(this.getConnection());
		nuovo.setColumns(this.getColumns());
		nuovo.setValues(this.getValues());
		nuovo.setTables(this.getTables());
		nuovo.setWhere(this.getWhere());
		nuovo.setOptions(this.getOptions());
		return nuovo;
	}

	/**
	 * Analizza l'errore SQL confrontandolo con quello specificato in funzione
	 * del database
	 * 
	 * @param e
	 * @param error
	 * @return
	 */
	public boolean analyseError(SQLException e, int error) {
		return SqlDB.analyseError(connection, e, error);
	}

	public boolean analyseErrorORA(SQLException e, int ORA_error) {
		return SqlDB.analyseError(SqlDB.ORACLE, e, ORA_error);
	}

	/**
	 * Aggiunge una condizione WHERE alla query con un elenco di valori
	 * 
	 * @param colonna
	 * @param valori
	 */
	public void addWhereIN(String colonna, String[] valori) {
		addWhereIN(colonna, false, valori);
	}

	public void addWhereNotIN(String colonna, String[] valori) {
		addWhereIN(colonna, true, valori);
	}

	private void addWhereIN(String colonna, boolean negato, String[] valori) {
		if (colonna == null)
			return;

		String condizione = "";
		if (valori != null && valori.length > 0) {
			if (valori.length > 1) {
				for (int i = 0; i < valori.length; i++) {
					condizione += (!condizione.equals("") ? "," : "") + fStr(valori[i]);
				}
				condizione = (negato ? " NOT" : "") + " IN (" + condizione + ")";
			} else {
				if (valori[0] == null) {
					condizione = " IS" + (negato ? " NOT" : "") + " null";
				} else {
					condizione = (negato ? " != " : " = ") + fStr(valori[0]);
				}
			}
		} else {
			condizione = " IS" + (negato ? " NOT" : "") + " null";
		}
		sWhere += (sWhere != null && !sWhere.equals("") ? " AND " : "") + colonna + condizione;
	}

	public void addWhereIN(String colonna, Integer[] valori) {
		addWhereIN(colonna, false, valori);
	}

	public void addWhereNotIN(String colonna, Integer[] valori) {
		addWhereIN(colonna, true, valori);
	}

	public void addWhereIN(String colonna, boolean negato, Integer[] valori) {
		if (colonna == null)
			return;

		String condizione = "";
		if (valori != null && valori.length > 0) {
			if (valori.length > 1) {
				for (int i = 0; i < valori.length; i++) {
					condizione += (!condizione.equals("") ? "," : "") + fNum(valori[i]);
				}
				condizione = (negato ? " NOT" : "") + " IN (" + condizione + ")";
			} else {
				if (valori[0] == null) {
					condizione = " IS" + (negato ? " NOT" : "") + " null";
				} else {
					condizione = (negato ? " != " : " = ") + fNum(valori[0]);
				}
			}
		} else {
			condizione = " IS" + (negato ? " NOT" : "") + " null";
		}
		sWhere += (sWhere != null && !sWhere.equals("") ? " AND " : "") + colonna + condizione;
	}

	private String fValue(Object object) {
		if (object instanceof String) {
			return fStr((String) object);

		} else if (object instanceof Integer) {
			return fNum((Integer) object);

		} else if (object instanceof BigDecimal) {
			return fNum((BigDecimal) object);

		} else if (object instanceof Long) {
			return fNum((BigDecimal) object);

		} else if (object instanceof EnterpriseCalendar) {
			return fData((EnterpriseCalendar) object);
		}

		// default
		return fStr(object.toString());
	}

	public void addWhereIN(String colonna, List<?> valori) {
		if (colonna == null)
			return;

		String condizione = "";
		if (valori != null && valori.size() > 0) {
			if (valori.size() > 1) {
				for (int i = 0; i < valori.size(); i++) {
					condizione += (!condizione.equals("") ? "," : "") + fValue(valori.get(i));
				}
				condizione = " IN (" + condizione + ")";
			} else {
				if (valori.get(0) == null) {
					condizione = " IS null";
				} else {
					condizione = " = " + fValue(valori.get(0));
				}
			}
		} else {
			condizione = " IS null";
		}
		sWhere += (sWhere != null && !sWhere.equals("") ? " AND " : "") + colonna + condizione;
	}

}