package it.softecspa.kahuna.sql;

import java.sql.Connection;
import java.sql.SQLException;

public class SqlDB {

	public static final String ORACLE = "Oracle";
	public static final String MYSQL = "MySql";
	public static final String SQLSERVER = "SQLServer";

	public static final int UNIQUE_CONSTRAINT = 1;
	public static final int INTEGRITY_CONSTRAINT = 2;
	public static final int TABLE_NOT_EXIST = 3;
	public static final int RESOUCE_BUSY = 4;
	public static final int UNSUPPORTED_FEATURE = 5;
	public static final int UNIMPLEMENTED_REQUESTED = 6;

	public static final char DECIMAL_SEPARATOR = '.';

	// Variabile statica utilizzata per forzare il database utilizzato
	private static String alternative_database = ORACLE;

	// Default
	public static final String ORACLE_DATE_FORMAT = "YYYY-MM-DD HH24:MI:SS";

	public static void alterSessionORACLE_DATE_FORMAT(Connection connection) throws java.sql.SQLException {
		if (SqlDB.isOracle(connection)) {
			java.sql.Statement st = connection.createStatement();
			st.execute("ALTER SESSION SET NLS_DATE_FORMAT = '" + SqlDB.ORACLE_DATE_FORMAT + "'");
		}
		return;
	}

	static public String getAlternativeDatabase() {
		return alternative_database;
	}

	static public String setAlternativeDatabase(String database) {
		return alternative_database = database;
	}

	static private boolean is(Connection connection, String confronto) {
		String productName = null;
		try {
			if (connection != null)
				productName = connection.getMetaData().getDatabaseProductName();
		} catch (SQLException e) {
			productName = alternative_database;
		} catch (Error e) {
			productName = alternative_database;
		}

		return confronto.equalsIgnoreCase(productName);
	}

	static public boolean isOracle(Connection connection) {
		return is(connection, ORACLE);
	}

	static public boolean isMySQL(Connection connection) {
		return is(connection, MYSQL);
	}

	static public boolean isSqlServer(Connection connection) {
		return is(connection, SQLSERVER);
	}

	/* Decodifica messaggi di errore frequentemente utilizzati */
	public static int analyseError(Connection connection, SQLException e) {
		if (connection == null)
			return 0;

		if (isOracle(connection)) {
			return convertiErroreOracle(e.getErrorCode());
		} else if (isMySQL(connection)) {
			return convertiErroreMySQL(e.getErrorCode());
		} else if (isSqlServer(connection)) {
			return convertiErroreSqlServer(e.getErrorCode());
		}
		return 0;
	}

	public static int analyseError(String tipo, SQLException e) {
		if (tipo.equalsIgnoreCase(ORACLE)) {
			return convertiErroreOracle(e.getErrorCode());
		} else if (tipo.equalsIgnoreCase(MYSQL)) {
			return convertiErroreMySQL(e.getErrorCode());
		} else if (tipo.equalsIgnoreCase(SQLSERVER)) {
			return convertiErroreSqlServer(e.getErrorCode());
		}
		return 0;
	}

	private static int convertiErroreOracle(int codice) {
		switch (codice) {
		// http://*.ora-code.com/
		// ORA-00001: unique constraint (string.string) violated
		case 1:
			return UNIQUE_CONSTRAINT;
			// ORA-01460: unimplemented or unreasonable conversion requested
		case 1460:
			return UNIMPLEMENTED_REQUESTED;
			// ORA-02291: integrity constraint <constraint name> violated -
			// parent key not found
		case 17023:
			return UNSUPPORTED_FEATURE;
			// ORA-17023: Unsupported feature
		case 2291:
			return INTEGRITY_CONSTRAINT;
			// ORA-02292: integrity constraint <constraint name> violated -
			// child record found
		case 2292:
			return INTEGRITY_CONSTRAINT;
			// ORA-00942: table or view does not exist
		case 942:
			return TABLE_NOT_EXIST;
			// ORA-00054: resource busy and acquire with NOWAIT specified
		case 54:
			return RESOUCE_BUSY;
			// ORA-30006: resource busy; acquire with WAIT timeout expired
		case 30006:
			return RESOUCE_BUSY;
		}
		return 0;
	}

	private static int convertiErroreMySQL(int codice) {
		switch (codice) {
		case 1062:
			return UNIQUE_CONSTRAINT;
		case 1217:
			return INTEGRITY_CONSTRAINT;
		case 1146:
			return TABLE_NOT_EXIST;
		}
		return 0;
	}

	private static int convertiErroreSqlServer(int codice) {
		// TODO lista da implementare
		return 0;
	}

	public static boolean analyseError(Connection connection, SQLException e, int confronto) {
		return analyseError(connection, e) == confronto;
	}

	public static boolean analyseError(String tipo, SQLException e, int confronto) {
		return analyseError(tipo, e) == confronto;
	}

}
