package it.softecspa.kahuna.util;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.log.KLogger;

import java.sql.Time;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

@SuppressWarnings("serial")
public class CalendarUTC extends GregorianCalendar {

	static KLogger log = KLogger.getLogger(CalendarUTC.class);

	public static final int DAY = 255;

	static private int[][] festiviITA = new int[][] { { 1, 1 } // Capodanno
			, { 1, 6 } 		// Epifania
			, { 4, 25 } 	// 25 Aprile
			, { 5, 1 } 		// 1 Maggio
			, { 6, 2 } 		// 2 Giugno
			, { 8, 15 } 	// Ferragosto (Assunzione di Maria)
			, { 11, 1 } 	// 1 Novembre
			, { 12, 8 } 	// 8 Dicembre
			, { 25, 12 } 	// Natale
			, { 26, 12 } }; // S.Stefano

	
	/* Date format pattern used to parse HTTP date headers in RFC 1123 format */
	public static final String PATTERN_RFC1123 = "EEE, dd MMM yyyy HH:mm:ss zzz";

	/* Date format pattern used to parse HTTP date headers in RFC 1036 format */
	public static final String PATTERN_RFC1036 = "EEEE, dd-MMM-yy HH:mm:ss zzz";
	
	/* Date format pattern used to parse HTTP date headers in ANSI C <code>asctime()</code> format */
	public static final String PATTERN_ASCTIME = "EEE MMM d HH:mm:ss yyyy";
	
	public static final String PATTERN_UTC_TIMEZONE = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
	public static final String PATTERN_UTC_ZULU = "yyyy-MM-dd'T'HH:mm:ss'Z'";
	
	public static final String[] PATTERNS_UTC = {PATTERN_UTC_TIMEZONE,PATTERN_UTC_ZULU} ;
	
	
	public static final String PATTERN_CANONICAL_IT = "dd/MM/yyyy HH:mm.ss";
	
	
	
	private final static TimeZone GMT = TimeZone.getTimeZone("GMT");
	private final static TimeZone UTC = TimeZone.getTimeZone("UTC");
	
	
	public CalendarUTC() {
		super();
	}

	public CalendarUTC(int year, int month, int day) {
		super(year, month - 1, day);
	}

	public CalendarUTC(int year, int month, int day, int hour, int minute) {
		super(year, month - 1, day, hour, minute);
	}

	public CalendarUTC(int year, int month, int day, int hour, int minute, int second) {
		super(year, month - 1, day, hour, minute, second);
		super.getTimeZone();
	}

	/*
	 * @deprecated
	public EnterpriseCalendar(String valore) throws java.text.ParseException {
		super();

		if ((valore != null) && !valore.trim().equals("")) {
			DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT, Locale.ITALY);
			java.util.Date data = df.parse(valore);
			// System.out.println(data);
			super.setTime(data);
		}
	}
	*/

	public CalendarUTC(String value, String pattern) throws Exception {
		this(value, new String[] {pattern}, null);		
	}
		
	public CalendarUTC(String value, String[] patterns) throws Exception {
		this(value, patterns, null);
	}
	
	public CalendarUTC(String value, String[] patterns, Locale locale) throws Exception {
		super();
		
		if (XString.isBlankNullTrim(value)) {
			throw new NullPointerException();
		}
	
		// Autocorrezione
		if (value.length() > 1 && value.startsWith("'") && value.endsWith("'")) {
			value = value.substring(1, value.length() - 1);
		}
		
		
		java.util.Date date = null;
		for (String format : patterns) {
			SimpleDateFormat dateParser = null;
			
			if (format.equals(PATTERN_RFC1036)) {
				dateParser = new SimpleDateFormat(PATTERN_RFC1036, Locale.US);
				dateParser.setTimeZone(GMT);				
			
			} else if (format.equals(PATTERN_RFC1123)) {
				dateParser = new SimpleDateFormat(PATTERN_RFC1123, Locale.US);
				dateParser.setTimeZone(GMT);
			
			} else if (format.equals(PATTERN_UTC_TIMEZONE)) {
				dateParser = new SimpleDateFormat(PATTERN_UTC_TIMEZONE);
				
			} else if (format.equals(PATTERN_UTC_ZULU)) {
				dateParser = new SimpleDateFormat(PATTERN_UTC_ZULU);
				dateParser.setTimeZone(UTC);
				
			} else {
				if (locale!=null) {
					dateParser = new SimpleDateFormat(format, locale);
				} else {
					dateParser = new SimpleDateFormat(format);
				}
			}
			
			try {
				date = dateParser.parse(value);	
				break;
			} catch (ParseException e) {
				// ignore this exception, we will try the next format
			}
		}
		
		if (date==null) {
			String list = "";
			for (String format : patterns) {
				list += (list.length()>0?",":"")+"{"+format+"}";
			}
			throw new Exception("No valid patters in list: "+list);
		}
		
		super.setTime(date);
		
	}	
	

	public CalendarUTC(Timestamp data) {
		super();
		super.setTime(data);
	}
	

	public CalendarUTC(java.util.Date data) {
		super();
		super.setTime(data);
	}
	

	public CalendarUTC(java.sql.Date data) {
		super();
		super.setTime(data);
	}
	

	public CalendarUTC(Calendar calendar) {
		super();
		super.setTime(calendar.getTime());
	}

	
	public CalendarUTC(long millisecondi) {
		super();
		super.setTimeInMillis(millisecondi);
	}

	
	public CalendarUTC(Locale locale) {
		super(locale);
	}

	
	public CalendarUTC(TimeZone zone) {
		super(zone);
	}

	public CalendarUTC(TimeZone zone, Locale locale) {
		super(zone, locale);
	}

	public void setDay(CalendarUTC giorno) {
		this.set(DAY_OF_MONTH, 1);
		this.set(YEAR, giorno.get(YEAR));
		this.set(MONTH, giorno.get(MONTH));
		this.set(DAY_OF_MONTH, giorno.get(DAY_OF_MONTH));
	}

	public void setDay(int year, int month, int day) {
		this.set(DAY_OF_MONTH, 1);
		this.set(YEAR, year);
		this.set(MONTH, month - 1);
		this.set(DAY_OF_MONTH, day);
	}

	public void setHours(CalendarUTC ore) {
		this.set(HOUR_OF_DAY, ore.get(HOUR_OF_DAY));
		this.set(MINUTE, ore.get(MINUTE));
		this.set(SECOND, ore.get(SECOND));
	}

	public void setHours(int hour, int minute, int second) {
		this.set(HOUR_OF_DAY, hour);
		this.set(MINUTE, minute);
		this.set(SECOND, second);
	}

	
	/**
	 * Formatta la data utilizzando la classe   {@link java.text.SimpleDataFormat}
	 * <table>
	 *  <tr><td><b>Letter</b></td><td><b>Date or Time Component</b></td><td><b>Presentation</b></td><td><b>Examples</b></td></tr>
	 *  <tr><td><b>G</b></td><td>Era designator      </td><td>Text   </td><td>AD            </td></tr>
	 *  <tr><td><b>y</b></td><td>Year                </td><td>Year   </td><td>1996; 96      </td></tr>
	 *  <tr><td><b>M</b></td><td>Month in year       </td><td>Month  </td><td>July; Jul; 07 </td></tr>
	 *  <tr><td><b>w</b></td><td>Week in year        </td><td>Number </td><td>27            </td></tr>
	 *  <tr><td><b>W</b></td><td>Week in month       </td><td>Number </td><td>2             </td></tr>
	 *  <tr><td><b>D</b></td><td>Day in year         </td><td>Number </td><td>189           </td></tr>
	 *  <tr><td><b>d</b></td><td>Day in month        </td><td>Number </td><td>10            </td></tr>
	 *  <tr><td><b>F</b></td><td>Day of week in month</td><td>Number </td><td>2             </td></tr>
	 *  <tr><td><b>E</b></td><td>Day in week         </td><td>Text   </td><td>Tuesday; Tue  </td></tr>
	 *  <tr><td><b>a</b></td><td>Am/pm marker        </td><td>Text   </td><td>PM            </td></tr>
	 *  <tr><td><b>H</b></td><td>Hour in day (0-23)  </td><td>Number </td><td>0             </td></tr>
	 *  <tr><td><b>k</b></td><td>Hour in day (1-24)  </td><td>Number </td><td>24            </td></tr>
	 *  <tr><td><b>K</b></td><td>Hour in am/pm (0-11)</td><td>Number </td><td>0             </td></tr>
	 *  <tr><td><b>h</b></td><td>Hour in am/pm (1-12)</td><td>Number </td><td>12            </td></tr>
	 *  <tr><td><b>m</b></td><td>Minute in hour      </td><td>Number </td><td>30            </td></tr>
	 *  <tr><td><b>s</b></td><td>Second in minute    </td><td>Number </td><td>55            </td></tr>
	 *  <tr><td><b>S</b></td><td>Millisecond         </td><td>Number </td><td>978           </td></tr>
	 *  <tr><td><b>z</b></td><td>Time zone           </td><td colsapn=2>General time zone  Pacific Standard Time; PST; GMT-08:00 </td></tr>
	 *  <tr><td><b>Z</b></td><td>Time zone           </td><td colsapn=2>RFC 822 time zone  -0800</td></tr>
	 *  <tr><td><b>Z</b></td><td>RFC1123           	 </td><td colsapn=2>RFC1123 "EEEE, dd-MMM-yy HH:mm:ss zzz"</td></tr>
	 *  <tr><td><b>Z</b></td><td>RFC1036           	 </td><td colsapn=2>RFC1036 "EEEE, dd-MMM-yy HH:mm:ss zzz"</td></tr>
	 *  <tr><td><b>Z</b></td><td>ASC time			 </td><td colsapn=2>ANSI C ASc time "EEE MMM d HH:mm:ss yyyy"</td></tr>
	 * </table>  
	 * 
	 * Formato standard italia: dd/MM/yyyy'&nbsp;'HH:mm.ss
	 */
	public String format(String formato) {
		SimpleDateFormat dateFormat = new SimpleDateFormat(formato);
		return dateFormat.format(this.getTime());
	}
	
	
	public String format(String pattern, Locale locale) {
		SimpleDateFormat formatter = new SimpleDateFormat(pattern, locale);
		return formatter.format(this.getTime());
	}
	
	public String format(String pattern, Locale locale, TimeZone zone) {
		SimpleDateFormat formatter = new SimpleDateFormat(pattern, locale);
		formatter.setTimeZone(zone);
		return formatter.format(this.getTime());
	}

	/**
	 * Formatta la data secondo lo standard RFC1123
	 * @return Una stringa con la data formattata
	 */
	public String formatRFC1123() {
		return format(PATTERN_RFC1123, Locale.US, GMT);
	}
	
	/**
	 * Formatta la data secondo lo standard RFC1036
	 * @return Una stringa con la data formattata
	 */
	public String formatRFC1036() {
		return format(PATTERN_RFC1036, Locale.US, GMT);
	}
	
	public String formatCanonical() {
		return this.format(PATTERN_CANONICAL_IT, Locale.ITALY);
	}
	
	public String formatISO8601zulu() {
		return format(PATTERN_UTC_ZULU, Locale.ENGLISH, UTC);
	}

	public String formatISO8601withTimezone() {
		return format(PATTERN_UTC_TIMEZONE);
	}
	
	
	/** @deprecated */
	static public int confrontaGiorno(GregorianCalendar primo,
			GregorianCalendar secondo) {
		// 0 = date uguali
		// 1 = primo > secondo
		// 2 = primo < secondo
		long g1 = ((primo.get(YEAR) * 100 + primo.get(MONTH) + 1) * 100)
				+ primo.get(DAY_OF_MONTH);
		long g2 = ((secondo.get(YEAR) * 100 + secondo.get(MONTH) + 1) * 100)
				+ secondo.get(DAY_OF_MONTH);
		if (g1 < g2)
			return 2;
		if (g1 > g2)
			return 1;
		return 0;
	}

	/** @deprecated */
	static public int confrontaGiornoOra(GregorianCalendar primo,
			GregorianCalendar secondo) {
		// 0 = date uguali
		// 1 = primo > secondo
		// 2 = primo < secondo
		double anno = primo.get(YEAR);
		double mese = (anno * 100) + primo.get(MONTH) + 1;
		double giorno = (mese * 100) + primo.get(DAY_OF_MONTH);
		double ore = (giorno * 100) + primo.get(HOUR_OF_DAY);
		double minuti = (ore * 100) + primo.get(MINUTE);
		double g1 = (minuti * 100) + primo.get(SECOND);
		// System.out.println(g1);

		anno = secondo.get(YEAR);
		mese = (anno * 100) + secondo.get(MONTH) + 1;
		giorno = (mese * 100) + secondo.get(DAY_OF_MONTH);
		ore = (giorno * 100) + secondo.get(HOUR_OF_DAY);
		minuti = (ore * 100) + secondo.get(MINUTE);
		double g2 = (minuti * 100) + secondo.get(SECOND);
		// System.out.println(g2);

		if (g1 < g2)
			return 2;
		if (g1 > g2)
			return 1;
		return 0;
	}

	public static CalendarUTC getOggi() {
		CalendarUTC adesso = CalendarUTC.getAdesso();
		return new CalendarUTC(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH), 0, 0, 0);
	}

	public static CalendarUTC getStanotte() {
		CalendarUTC adesso = CalendarUTC.getAdesso();
		return new CalendarUTC(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH), 23, 59, 29);
	}

	public static CalendarUTC getIeri() {
		CalendarUTC adesso = CalendarUTC.getAdesso();
		return new CalendarUTC(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH) - 1, 0, 0, 0);
	}

	/**
	 * Restituisce una istanza di EnterpriseCalendar del giorno anteriore o posteriore
	 * ad oggi
	 * 
	 * @param giorni
	 *            di differenza
	 * @return
	 */
	public static CalendarUTC getGiorno(int giorni) {
		CalendarUTC adesso = CalendarUTC.getAdesso();
		return new CalendarUTC(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH) + giorni, 0, 0, 0);
	}

	public static CalendarUTC getDomani() {
		CalendarUTC adesso = CalendarUTC.getAdesso();
		return new CalendarUTC(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH) + 1, 0, 0, 0);
	}

	public static CalendarUTC getAdesso() {
		return new CalendarUTC();
	}

	
	public static String getMese(int mese) {
		switch (mese) {
		case 1:
			return "gennaio";
		case 2:
			return "febbraio";
		case 3:
			return "marzo";
		case 4:
			return "aprile";
		case 5:
			return "maggio";
		case 6:
			return "giugno";
		case 7:
			return "luglio";
		case 8:
			return "agosto";
		case 9:
			return "settembre";
		case 10:
			return "ottobre";
		case 11:
			return "novembre";
		case 12:
			return "dicembre";
		}

		return null;
	}

	public static String getPeriodo(java.util.Date inizio, java.util.Date fine) {
		CalendarUTC dInizio = (inizio != null ? new CalendarUTC(inizio) : null);
		CalendarUTC dFine = (fine != null ? new CalendarUTC(fine) : null);

		return CalendarUTC.getPeriodo(dInizio, dFine);
	}

	public static String getPeriodo(CalendarUTC inizio, CalendarUTC fine) {
		if (inizio == null && fine == null)
			return "";

		String s = "";
		boolean stessoAnno = false;
		boolean stessoMese = false;
		boolean stessoGiorno = false;
		if (inizio != null && fine != null) {
			if (inizio.get(CalendarUTC.YEAR) == fine.get(CalendarUTC.YEAR)) {
				stessoAnno = true;
				if (inizio.get(CalendarUTC.MONTH) == fine.get(CalendarUTC.MONTH)) {
					stessoMese = true;
					if (inizio.get(CalendarUTC.DAY_OF_MONTH) == fine
							.get(CalendarUTC.DAY_OF_MONTH)) {
						stessoGiorno = true;
					}
				}
			}

			// System.out.println(stessoAnno);
			// System.out.println(stessoMese);
			// System.out.println(stessoGiorno);
			if (stessoAnno && stessoMese && stessoGiorno) {
				s = "il " + inizio.format("dd/MM/yyyy");
			} else if (stessoAnno && stessoMese) {
				s = "dal " + inizio.format("dd") + " al " + fine.format("dd")
						+ " " + getMese(inizio.get(CalendarUTC.MONTH) + 1) + " "
						+ inizio.format("yyyy");
			} else if (stessoAnno) {
				s = "dal " + inizio.format("dd/MM") + " al "
						+ fine.format("dd/MM") + " " + inizio.format("yyyy");
			} else {
				s = "dal " + inizio.format("dd/MM/yyyy") + " al "
						+ fine.format("dd/MM/yyyy");
			}
		} else {
			if (inizio != null) {
				s = "dal " + inizio.format("dd/MM/yyyy");
			} else if (fine != null) {
				s = "al " + fine.format("dd/MM/yyyy");
			}
		}
		return s;
	}

	/**
	 * Restituisce il giorno di Pasqua per l'anno impostato
	 * 
	 * @param int anno
	 * @return EnterpriseCalendar
	 */
	public static CalendarUTC getPasqua(int anno) {
		/*
		 * Un metodo per il calcolo della data di Pasqua lo si trova descritto
		 * nel libro di Jean Meeus
		 * "Astronomia con il computer - Formule, metodi di calcolo, esempi numerici"
		 * - ed.Hoepli - (Lire 25000 nel 1990 - libro molto interessante -
		 * titolo originale dell'opera "Astronomical Formul for Calculators" )
		 * ove si dice, a proposito del calcolo della data di Pasqua: Il medodo
		 * qui descritto  stato pubblicato da Spencer Jones nel suo libro
		 * General Astronomy (pagg.73-74 dell'edizione del 1922).  stato
		 * ripubblicato nel Journal of the British Astronomical Association,
		 * vol.88, pag.91 (dicembre 1977), dove si dice che fu sviluppato nel
		 * 1876 e che comparve nell'Ecclesiastic Calendar di Butcher. A
		 * differenza della formula data da Gauss, questo metodo non ha
		 * eccezioni ed  valido per tutti gli anni nel EnterpriseCalendar Gregoriano,
		 * ossia dal 1583 in poi. La procedura per determinare la data della
		 * Pasqua  la seguente:
		 * 
		 * dividere per quoziente resto
		 * 
		 * l'anno x 19 - a l'anno x 100 b c b 4 d e b+8 25 f - b-f+1 3 g -
		 * 19a+b-d-g+15 30 - h c 4 i k 32+2e+2i-h-k 7 - l a+11h-22l 451 m -
		 * h+l-7m+114 31 n p
		 * 
		 * Allora n = numero del mese (3=marzo, 4=aprile) p+1 = giorno del mese
		 * in cui cade la domenica di Pasqua (...omissis...) Le date estreme
		 * della Pasqua sono Marzo 22 (come nel 1818 e nel 2285) ed Aprile 25
		 * (come nel 1886, nel 1943 e nel 2038) Copyright  1998 G.Paolo Bizzaro
		 */

		CalendarUTC pasqua = new CalendarUTC();
		int a, b, c, p, q, r;
		a = anno % 19;
		b = anno / 100;
		c = anno % 100;
		p = (19 * (a) + b - (b / 4) - ((b - ((b + 8) / 25) + 1) / 3) + 15) % 30;
		q = (32 + 2 * ((b % 4) + (c / 4)) - p - (c % 4)) % 7;
		r = (p + q - 7 * ((a + 11 * p + 22 * q) / 451) + 114);
		pasqua.set(anno, (r / 31) - 1, (r % 31) + 1, 0, 0, 0);
		return pasqua;
	}

	public boolean isPasqua() {
		CalendarUTC pasqua = CalendarUTC.getPasqua(this.get(YEAR));
		return this.get(MONTH) == pasqua.get(MONTH)
				&& this.get(DAY_OF_MONTH) == pasqua.get(DAY_OF_MONTH);
	}

	/**
	 * Restituisce il primo giorno dell'anno per l'anno impostato
	 * 
	 * @param int anno
	 * @return EnterpriseCalendar
	 */
	public static CalendarUTC getCapodanno(int anno) {
		return new CalendarUTC(anno, 1, 1, 0, 0, 0);
	}

	/**
	 * Restituisce il giorno di Natale per l'anno impostato
	 * 
	 * @param int anno
	 * @return EnterpriseCalendar
	 */
	public static CalendarUTC getNatale(int anno) {
		return new CalendarUTC(anno, 12, 25, 0, 0, 0);
	}

	public boolean isCapodanno() {
		return this.get(MONTH) == JANUARY && this.get(DAY_OF_MONTH) == 1;
	}

	public boolean isNatale() {
		return this.get(MONTH) == DECEMBER && this.get(DAY_OF_MONTH) == 25;
	}

	public boolean isPeriodoNatale() {
		return isPeriodoNatale(8, 6);
	}

	public boolean isPeriodoNatale(int dicembre, int gennaio) {
		return (this.get(MONTH) == DECEMBER && this.get(DAY_OF_MONTH) >= dicembre)
				|| (this.get(MONTH) == JANUARY && this.get(DAY_OF_MONTH) <= gennaio);
	}

	public CalendarUTC getUltimoDelMese() {
		return getUltimoDelMese(this.get(YEAR), this.get(MONTH) + 1);
	}

	public static CalendarUTC getUltimoDelMese(int anno, int mese) {
		CalendarUTC data = new CalendarUTC(anno, mese, 1, 0, 0, 0);
		int giorno = data.getActualMaximum(DAY_OF_MONTH);
		data.set(DAY_OF_MONTH, giorno);
		return data;
	}

	public CalendarUTC getPrimoDelMese() {
		return getPrimoDelMese(this.get(YEAR), this.get(MONTH) + 1);
	}

	public static CalendarUTC getPrimoDelMese(int anno, int mese) {
		CalendarUTC data = new CalendarUTC(anno, mese, 1, 0, 0, 0);
		return data;
	}

	/**
	 * Restituisce TRUE se il giorno indicato  festivo Per il calcolo sono
	 * utilizzate le festivit del calendario italiano
	 * 
	 * @param data
	 * @return
	 */
	public static boolean isFestivo(GregorianCalendar data) {
		// La Domenica  giorno festivo
		int week = data.get(DAY_OF_WEEK);
		if (week == SUNDAY)
			return true;

		// Verifica festivit canonica
		int mese = data.get(MONTH) + 1;
		int giorno = data.get(DAY_OF_MONTH);
		for (int i = 0; i < festiviITA.length; i++) {
			if (festiviITA[i][0] == mese) {
				if (festiviITA[i][1] == giorno) {
					return true;
				}
			}
		}

		// Verifica Pasqua
		CalendarUTC pasqua = getPasqua(data.get(YEAR));
		if (pasqua.getMonth() == mese) {
			if (pasqua.getDay() == giorno) {
				return true;
			}
		}

		// Verifica Luned dell'Angelo
		pasqua.add(DAY_OF_MONTH, 1);
		if (pasqua.getMonth() == mese) {
			if (pasqua.getDay() == giorno) {
				return true;
			}
		}

		return false;
	}

	public boolean isFestivo() {
		return isFestivo(this);
	}

	/**
	 * Restituisce TRUE se il giorno indicato  feriale Per il calcolo sono
	 * utilizzate le festivit del calendario italiano
	 * 
	 * @param data
	 * @return
	 */
	public static boolean isFeriale(GregorianCalendar data) {
		return !isFestivo(data);
	}

	public boolean isFeriale() {
		return !isFestivo(this);
	}

	/**
	 * Restituisce TRUE se il giorno indicato  lavorativo Il Sabato e la
	 * Domenica non si lavora! Per il calcolo sono utilizzate le festivit del
	 * calendario italiano
	 * 
	 * @param data
	 * @return
	 */

	public static boolean isLavorativo(GregorianCalendar data) {
		int week = data.get(DAY_OF_WEEK);

		// Se Sabato non si lavora
		if (week == SATURDAY)
			return false;

		return !isFestivo(data);
	}

	public boolean isLavorativo() {
		return isLavorativo(this);
	}

	/**
	 * Restituisce il primo giorno lavorativo dopo la data specificata
	 * 
	 * @param data
	 * @return
	 */
	public static CalendarUTC getFirstLavorativoAfter(GregorianCalendar data) {
		CalendarUTC confronto = newInstance(data);
		if (confronto == null)
			return null;

		do {
			confronto.add(DAY_OF_MONTH, 1);
			// int week = confronto.getWeek();
			// if (week!=SUNDAY) {confronto.add(DAY_OF_MONTH, 1); continue;}
			// if (week!=SATURDAY) {confronto.add(DAY_OF_MONTH, 1); continue;}
			if (!confronto.isLavorativo())
				continue;

			return confronto;
		} while (true);
	}

	public CalendarUTC getFirstLavorativoAfter() {
		return getFirstLavorativoAfter(this);
	}

	public long differenza(CalendarUTC confronto) {
		return differenza(confronto, DAY);
	}

	public long differenza(CalendarUTC confronto, int field) {
		if (field == MILLISECOND)
			return getTimeInMillis() - confronto.getTimeInMillis();

		long anni = this.get(YEAR) - confronto.get(YEAR);
		if (field == YEAR)
			return anni;

		if (field == MONTH) {
			return (anni * 12) + this.get(MONTH) - confronto.get(MONTH);
		}

		long diff = (anni * 365) + this.get(DAY_OF_YEAR)
				- confronto.get(DAY_OF_YEAR);
		if (field == DAY_OF_YEAR || field == DAY_OF_MONTH || field == DAY)
			return diff;

		diff = (diff * 24) + this.get(HOUR_OF_DAY) - confronto.get(HOUR_OF_DAY);
		if (field == HOUR || field == HOUR_OF_DAY)
			return diff;

		diff = (diff * 60) + this.get(MINUTE) - confronto.get(MINUTE);
		if (field == MINUTE)
			return diff;

		diff = (diff * 60) + this.get(SECOND) - confronto.get(SECOND);
		if (field == SECOND)
			return diff;

		return 0;
	}

	/**
	 * Descrizione del'intervallo
	 * 
	 * @param secondi
	 * @return
	 */
	public static String descrizioneIntervallo(long secondi) {
		// Calcolo della durata esperessa in secondi
		String durata = "";
		if (secondi == 0) {
			durata += "<1 secondo";
		} else if (secondi == 1) {
			durata += secondi + " secondo";
		} else if (secondi < 60) {
			durata += secondi + " secondi";
		} else {
			long mm = (secondi / 60);
			long ss = (secondi % 60);
			if (mm == 1) {
				durata += mm + " minuto";
			} else if (mm < 60) {
				durata += mm + " minuti";
			} else {
				long hh = (mm / 60);
				mm = mm % 60;
				durata += hh + " ore, " + mm + " minut" + (mm == 1 ? "o" : "i");
			}
			durata += " e " + ss + " second" + (ss == 1 ? "o" : "i");
		}
		return durata;
	}

	public String toString() {
		return this.format("yyyy-MM-dd HH:mm.ss,SSS E");

		/**
		 * Letter Date or Time Component Presentation Examples G Era designator
		 * Text AD y Year Year 1996; 96 M Month in year Month July; Jul; 07 w
		 * Week in year Number 27 W Week in month Number 2 D Day in year Number
		 * 189 d Day in month Number 10 F Day of week in month Number 2 E Day in
		 * week Text Tuesday; Tue a Am/pm marker Text PM H Hour in day (0-23)
		 * Number 0 k Hour in day (1-24) Number 24 K Hour in am/pm (0-11) Number
		 * 0 h Hour in am/pm (1-12) Number 12 m Minute in hour Number 30 s
		 * Second in minute Number 55 S Millisecond Number 978 z Time zone
		 * General time zone Pacific Standard Time; PST; GMT-08:00 Z Time zone
		 * RFC 822 time zone -0800
		 */

	}
	
	
	/**
	 * Data scritta in formato unix_timestamp
	 * @param data
	 * @return
	 */
	public int toUnixTimestamp() {
		long epoch = this.getTimeInMillis()/1000;
		return (int)epoch;
	}
	
	/**
	 * Restituisce una istanza di calendario a partire da un unix_timestamp
	 * @param unixtimestamp
	 * @return
	 */
	public static CalendarUTC fromUnixTimestamp(long unixtimestamp) {
		return new CalendarUTC(unixtimestamp*1000);		
	}
	
	/**
	 * Restituisce una istanza di calendario a partire da un unix_timestamp
	 * @param unixtimestamp
	 * @return
	 */
	public static CalendarUTC fromUnixTimestamp(Long unixtimestamp) {
		return unixtimestamp!=null?new CalendarUTC(unixtimestamp*1000):null;		
	}
	
	
	
	
	

	public static CalendarUTC newInstance(Object valore) {
		if (valore == null)
			return null;

		if (valore instanceof Timestamp) {
			return newInstance((Timestamp) valore);
		} else if (valore instanceof java.util.Date) {
			return newInstance((java.util.Date) valore);
		} else if (valore instanceof java.sql.Date) {
			return newInstance((java.sql.Date) valore);
		} else if (valore instanceof Time) {
			return newInstance((Time) valore);
		} else if (valore instanceof Calendar) {
			return newInstance((Calendar) valore);
		} else if (valore instanceof Long) {
			return newInstance((Long) valore);
		}
		return null;
	}

	public static CalendarUTC newInstance(Long valore) {
		return valore != null ? new CalendarUTC(valore) : null;
	}

	public static CalendarUTC newInstance(Timestamp valore) {
		return valore != null ? new CalendarUTC(valore) : null;
	}

	public static CalendarUTC newInstance(java.util.Date valore) {
		return valore != null ? new CalendarUTC(valore) : null;
	}

	public static CalendarUTC newInstance(java.sql.Date valore) {
		return valore != null ? new CalendarUTC(valore) : null;
	}

	public static CalendarUTC newInstance(Time valore) {
		return valore != null ? new CalendarUTC(valore) : null;
	}

	public static CalendarUTC newInstance(Calendar valore) {
		return valore != null ? new CalendarUTC(valore) : null;
	}

	
	public static CalendarUTC newInstance(String valore, String formato) throws Exception {
		return XString.isNotBlankNullTrim(valore) ? new CalendarUTC(valore, formato) : null;
	}
	
	public static CalendarUTC newInstance(String valore, String[] formati) throws Exception {
		return XString.isNotBlankNullTrim(valore) ? new CalendarUTC(valore, formati) : null;
	}
	
	
	
	public int getYear() {
		return this.get(CalendarUTC.YEAR);
	}

	public int getMonth() {
		return this.get(CalendarUTC.MONTH) + 1;
	}

	public int getDay() {
		return this.get(CalendarUTC.DAY_OF_MONTH);
	}

	public int getHour() {
		return this.get(CalendarUTC.HOUR_OF_DAY);
	}

	public int getMinute() {
		return this.get(CalendarUTC.MINUTE);
	}

	public int getSecond() {
		return this.get(CalendarUTC.SECOND);
	}

	public int getWeek() {
		return this.get(CalendarUTC.DAY_OF_WEEK);
	}

	/**
	 * Restituisce una istanza di calendario con ore, minuti, secondi uguale a 0
	 * 
	 * @return
	 */
	public CalendarUTC getMidnight() {
		CalendarUTC appoggio = new CalendarUTC(this);
		appoggio.setHours(0, 0, 0);
		return appoggio;
	}

	public Timestamp getTimestamp() {
		return new Timestamp(this.getTimeInMillis());
	}

	/**
	 * Restituisce uan istanza di EnterpriseCalendar contenente soltanto il giorno
	 * 
	 * @return
	 */
	public CalendarUTC getDatePart() {
		return new CalendarUTC(this.getYear(), this.getMonth(), this.getDay());
	}

	public static CalendarUTC nvl(CalendarUTC valore1, CalendarUTC valore2) {
		if (valore1 != null)
			return valore1;
		return valore2;
	}

	
	public static String delayToString(long differenza) {
		long seconds = differenza/1000;
		long millis = differenza%1000;				
		
		long minutes = seconds/60;
		seconds = seconds%60;
		
		long hours = minutes/60;
		minutes = minutes%60;
		
		long days = hours/24;
		hours = hours%24;
			
		return (days>0?days+" days ":"") + 
			   (hours>0?hours+" hours ":"") +
			   (minutes>0?minutes+" minutes ":"") +
			   (seconds>0?seconds+" seconds ":"") +
			   (millis>0?millis+" milliseconds":"");
	}
	
	
	/**
	 * Fa in modo che una data in TimeZone GMT inserita in un calendario con
	 * TimeZone CEST sia vista correttmanete eliminanto il ZoneOffSet
	 * 
	 * @param origine
	 *            CST in formato GMT
	 * @return data in formato CST
	 */
	public static CalendarUTC alchemyGMT2CEST(Calendar origine) {
		if (origine == null)
			return (CalendarUTC) null;

		if (log.isTraceEnabled()) {
			SimpleDateFormat dateFormat = new SimpleDateFormat("dd/MM/yyyy HH:mm.ss,SSS Z");
			log.trace("EnterpriseCalendar source: "
					 + origine.get(CalendarUTC.YEAR)
					 + "-"
					 + XString.fillerZero(origine.get(CalendarUTC.MONTH) + 1, 2)
					 + "-"
					 + XString.fillerZero(origine.get(CalendarUTC.DAY_OF_MONTH),2)
					 + "T"
					 + XString.fillerZero(origine.get(CalendarUTC.HOUR_OF_DAY), 2)
					 + ":"
					 + XString.fillerZero(origine.get(CalendarUTC.MINUTE), 2)
					 + "."
					 + XString.fillerZero(origine.get(CalendarUTC.SECOND), 2)
					 + " offset " 
					 + origine.get(CalendarUTC.ZONE_OFFSET) 
					 + " << "
					 + dateFormat.format(origine.getTime()));
			
		}
		origine.set(CalendarUTC.ZONE_OFFSET, 0);

		CalendarUTC destinazione = new CalendarUTC(origine.get(CalendarUTC.YEAR)
				                               , origine.get(CalendarUTC.MONTH) + 1
				                               , origine.get(CalendarUTC.DAY_OF_MONTH)
				                               , origine.get(CalendarUTC.HOUR_OF_DAY)
				                               , origine.get(CalendarUTC.MINUTE)
				                               , origine.get(CalendarUTC.SECOND));
		if (log.isTraceEnabled()) {
			log.trace("EnterpriseCalendar target: "
					 + destinazione.get(CalendarUTC.YEAR)
					 + "-"
					 + XString.fillerZero(destinazione.get(CalendarUTC.MONTH) + 1, 2)
					 + "-"
					 + XString.fillerZero(destinazione.get(CalendarUTC.DAY_OF_MONTH), 2)
					 + "T"
					 + XString.fillerZero(destinazione.get(CalendarUTC.HOUR_OF_DAY),2) 
					 + ":"
					 + XString.fillerZero(destinazione.get(CalendarUTC.MINUTE), 2)
					 + "."
					 + XString.fillerZero(destinazione.get(CalendarUTC.SECOND), 2)
					 + " offset " 
					 + destinazione.get(CalendarUTC.ZONE_OFFSET)
					 + " >> " 
					 + destinazione.format("dd/MM/yyyy HH:mm.ss,SSS Z"));
		}
		return destinazione;
	}
}