package it.softecspa.kahuna.util;

import it.softecspa.kahuna.io.File;
import it.softecspa.kahuna.lang.XString;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.Enumeration;

@SuppressWarnings("serial")
public class Properties implements Serializable {

	private static final int PROPERTIES = 0;
	private static final int XML = 1;
	private int formato;

	private File file;
	private InputStream input; // appoggio!
	private boolean loaded;

	private java.util.Properties prop;
	private PropertiesCheck checklist;

	
	public Properties() {
		prop = new java.util.Properties();
	}

	public Properties(java.util.Properties defaults) {
		prop = defaults;
		loaded = false;
	}

	public Properties(Properties defaults) {
		prop = defaults.getProperties();
		loaded = false;
	}

	
	protected void changeProperties(Properties prop) {
		this.prop = prop.getProperties();
		this.file = prop.getFile();
		this.loaded = prop.loaded;
	}
	
	public java.util.Properties getProperties() {
		return prop;
	}

	/**
	 * Esegue il merge dei due file di properties
	 * In caso di "doppioni", i valori del secondo hanno la precedenza su quelli del primo e quindi lo sovrascrivono
	 * @param source Properties sorgente 
	 */
	public void merge(Properties winner) {
		if (winner==null) return;
		
		for (Enumeration<Object> keys = winner.keys(); keys.hasMoreElements();) {
			String key = (String)keys.nextElement();
			String value = winner.get(key);
			if (value!=null) prop.setProperty(key, value);
		}
	}
	
	/**
	 * Ricarica le propriet dal file
	 * 
	 * @return
	 * @throws IOException
	 * @throws FileNotFoundException
	 */
	public synchronized void reload() throws IOException {
		prop.clear();
		loadAll();
	}

	public synchronized void clear() {
		prop.clear();
		loaded = false;
	}

	public void load(String filename) throws IOException {
		load(filename, null);
	}

	public void load(String filename, PropertiesCheck checklist) throws IOException {
		if (XString.isBlankNull(filename))
			throw new IOException("@param filename null or not indicated");
		load(new File(filename), checklist);
	}

	public void load(File file) throws IOException {
		load(file, null);
	}

	public void load(File file, PropertiesCheck checklist) throws IOException {
		load(file, checklist, PROPERTIES);
	}

	public void loadFromXML(String filename) throws IOException {
		loadFromXML(filename, null);
	}

	public void loadFromXML(String filename, PropertiesCheck checklist) throws IOException {
		if (XString.isBlankNull(filename))
			throw new IOException("@param filename null or not indicated");
		loadFromXML(new File(filename), checklist);
	}

	public void loadFromXML(File file) throws IOException {
		loadFromXML(file, null);
	}

	public void loadFromXML(File file, PropertiesCheck checklist) throws IOException {
		load(file, checklist, XML);
	}

	public synchronized void loadFromXML(InputStream input) throws IOException {
		this.input = input;
		this.formato = XML;

		loadAll();
	}

	private synchronized void load(File file, PropertiesCheck checklist, int formato) throws IOException {
		this.file = file;
		if (this.checklist != null)
			this.checklist = checklist;
		this.formato = formato;

		loadAll();
	}

	private synchronized void loadAll() throws IOException {
		if (file == null && input == null)
			throw new IOException("@param file not indicated");
		
		try {
			if (input == null)
				input = new FileInputStream(file);

			if (formato == XML) {
				// XMLUtils.load(prop, input);
				prop.loadFromXML(input);

			} else if ("xml".equalsIgnoreCase(file.getExtension())) {
				formato = XML;
				// XMLUtils.load(prop, input);
				prop.loadFromXML(input);

			} else {
				// Comune file di properties
				formato = PROPERTIES;
				prop.load(input);
			}
			loaded = true;

		} catch (IOException e) {
			throw e;

		} catch (Exception e) {
			throw new IOException(e.toString());

		} finally {
			try {
				if (input != null)
					input.close();
			} catch (IOException e) {/* nulla */
			}
			input = null;
		}

		if (loaded && checklist != null) {
			checklist.check(this);
		}
	}

	public boolean checked() {
		return checklist != null ? checklist.checked() : true;
	}

	public void store(PrintStream out) throws IOException {
		store(out, null);
	}

	public void store(String commento) throws IOException {
		store(System.out, commento);
	}

	public void store() throws IOException {
		store(System.out, null);
	}

	private synchronized void store(PrintStream out, String commento) throws IOException {
		FileOutputStream output = null;
		try {
			output = new FileOutputStream(this.getFile());
			if (formato == PROPERTIES) {
				prop.store(output, commento);
			} else {
				// XMLUtils.save(prop,output,commento,"UTF-8");
				prop.storeToXML(output, commento);
			}
			output.close();

		} catch (IOException e) {
			throw e;
		} finally {
			try {
				if (output != null)
					output.close();
			} catch (IOException e) {/* nulla */
			}
		}
	}

	private static String toNull(String valore) {
		return (XString.isBlankNullTrim(valore) ? null : valore);
	}

	public synchronized boolean isNull(String name) {
		return toNull(prop.getProperty(name)) == null;
	}

	public String get(String name) {
		return get(name, null);
	}

	public synchronized String get(String name, String defaults) {
		String valore = toNull(prop.getProperty(name));
		if (valore == null) {
			if (defaults == null)
				return null;
			prop.setProperty(name, defaults);
		}
		return prop.getProperty(name);
	}

	public synchronized String get(String name, String defaults, C007 decoder) {
		String valore = toNull(prop.getProperty(name));
		if (valore == null && defaults != null)
			this.put(name, defaults, decoder);
		return decoder.decodifica(prop.getProperty(name));
	}

	public int getInt(String name) {
		return getInt(name, 0);
	}

	public synchronized int getInt(String name, int defaults) {
		try {
			return Integer.parseInt(this.get(name, Integer.toString(defaults)));
		} catch (Exception e) {
			prop.setProperty(name, Integer.toString(defaults));
			return Integer.parseInt(prop.getProperty(name));
		}
	}

	public long getLong(String name) {
		return getLong(name, 0);
	}

	public synchronized long getLong(String name, long defaults) {
		try {
			return Long.parseLong(this.get(name, Long.toString(defaults)));
		} catch (Exception e) {
			prop.setProperty(name, Long.toString(defaults));
			return Long.parseLong(prop.getProperty(name));
		}
	}

	public double getDouble(String name) {
		return getDouble(name, 0);
	}

	public synchronized double getDouble(String name, double defaults) {
		try {
			return Double.parseDouble(this.get(name, Double.toString(defaults)));
		} catch (Exception e) {
			prop.setProperty(name, Double.toString(defaults));
			return Double.parseDouble(prop.getProperty(name));
		}
	}

	public boolean getBoolean(String name) {
		return getBoolean(name, false);
	}

	public synchronized boolean getBoolean(String name, boolean defaults) {
		String valore = get(name);
		if (valore != null) {
			if (valore.equalsIgnoreCase("true") || 
				valore.equalsIgnoreCase("yes") || 
				valore.equalsIgnoreCase("1"))
				return true;
			if (valore.equalsIgnoreCase("false") || 
				valore.equalsIgnoreCase("no") || 
				valore.equalsIgnoreCase("0"))
				return false;			
		}
		prop.setProperty(name, defaults ? "true" : "false");
		return defaults;
	}

	public synchronized void put(String name, String valore, C007 encoder) {
		if (valore == null)
			valore = "";
		prop.setProperty(name, encoder.codifica(valore));
	}

	public synchronized void put(String name, String valore) {
		if (valore == null)
			valore = "";
		prop.setProperty(name, valore);
	}

	public synchronized void put(String name, int valore) {
		prop.setProperty(name, Integer.toString(valore));
	}

	public synchronized void put(String name, long valore) {
		prop.setProperty(name, Long.toString(valore));
	}

	public synchronized void put(String name, double valore) {
		prop.setProperty(name, Double.toString(valore));
	}

	public synchronized void put(String name, boolean valore) {
		prop.setProperty(name, valore ? "true" : "false");
	}

	public String getName() {
		return (file != null ? file.getName() : null);
	}

	public String getParent() {
		return (file != null ? file.getParent() : null);
	}

	public String getAbsolutePath() {
		return (file != null ? file.getAbsolutePath() : null);
	}

	public File getFile() {
		return file;
	}

	public String toString() {
		return prop != null ? prop.toString() : null;
	}

	public Enumeration<Object> keys() {
		return prop != null ? prop.keys() : null;
	}

	public void list() {
		list(System.out);
	}

	public void list(PrintStream out) {
		out.println("* --- Properties list ----------------------------------------------------- *");
		prop.list(out);
		out.println("* ------------------------------------------------------------------------- *");
	}

	public void list(PrintWriter out) {
		out.println("* --- Properties list ----------------------------------------------------- *");
		prop.list(out);
		out.println("* ------------------------------------------------------------------------- *");
	}

	public boolean isLoaded() {
		return loaded;
	}

}
