package it.softecspa.kahuna.util.calendar;

import it.softecspa.kahuna.lang.XString;

import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;

@SuppressWarnings("serial")
public class EnterpriseCalendar extends KahunaCalendar<EnterpriseCalendar> {
	
	public EnterpriseCalendar() {
		super(EU);
	}

	public EnterpriseCalendar(int year, int month, int day) {
		super(EU, year, month, day, EU);
	}

	public EnterpriseCalendar(int year, int month, int day, int hour, int minute) {
		super(EU, year, month, day, hour, minute, EU);
	}

	public EnterpriseCalendar(int year, int month, int day, int hour, int minute, int second) {
		super(EU, year, month, day, hour, minute, second, EU);
	}

	
	public EnterpriseCalendar(String value) throws Exception {
		this(value, PATTERNS_UTC);		
	}
	
	public EnterpriseCalendar(String value, String ... patterns) throws Exception {
		super(EU, value, patterns);
	}
		
	public EnterpriseCalendar(Timestamp data) {
		super(EU, data);
	}	

	public EnterpriseCalendar(java.util.Date data) {
		super(EU, data);
	}
	
	public EnterpriseCalendar(java.sql.Date data) {
		super(EU, data);
	}

	public EnterpriseCalendar(Calendar calendar) {
		super(EU, calendar);
	}
	
	public EnterpriseCalendar(long millis) {
		super(EU, millis);
	}
	
	
	public static EnterpriseCalendar now() {
		return new EnterpriseCalendar();
	}
	
	
	public static EnterpriseCalendar today() {
		EnterpriseCalendar adesso = EnterpriseCalendar.now();
		return new EnterpriseCalendar(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH), 0, 0, 0);
	}
	
	public static EnterpriseCalendar tonight() {
		EnterpriseCalendar adesso = EnterpriseCalendar.now();
		return new EnterpriseCalendar(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH), 23, 59, 29);
	}
	
	public static EnterpriseCalendar yesterday() {
		EnterpriseCalendar adesso = EnterpriseCalendar.now();
		return new EnterpriseCalendar(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH) - 1, 0, 0, 0);
	}

	public static EnterpriseCalendar tomorrow() {
		EnterpriseCalendar adesso = EnterpriseCalendar.now();
		return new EnterpriseCalendar(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH) + 1, 0, 0, 0);
	}

	
	public static EnterpriseCalendar futureDay(int days) {
		EnterpriseCalendar adesso = EnterpriseCalendar.now();
		return new EnterpriseCalendar(adesso.get(YEAR), adesso.get(MONTH) + 1, adesso.get(DAY_OF_MONTH) + days, 0, 0, 0);
	}

	/**
	 * Restituisce il giorno di Pasqua per l'anno impostato
	 * 
	 * @param int anno
	 * @return EnterpriseCalendar
	 */
	public static EnterpriseCalendar easter(int anno) {
		/*
		 * Un metodo per il calcolo della data di Pasqua lo si trova descritto
		 * nel libro di Jean Meeus
		 * "Astronomia con il computer - Formule, metodi di calcolo, esempi numerici"
		 * - ed.Hoepli - (Lire 25000 nel 1990 - libro molto interessante -
		 * titolo originale dell'opera "Astronomical Formul for Calculators" )
		 * ove si dice, a proposito del calcolo della data di Pasqua: Il medodo
		 * qui descritto  stato pubblicato da Spencer Jones nel suo libro
		 * General Astronomy (pagg.73-74 dell'edizione del 1922).  stato
		 * ripubblicato nel Journal of the British Astronomical Association,
		 * vol.88, pag.91 (dicembre 1977), dove si dice che fu sviluppato nel
		 * 1876 e che comparve nell'Ecclesiastic Calendar di Butcher. A
		 * differenza della formula data da Gauss, questo metodo non ha
		 * eccezioni ed  valido per tutti gli anni nel EnterpriseCalendar Gregoriano,
		 * ossia dal 1583 in poi. La procedura per determinare la data della
		 * Pasqua  la seguente:
		 * 
		 * dividere per quoziente resto
		 * 
		 * l'anno x 19 - a l'anno x 100 b c b 4 d e b+8 25 f - b-f+1 3 g -
		 * 19a+b-d-g+15 30 - h c 4 i k 32+2e+2i-h-k 7 - l a+11h-22l 451 m -
		 * h+l-7m+114 31 n p
		 * 
		 * Allora n = numero del mese (3=marzo, 4=aprile) p+1 = giorno del mese
		 * in cui cade la domenica di Pasqua (...omissis...) Le date estreme
		 * della Pasqua sono Marzo 22 (come nel 1818 e nel 2285) ed Aprile 25
		 * (come nel 1886, nel 1943 e nel 2038) Copyright  1998 G.Paolo Bizzaro
		 */

		EnterpriseCalendar pasqua = new EnterpriseCalendar();
		int a, b, c, p, q, r;
		a = anno % 19;
		b = anno / 100;
		c = anno % 100;
		p = (19 * (a) + b - (b / 4) - ((b - ((b + 8) / 25) + 1) / 3) + 15) % 30;
		q = (32 + 2 * ((b % 4) + (c / 4)) - p - (c % 4)) % 7;
		r = (p + q - 7 * ((a + 11 * p + 22 * q) / 451) + 114);
		pasqua.set(anno, (r / 31) - 1, (r % 31) + 1, 0, 0, 0);
		return pasqua;
	}

	
	
	public EnterpriseCalendar getMidnight() {
		EnterpriseCalendar appoggio = new EnterpriseCalendar(this);
		appoggio.setHours(0, 0, 0);
		return appoggio;
	}
	
	public EnterpriseCalendar get235959() {
		EnterpriseCalendar appoggio = new EnterpriseCalendar(this);
		appoggio.setHours(23, 59, 59);
		return appoggio;
	}

	
	public boolean isEaster() {
		EnterpriseCalendar pasqua = EnterpriseCalendar.easter(this.get(YEAR));
		return this.get(MONTH) == pasqua.get(MONTH)
				&& this.get(DAY_OF_MONTH) == pasqua.get(DAY_OF_MONTH);
	}

	
	/**
	 * Restituisce il primo giorno dell'anno per l'anno impostato
	 * 
	 * @param int anno
	 * @return EnterpriseCalendar
	 */
	public static EnterpriseCalendar newYear(int anno) {
		return new EnterpriseCalendar(anno, 1, 1, 0, 0, 0);
	}

	/**
	 * Restituisce il giorno di Natale per l'anno impostato
	 * 
	 * @param int anno
	 * @return EnterpriseCalendar
	 */
	public static EnterpriseCalendar christmasDay(int anno) {
		return new EnterpriseCalendar(anno, 12, 25, 0, 0, 0);
	}

	public boolean isNewYear() {
		return this.get(MONTH) == JANUARY && this.get(DAY_OF_MONTH) == 1;
	}

	public boolean isChristmasDay() {
		return this.get(MONTH) == DECEMBER && this.get(DAY_OF_MONTH) == 25;
	}

	
	public EnterpriseCalendar getLastDayOfMonth() {
		return lastDayOfMonth(this.get(YEAR), this.get(MONTH) + 1);
	}

	public static EnterpriseCalendar lastDayOfMonth(int anno, int mese) {
		EnterpriseCalendar data = new EnterpriseCalendar(anno, mese, 1, 0, 0, 0);
		int giorno = data.getActualMaximum(DAY_OF_MONTH);
		data.set(DAY_OF_MONTH, giorno);
		return data;
	}
	

	public EnterpriseCalendar getFirstDayOfMonth() {
		return firstDaOfMonth(this.get(YEAR), this.get(MONTH) + 1);
	}

	public static EnterpriseCalendar firstDaOfMonth(int anno, int mese) {
		EnterpriseCalendar data = new EnterpriseCalendar(anno, mese, 1, 0, 0, 0);
		return data;
	}

	
	/*
	public static boolean isFestivo(GregorianCalendar data) {
		// La Domenica  giorno festivo
		int week = data.get(DAY_OF_WEEK);
		if (week == SUNDAY)
			return true;

		// Verifica festivit canonica
		int mese = data.get(MONTH) + 1;
		int giorno = data.get(DAY_OF_MONTH);
		for (int i = 0; i < festiviITA.length; i++) {
			if (festiviITA[i][0] == mese) {
				if (festiviITA[i][1] == giorno) {
					return true;
				}
			}
		}

		// Verifica Pasqua
		Kalendario pasqua = getPasqua(data.get(YEAR));
		if (pasqua.getMonth() == mese) {
			if (pasqua.getDay() == giorno) {
				return true;
			}
		}

		// Verifica Luned dell'Angelo
		pasqua.add(DAY_OF_MONTH, 1);
		if (pasqua.getMonth() == mese) {
			if (pasqua.getDay() == giorno) {
				return true;
			}
		}

		return false;
	}

	public boolean isFestivo() {
		return isFestivo(this);
	}
	 
	public static boolean isFeriale(GregorianCalendar data) {
		return !isFestivo(data);
	}

	public boolean isFeriale() {
		return !isFestivo(this);
	}
	*/

	/* *
	 * Restituisce TRUE se il giorno indicato  lavorativo Il Sabato e la
	 * Domenica non si lavora! Per il calcolo sono utilizzate le festivit del
	 * calendario italiano
	 * 
	 * @param data
	 * @return
	 * /
	public static boolean isLavorativo(GregorianCalendar data) {
		int week = data.get(DAY_OF_WEEK);

		// Se Sabato non si lavora
		if (week == SATURDAY)
			return false;

		return !isFestivo(data);
	}

	public boolean isLavorativo() {
		return isLavorativo(this);
	}
	*/

	/* *
	 * Restituisce il primo giorno lavorativo dopo la data specificata
	 * 
	 * @param data
	 * @return
	 * /
	public static Kalendario getFirstLavorativoAfter(GregorianCalendar data) {
		Kalendario confronto = newInstance(data);
		if (confronto == null)
			return null;

		do {
			confronto.add(DAY_OF_MONTH, 1);
			// int week = confronto.getWeek();
			// if (week!=SUNDAY) {confronto.add(DAY_OF_MONTH, 1); continue;}
			// if (week!=SATURDAY) {confronto.add(DAY_OF_MONTH, 1); continue;}
			if (!confronto.isLavorativo())
				continue;

			return confronto;
		} while (true);
	}

	public Kalendario getFirstLavorativoAfter() {
		return getFirstLavorativoAfter(this);
	}
	*/

	
	/**
	 * Descrizione del'intervallo
	 * 
	 * @param secondi
	 * @return
	 */
	public static String descrizioneIntervallo(long secondi) {
		// Calcolo della durata esperessa in secondi
		String durata = "";
		if (secondi == 0) {
			durata += "<1 secondo";
		} else if (secondi == 1) {
			durata += secondi + " secondo";
		} else if (secondi < 60) {
			durata += secondi + " secondi";
		} else {
			long mm = (secondi / 60);
			long ss = (secondi % 60);
			if (mm == 1) {
				durata += mm + " minuto";
			} else if (mm < 60) {
				durata += mm + " minuti";
			} else {
				long hh = (mm / 60);
				mm = mm % 60;
				durata += hh + " ore, " + mm + " minut" + (mm == 1 ? "o" : "i");
			}
			durata += " e " + ss + " second" + (ss == 1 ? "o" : "i");
		}
		return durata;
	}

	
		
	/**
	 * Restituisce una istanza di calendario a partire da un unix_timestamp
	 * @param unixtimestamp
	 * @return
	 */
	public static EnterpriseCalendar fromUnixTimestamp(long unixtimestamp) {
		return new EnterpriseCalendar(unixtimestamp*1000);		
	}
	
	/**
	 * Restituisce una istanza di calendario a partire da un unix_timestamp
	 * @param unixtimestamp
	 * @return
	 */
	public static EnterpriseCalendar fromUnixTimestamp(Long unixtimestamp) {
		return unixtimestamp!=null?new EnterpriseCalendar(unixtimestamp*1000):null;		
	}
	
	
		
	
	public static EnterpriseCalendar newInstance(Object valore) {
		if (valore == null)
			return null;

		if (valore instanceof Timestamp) {
			return newInstance((Timestamp) valore);
		} else if (valore instanceof java.util.Date) {
			return newInstance((java.util.Date) valore);
		} else if (valore instanceof java.sql.Date) {
			return newInstance((java.sql.Date) valore);
		} else if (valore instanceof Time) {
			return newInstance((Time) valore);
		} else if (valore instanceof Calendar) {
			return newInstance((Calendar) valore);
		} else if (valore instanceof Long) {
			return newInstance((Long) valore);
		} else if (valore instanceof String) {
			try {
				return newInstance((String) valore, PATTERNS_UTC);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		} 
		
		return null;
	}
	

	public static EnterpriseCalendar newInstance(Long valore) {
		return valore != null ? new EnterpriseCalendar(valore) : null;
	}

	public static EnterpriseCalendar newInstance(Timestamp valore) {
		return valore != null ? new EnterpriseCalendar(valore) : null;
	}

	public static EnterpriseCalendar newInstance(java.util.Date valore) {
		return valore != null ? new EnterpriseCalendar(valore) : null;
	}

	public static EnterpriseCalendar newInstance(java.sql.Date valore) {
		return valore != null ? new EnterpriseCalendar(valore) : null;
	}

	public static EnterpriseCalendar newInstance(Time valore) {
		return valore != null ? new EnterpriseCalendar(valore) : null;
	}

	public static EnterpriseCalendar newInstance(Calendar valore) {
		return valore != null ? new EnterpriseCalendar(valore) : null;
	}

	/*
	public static EnterpriseCalendar newInstance(String valore, String formato) throws Exception {
		return XString.isNotBlankNullTrim(valore) ? new EnterpriseCalendar(valore, formato) : null;
	}
	*/
	
	public static EnterpriseCalendar newInstance(String valore, String ... formati) throws Exception {
		return XString.isNotBlankNullTrim(valore) ? new EnterpriseCalendar(valore, formati) : null;
	}
	
	
	



	/**
	 * Restituisce uan istanza di EnterpriseCalendar contenente soltanto il giorno
	 * 
	 * @return
	 */
	public EnterpriseCalendar getDatePart() {
		return new EnterpriseCalendar(this.getYear(), this.getMonth(), this.getDay());
	}


	
	public static EnterpriseCalendar nvl(EnterpriseCalendar valore1, EnterpriseCalendar valore2) {
		if (valore1 != null)
			return valore1;
		return valore2;
	}

	
	public static String delayToString(long differenza) {
		long seconds = differenza/1000;
		long millis = differenza%1000;				
		
		long minutes = seconds/60;
		seconds = seconds%60;
		
		long hours = minutes/60;
		minutes = minutes%60;
		
		long days = hours/24;
		hours = hours%24;
			
		return (days>0?days+" days ":"") + 
			   (hours>0?hours+" hours ":"") +
			   (minutes>0?minutes+" minutes ":"") +
			   (seconds>0?seconds+" seconds ":"") +
			   (millis>0?millis+" milliseconds":"");
	}
	
	
	public EnterpriseCalendarUTC getCalendarUTC() {		
		return new EnterpriseCalendarUTC(this);
	}
	
	public EnterpriseCalendarIT getCalendarIT() {		
		return new EnterpriseCalendarIT(this);
	}
	
}