package it.softecspa.kahuna.util.calendar;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.log.KLogger;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

@SuppressWarnings("serial")
public abstract class KahunaCalendar<K> extends GregorianCalendar {

	static KLogger log = KLogger.getLogger(KahunaCalendar.class);

	public static final int DAY = 255;

	
	/* Date format pattern used to parse HTTP date headers in RFC 1123 format */
	public static final String PATTERN_RFC1123 = "EEE, dd MMM yyyy HH:mm:ss zzz";

	/* Date format pattern used to parse HTTP date headers in RFC 1036 format */
	public static final String PATTERN_RFC1036 = "EEEE, dd-MMM-yy HH:mm:ss zzz";
	
	/* Date format pattern used to parse HTTP date headers in ANSI C <code>asctime()</code> format */
	public static final String PATTERN_ASCTIME = "EEE MMM d HH:mm:ss yyyy";
	
	public static final String PATTERN_UTC_TIMEZONE = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
	public static final String PATTERN_UTC_ZULU = "yyyy-MM-dd'T'HH:mm:ss'Z'";
	
	public static final String[] PATTERNS_UTC = {PATTERN_UTC_ZULU,PATTERN_UTC_TIMEZONE} ;
	
	
	
	static public final TimeZone GMT 	= TimeZone.getTimeZone("GMT");
	static public final TimeZone UTC 	= TimeZone.getTimeZone("UTC");
	static public final TimeZone EU 	= TimeZone.getTimeZone("Europe/Copenhagen");
		
	
	
	KahunaCalendar(TimeZone zone) {
		super();
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}

	KahunaCalendar(TimeZone zone, int year, int month, int day, TimeZone origine) {
		super();
		if (origine!=null) {
			// Gestione deltimezone di origine del dato
			setTimeZone(origine);
		}
		setDay(year, month, day);			
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}

	KahunaCalendar(TimeZone zone, int year, int month, int day, int hour, int minute, TimeZone origine) {
		this(zone, year, month, day, hour, minute, 0, origine);
	}

	KahunaCalendar(TimeZone zone, int year, int month, int day, int hour, int minute, int second, TimeZone origine) {
		super();
		if (origine!=null) {
			// Gestione del timezone di origine del dato
			setTimeZone(origine);
		}
		setDay(year, month, day);
		setHours(hour, minute, second);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}

	/*
	KahunaCalendar(TimeZone zone, String value, String pattern) throws Exception {
		this(zone, value, new String[] {pattern}, null);		
	}
	*/
		
	KahunaCalendar(TimeZone zone, String value, String ... patterns) throws Exception {
		this(zone, value, null, patterns);
	}
	
	private KahunaCalendar(TimeZone zone, String value, Locale locale, String ... patterns) throws Exception {
		super();
		
		if (XString.isBlankNullTrim(value)) {
			throw new NullPointerException();
		}
	
		// Autocorrezione
		if (value.length() > 1 && value.startsWith("'") && value.endsWith("'")) {
			value = value.substring(1, value.length() - 1);
		}
		
		
		java.util.Date date = null;
		for (String format : patterns) {
			SimpleDateFormat dateParser = null;
			
			if (format.equals(PATTERN_RFC1036)) {
				dateParser = new SimpleDateFormat(PATTERN_RFC1036, Locale.US);
				dateParser.setTimeZone(GMT);				
			
			} else if (format.equals(PATTERN_RFC1123)) {
				dateParser = new SimpleDateFormat(PATTERN_RFC1123, Locale.US);
				dateParser.setTimeZone(GMT);
			
			} else if (format.equals(PATTERN_UTC_TIMEZONE)) {
				dateParser = new SimpleDateFormat(PATTERN_UTC_TIMEZONE);
				
			} else if (format.equals(PATTERN_UTC_ZULU)) {
				dateParser = new SimpleDateFormat(PATTERN_UTC_ZULU);
				dateParser.setTimeZone(UTC);
				
			} else {
				if (locale!=null) {
					dateParser = new SimpleDateFormat(format, locale);
				} else {
					dateParser = new SimpleDateFormat(format);
				}
			}
			
			try {
				date = dateParser.parse(value);	
				break;
			} catch (ParseException e) {
				// ignore this exception, we will try the next format
			}
		}
		
		if (date==null) {
			String list = "";
			for (String format : patterns) {
				list += (list.length()>0?",":"")+"{"+format+"}";
			}
			throw new Exception("No valid patters in list: "+list);
		}
		
		setTime(date);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}	
	

	KahunaCalendar(TimeZone zone, Timestamp data) {
		super();
		super.setTime(data);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
		
	}
	

	KahunaCalendar(TimeZone zone, java.util.Date data) {
		super();
		super.setTime(data);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}
	

	KahunaCalendar(TimeZone zone, java.sql.Date data) {
		super();
		super.setTime(data);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}
	

	KahunaCalendar(TimeZone zone, Calendar calendar) {
		super();
		super.setTime(calendar.getTime());
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}

	
	KahunaCalendar(TimeZone zone, long millis) {
		super();
		super.setTimeInMillis(millis);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}

	
	KahunaCalendar(TimeZone zone, Locale locale) {
		super(locale);
		// Conversione del calendario nel TimeZone desiderato
		setTimeZone(zone);
	}
	
	
	/*
	public void setDay(Calendar giorno) {
		this.set(DAY_OF_MONTH, 1);
		this.set(YEAR, giorno.get(YEAR));
		this.set(MONTH, giorno.get(MONTH));
		this.set(DAY_OF_MONTH, giorno.get(DAY_OF_MONTH));
	}
	*/

	public void setDay(int year, int month, int day) {
		this.set(DAY_OF_MONTH, 1);
		this.set(YEAR, year);
		this.set(MONTH, month - 1);
		this.set(DAY_OF_MONTH, day);
	}

	public void setHours(int hour, int minute, int second) {
		this.set(HOUR_OF_DAY, hour);
		this.set(MINUTE, minute);
		this.set(SECOND, second);
		this.set(MILLISECOND, 0);
	}

	
	/**
	 * Formatta la data utilizzando la classe   {@link java.text.SimpleDataFormat}
	 * <table>
	 *  <tr><td><b>Letter</b></td><td><b>Date or Time Component</b></td><td><b>Presentation</b></td><td><b>Examples</b></td></tr>
	 *  <tr><td><b>G</b></td><td>Era designator      </td><td>Text   </td><td>AD            </td></tr>
	 *  <tr><td><b>y</b></td><td>Year                </td><td>Year   </td><td>1996; 96      </td></tr>
	 *  <tr><td><b>M</b></td><td>Month in year       </td><td>Month  </td><td>July; Jul; 07 </td></tr>
	 *  <tr><td><b>w</b></td><td>Week in year        </td><td>Number </td><td>27            </td></tr>
	 *  <tr><td><b>W</b></td><td>Week in month       </td><td>Number </td><td>2             </td></tr>
	 *  <tr><td><b>D</b></td><td>Day in year         </td><td>Number </td><td>189           </td></tr>
	 *  <tr><td><b>d</b></td><td>Day in month        </td><td>Number </td><td>10            </td></tr>
	 *  <tr><td><b>F</b></td><td>Day of week in month</td><td>Number </td><td>2             </td></tr>
	 *  <tr><td><b>E</b></td><td>Day in week         </td><td>Text   </td><td>Tuesday; Tue  </td></tr>
	 *  <tr><td><b>a</b></td><td>Am/pm marker        </td><td>Text   </td><td>PM            </td></tr>
	 *  <tr><td><b>H</b></td><td>Hour in day (0-23)  </td><td>Number </td><td>0             </td></tr>
	 *  <tr><td><b>k</b></td><td>Hour in day (1-24)  </td><td>Number </td><td>24            </td></tr>
	 *  <tr><td><b>K</b></td><td>Hour in am/pm (0-11)</td><td>Number </td><td>0             </td></tr>
	 *  <tr><td><b>h</b></td><td>Hour in am/pm (1-12)</td><td>Number </td><td>12            </td></tr>
	 *  <tr><td><b>m</b></td><td>Minute in hour      </td><td>Number </td><td>30            </td></tr>
	 *  <tr><td><b>s</b></td><td>Second in minute    </td><td>Number </td><td>55            </td></tr>
	 *  <tr><td><b>S</b></td><td>Millisecond         </td><td>Number </td><td>978           </td></tr>
	 *  <tr><td><b>z</b></td><td>Time zone           </td><td colsapn=2>General time zone  Pacific Standard Time; PST; GMT-08:00 </td></tr>
	 *  <tr><td><b>Z</b></td><td>Time zone           </td><td colsapn=2>RFC 822 time zone  -0800</td></tr>
	 *  <tr><td><b>Z</b></td><td>RFC1123           	 </td><td colsapn=2>RFC1123 "EEEE, dd-MMM-yy HH:mm:ss zzz"</td></tr>
	 *  <tr><td><b>Z</b></td><td>RFC1036           	 </td><td colsapn=2>RFC1036 "EEEE, dd-MMM-yy HH:mm:ss zzz"</td></tr>
	 *  <tr><td><b>Z</b></td><td>ASC time			 </td><td colsapn=2>ANSI C ASc time "EEE MMM d HH:mm:ss yyyy"</td></tr>
	 * </table>  
	 * 
	 * Formato standard italia: dd/MM/yyyy'&nbsp;'HH:mm.ss
	 */
	public String format(String formato) {
		SimpleDateFormat dateFormat = new SimpleDateFormat(formato);
		return dateFormat.format(this.getTime());
	}
	
	
	public String format(String pattern, Locale locale) {
		SimpleDateFormat formatter = new SimpleDateFormat(pattern, locale);
		return formatter.format(this.getTime());
	}
	
	public String format(String pattern, Locale locale, TimeZone zone) {
		SimpleDateFormat formatter = new SimpleDateFormat(pattern, locale);
		formatter.setTimeZone(zone);
		return formatter.format(this.getTime());
	}

	/**
	 * Formatta la data secondo lo standard RFC1123
	 * @return Una stringa con la data formattata
	 */
	public String formatRFC1123() {
		return format(PATTERN_RFC1123, Locale.US, GMT);
	}
	
	/**
	 * Formatta la data secondo lo standard RFC1036
	 * @return Una stringa con la data formattata
	 */
	public String formatRFC1036() {
		return format(PATTERN_RFC1036, Locale.US, GMT);
	}
	
	
	
	public String formatISO8601zulu() {
		return format(PATTERN_UTC_ZULU, Locale.ENGLISH, UTC);
	}

	public String formatISO8601withTimezone() {
		return format(PATTERN_UTC_TIMEZONE);
	}
	
	
	public String toStringUTC() {
		return toStringUTC(" ");
	}
	
	public String toStringUTC(String blank) {
		return format("yyyy-MM-dd'"+blank+"'HH:mm:ss'"+blank+"UTC'", Locale.ENGLISH, UTC);
	}
		
	
	
	
	public abstract K getMidnight();

	
	
	
	public long difference(Calendar confronto) {
		return difference(confronto, DAY);
	}

	public long difference(Calendar confronto, int field) {
		if (field == MILLISECOND)
			return getTimeInMillis() - confronto.getTimeInMillis();

		long anni = this.get(YEAR) - confronto.get(YEAR);
		if (field == YEAR)
			return anni;

		if (field == MONTH) {
			return (anni * 12) + this.get(MONTH) - confronto.get(MONTH);
		}

		long diff = (anni * 365) + this.get(DAY_OF_YEAR)
				- confronto.get(DAY_OF_YEAR);
		if (field == DAY_OF_YEAR || field == DAY_OF_MONTH || field == DAY)
			return diff;

		diff = (diff * 24) + this.get(HOUR_OF_DAY) - confronto.get(HOUR_OF_DAY);
		if (field == HOUR || field == HOUR_OF_DAY)
			return diff;

		diff = (diff * 60) + this.get(MINUTE) - confronto.get(MINUTE);
		if (field == MINUTE)
			return diff;

		diff = (diff * 60) + this.get(SECOND) - confronto.get(SECOND);
		if (field == SECOND)
			return diff;

		return 0;
	}

	
	public String toString() {
		/*
		 * Letter Date or Time Component Presentation Examples G Era designator
		 * Text AD y Year Year 1996; 96 M Month in year Month July; Jul; 07 w
		 * Week in year Number 27 W Week in month Number 2 D Day in year Number
		 * 189 d Day in month Number 10 F Day of week in month Number 2 E Day in
		 * week Text Tuesday; Tue a Am/pm marker Text PM H Hour in day (0-23)
		 * Number 0 k Hour in day (1-24) Number 24 K Hour in am/pm (0-11) Number
		 * 0 h Hour in am/pm (1-12) Number 12 m Minute in hour Number 30 s
		 * Second in minute Number 55 S Millisecond Number 978 z Time zone
		 * General time zone Pacific Standard Time; PST; GMT-08:00 Z Time zone
		 * RFC 822 time zone -0800
		 */
		
		return this.format("yyyy-MM-dd HH:mm.ss,SSS E z", Locale.ENGLISH,getTimeZone());
	}
	
	
	/**
	 * Data scritta in formato unix_timestamp
	 * @param data
	 * @return
	 */
	public int toUnixTimestamp() {
		long epoch = this.getTimeInMillis()/1000;
		return (int)epoch;
	}

	
	
	public int getYear() {
		return this.get(YEAR);
	}

	public int getMonth() {
		return this.get(MONTH) + 1;
	}

	public int getDay() {
		return this.get(DAY_OF_MONTH);
	}

	public int getHour() {
		return this.get(HOUR_OF_DAY);
	}

	public int getMinute() {
		return this.get(MINUTE);
	}

	public int getSecond() {
		return this.get(SECOND);
	}

	public int getWeek() {
		return this.get(Calendar.DAY_OF_WEEK);
	}
	
	
	public Timestamp getTimestamp() {
		return new Timestamp(this.getTimeInMillis());
	}

	
	public abstract K getDatePart();

	
	
	
}