package it.softecspa.kahuna.util.http;

import it.softecspa.database.reverser.ReverseException;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.math.BigDecimal;
import java.util.Enumeration;

import javax.servlet.http.HttpServletRequest;


public abstract class RequestParameter {

	
	private HttpServletRequest request;
	
	
	public RequestParameter(HttpServletRequest request) throws RequestParameterException {
		this(request, true);
	}
		
	public RequestParameter(HttpServletRequest request, boolean validate) throws RequestParameterException {
		this.request = request;		
		if (validate) {
			try {
				validate();			
			} catch (RequestParameterException e) {
				throw e;		
			} catch (Exception e) {
				throw new RequestParameterException("Unhandled exception in validate operation", e);
			}
		}		
		
	}
	
	
	
	public abstract void validate() throws RequestParameterException;
	
	
	
	/**
	 * Restituisce una eccezione in caso di mancata verifica del campo obbligatorio
	 * @param sorgente
	 * @param variabile
	 * @return
	 * @throws ReverseException
	 */
	protected <K> K checkRequestMandatory(String nome, Class<K> type) throws RequestParameterException {
		return checkMandatory(request, nome, type);
	}
	
	protected <K> K getRequest(String nome, Class<K> type) throws RequestParameterException {
		return get(request, nome, type);
	}
	
	protected <K> K getRequest(String nome, Class<K> type, K _default) throws RequestParameterException {
		K value = get(request, nome, type);
		if (value==null) return _default;
		return value;
	}
		
	/**
	 * Restituisce TRUE se la chiave  presente tra i parametri
	 * @param nome
	 * @return
	 * @throws RequestParameterException
	 */
	@SuppressWarnings("unchecked")
	protected boolean is(String nome) throws RequestParameterException {
		for (Enumeration<String> name= request.getParameterNames(); name.hasMoreElements();) {
			if (name.nextElement().equals(nome)) {
				return true;
			}			
		}
		return false;
	}
	
	
	@SuppressWarnings("unchecked")
	private <K> K get(HttpServletRequest sorgente, String nome, Class<K> type) throws RequestParameterException {
		String value = sorgente.getParameter(nome);
		if (XString.isBlankNull(value)) {
			value = (String)sorgente.getAttribute(nome);
			if (XString.isBlankNull(value)) return null;	
		}
		
		try {
			if (type.equals(Integer.class)) {
				try {
					return (K)new Integer(value);
				} catch (ClassCastException e) {
					throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid value; (Integer)");
				}
				
			} else if (type.equals(BigDecimal.class)) {
				try {
					return (K)new BigDecimal(value);
				} catch (ClassCastException e) {
					throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid value; (BigDecimal)");
				}
				
			} else if (type.equals(Long.class)) {
				try {
					return (K)new Long(value);
				} catch (ClassCastException e) {
					throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid value; (Long)");
				}
				
			} else if (type.equals(String.class)) {
				try {
					return (K)value;
				} catch (ClassCastException e) {
					throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid value; (String)");
				}
				
			} else if (type.equals(EnterpriseCalendar.class)) {
				try {
					return (K)new EnterpriseCalendar(value, EnterpriseCalendar.PATTERN_UTC_TIMEZONE);
				} catch (ClassCastException e) {
					throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid value; (String)");
				} catch (Exception e) {
					throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid format for date value: '"+EnterpriseCalendar.PATTERN_UTC_TIMEZONE+"'");
				}
			}
			
			throw new RequestParameterException("Parameter (or attribute) '"+nome+"' not parsed");
		} catch (Exception e) {
			throw new RequestParameterException("Parameter (or attribute) '"+nome+"' has no valid value: "+e.toString());
		}
		
	}
	
	
	private <K> K checkMandatory(HttpServletRequest sorgente, String nome, Class<K> type) throws RequestParameterException {
		K value = get(sorgente, nome, type);
		if (value==null) throw new RequestParameterException("Parameter (or attribute) '"+nome+"' is mandatory");
		return value;
	}

}
