package it.softecspa.kahuna.util.xml;


import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.util.Calendar;

import org.apache.log4j.Logger;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Classe di utilit spicciola
 * @author m.veroni
 */
public class NodeElement {

	
	private static Logger log = Logger.getLogger(NodeElement.class);
	private Element element;

	
	private NodeElement(Element element) {
		this.element = element;
	}
	
	
	public String getTextContent() {
		return element.getTextContent();
	}

	public String getName() {
		return element.getNodeName();
	}
	
	public String getNodeValue() {
		return element.getNodeValue();
	}
	
	
	public NodeType getType() {
		for (NodeType type : NodeType.values()) {
			if (type.getValue()==element.getNodeType()) {
				return type;
			}
		}
		return NodeType.UNCKNOWN;
			
	}

	public static NodeElement getInstance(Node node, String name) {
		if (node.getNodeType() == Node.ELEMENT_NODE) {
			if (node.getNodeName().equals(name)) {
				return new NodeElement((Element) node);
			}
			if (log.isDebugEnabled()) log.debug("Element not match '"+node.getNodeName()+"' vs '"+name+"', jump!");	
		}
		return null;			
	}
	
	
	public static NodeElement getInstance(Node node) {
		if (node.getNodeType() == Node.ELEMENT_NODE) {
			return new NodeElement((Element) node);				
		}
		return null;			
	}
	
	
	public static NodeElement getInstanceROOT(Element root, String name) throws XmlParserException {
		if (root==null || !root.getNodeName().equals(name)) {
			throw new XmlParserException("Malformed XML document,ROOT element must be '"+name+"'");		
		}
		return new NodeElement((Element) root);				
	}
	
	public boolean checkElementNode(String name) {
		return element.getNodeName().equalsIgnoreCase(name);
	}
		
	public NamedNodeMap hasAttributes() {
		if (element.getAttributes()!=null) {
			return element.getAttributes();
		}
		if (log.isDebugEnabled()) log.debug("No attributes found for node '"+element.getNodeName()+"', jump!");
		return null;
	}
	
	
	public String getAttribute(String name) {
		return getAttribute(name, null);
	}
	
	public String getAttribute(String name, String _default) {
		String value = element.getAttribute(name);
		if (XString.isBlankNull(value)) return _default;
		return value;
	}
	
	public String checkAttributeMandatory(String name) throws XmlParserException {
		String value = getAttribute(name);
		if (XString.isBlankNullTrim(value)) throw new XmlParserException(this, name, String.class);
		return value;
	}
	
	
	public Integer getAttributeAsInteger(String name) {
		return getAttributeAsInteger(name, null);
	}
	
	public Integer getAttributeAsInteger(String name, Integer _default) {
		String value = element.getAttribute(name);
		if (XString.isBlankNull(value)) return _default;
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException e) {
			return _default;
		}
		
	}
	
	public Integer checkAttributeMandatoryAsInteger(String name) throws XmlParserException {
		Integer value = getAttributeAsInteger(name);
		if (value==null) throw new XmlParserException(this, name, Integer.class);
		return value;
	}
	
	
	public Boolean getAttributeAsBoolean(String name) {
		return getAttributeAsBoolean(name, Boolean.FALSE);
	}
	
	public Boolean getAttributeAsBoolean(String name, Boolean _default) {
		String value = element.getAttribute(name);
		if (XString.isBlankNull(value)) return _default;
		if ("1".equals(value)) return Boolean.TRUE;
		if ("true".equalsIgnoreCase(value)) return Boolean.TRUE;
		return _default;
	}

	public Boolean checkAttributeMandatoryAsBoolean(String name) throws XmlParserException {
		String value = element.getAttribute(name);
		if (XString.isBlankNullTrim(value)) throw new XmlParserException(this, name, Boolean.class);
		return getAttributeAsBoolean(name);
	}
	
	
	
	public EnterpriseCalendar getAttributeAsEnterpriseCalendar(String name) {
		String value = element.getAttribute(name);
		if ("".equals(value)) return null;
		try {
			return EnterpriseCalendar.newInstance(value, EnterpriseCalendar.PATTERNS_UTC);
		} catch (Exception e) {
			log.warn("Attribute date format not match '"+value+"' vs '"+EnterpriseCalendar.PATTERN_UTC_ZULU+"', jump!");	
		}
		return null;
	}
	
	public EnterpriseCalendar checkAttributeMandatoryAsEnterpriseCalendar(String name) throws XmlParserException {
		EnterpriseCalendar value = getAttributeAsEnterpriseCalendar(name);
		if (value==null) throw new XmlParserException(this, name, EnterpriseCalendar.class);
		return value;
	}
	
	
	/**
	 * @deprecated
	 * @param name
	 * @return
	 */
	public Calendar getAttributeAsCalendar(String name) {
		//if (element.getAttributes()==null) return null;
		String value = element.getAttribute(name);
		if ("".equals(value)) return null;
		try {
			return EnterpriseCalendar.newInstance(value, EnterpriseCalendar.PATTERNS_UTC);
		} catch (Exception e) {
			log.warn("Attribute date format not match '"+value+"' vs '"+EnterpriseCalendar.PATTERN_UTC_ZULU+"', jump!");	
		}
		return null;
	}
	
	

	public Element get() {
		return element;
	}
	
	public NodeList getChilds() {
		return element.getChildNodes();
	}


	public Node getNextSibling() {
		return element.getNextSibling();
	}
	
	
	public NodeElement getChildElement() {
		if (element.getChildNodes().getLength()==1) return null;
		
		NodeList items = element.getChildNodes();
		for (int i=0; i<items.getLength(); i++) {	
			Node node = items.item(i);			
			if (node.getNodeType() == Node.ELEMENT_NODE) {
				return new NodeElement((Element) node);					
			}
		}
		return null;
	}


	
}
