package it.softecspa.kahuna.util.xml;

import it.softecspa.kahuna.lang.XString;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Classe  per la gestione automatica di file XML
 * 
 * @author m.veroni
 *
 */
public abstract class XmlElement {

	protected String entityName;
	protected String entityValue;	
	
	// ENTITY* annidate
	protected HashMap<String, List<XmlElement>> childs;	
	
	
	protected XmlElement() {
		this.entityName = getClass().getSimpleName().toLowerCase();
	}

	protected XmlElement(String tagName) {
		this.entityName = (tagName!=null?tagName:getClass().getSimpleName().toLowerCase());
	}	
		
	
	public String getEntityName() {
		return entityName;
	}
	
	public String getEntityValue() {
		return entityValue;
	}

	
	public Integer getEntityValueAsInteger() {
		if (XString.isBlankNullTrim(entityValue)) return null;
		return Integer.parseInt(entityValue);
	}
	
	public Long getEntityValueAsLong() {
		if (XString.isBlankNullTrim(entityValue)) return null;
		return Long.parseLong(entityValue);
	}
	
	
	
	public void setEntityValue(String entityValue) {
		this.entityValue = entityValue;
	}
	
	
	
	public void addElementChilds(String name, List<XmlElement> elements) {
		if (childs==null) childs = new HashMap<String, List<XmlElement>>();
		
		List<XmlElement> appoggio = childs.get(name);
		if (appoggio==null) {
			childs.put(name, elements);			
		} else {
			for(XmlElement element : elements) {
				appoggio.add(element);
			}
		}		
	}
	
	/**
	 * Aggiunge un child all'interno di un elemento aggregante
	 * @param name
	 * @param element
	 */
	private void addElementChild(String name, XmlElement element) {
		if (childs==null) childs = new HashMap<String, List<XmlElement>>();
		
		List<XmlElement> appoggio = childs.get(name);
		if (appoggio==null) {
			appoggio = new ArrayList<XmlElement>();
			childs.put(name, appoggio);			
		}
		appoggio.add(element);
	}
	
	/**
	 * Aggiunge un child all'interno dell XML
	 * @param name
	 * @param element
	 */
	public void addElementChild(XmlElement element) {
		addElementChild(null, element);
	}


	public HashMap<String, List<XmlElement>> getElementChild() {
		return childs;
	}
	
	
	/**
	 * Parser dell XML e valorizzazione dei tag
	 * @param father
	 * @throws XmlParserException 
	 */
	public abstract void parse(NodeElement father) throws XmlParserException;
	
	
	/**
	 * Controllo se il valore inserito  non nullo
	 * @param value
	 * @throws XmlParserException
	 */
	protected String checkMandatory(String value) throws XmlParserException {
		if (XString.isBlankNullTrim(value)) throw new XmlParserException(this);
		return value;
	}
	
	/**
	 * Controllo se il valore inserito  un Integer
	 * @param value
	 * @throws XmlParserException
	 */
	protected Integer checkInteger(String value) throws XmlParserException {
		if (XString.isBlankNullTrim(value)) return null;
		return checkIntegerMandatory(value);
	}
	
	
	protected Integer checkIntegerMandatory(String value) throws XmlParserException {
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException e) {
			throw new XmlParserException(this,e);
		} catch (NullPointerException e) {
			throw new XmlParserException(this);
		} catch (Exception e) {
			throw new XmlParserException(this,e);
		}
	}
	
	/**
	 * Controllo se il valore inserito  un Long
	 * @param value
	 * @throws XmlParserException
	 */
	protected Long checkLong(String value) throws XmlParserException {
		if (XString.isBlankNullTrim(value)) return null;
		return checkLongMandatory(value);
	}
	
	protected Long checkLongMandatory(String value) throws XmlParserException {
		try {
			return Long.parseLong(value);
		} catch (NumberFormatException e) {
			throw new XmlParserException(this,e);
		} catch (NullPointerException e) {
			throw new XmlParserException(this);
		} catch (Exception e) {
			throw new XmlParserException(this,e);
		}
	}
	
	
	/**
	 * Applica il <![CDATA[ ... ]]>	
	 * @param value
	 * @return
	 */
	public static String toCDATA(String value) {
		return "<![CDATA["+value+"]]>";
	}
	
}
