package it.softecspa.kahuna.util.xml;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import com.sun.org.apache.xml.internal.serialize.OutputFormat;
import com.sun.org.apache.xml.internal.serialize.XMLSerializer;

/**
 * Classe  per la gestione automatica di file XML
 * Elemento ROOT
 * 
 * @author m.veroni
 *
 */
public abstract class XmlRoot extends XmlElement {

	protected Logger log = Logger.getLogger(getClass());
	protected Document inputDocument;
	
	protected boolean apply_cdata; 
	
	protected XmlRoot() {
		super();
	}
	
	
	protected XmlRoot(String tagName) {
		super(tagName);
	}
	
	
	public byte[] generateXML(String charsetName) throws UnsupportedEncodingException {
		return generateXML(charsetName, null, true);
	}
	
	public byte[] generateXML(String charsetName, boolean pretty) throws UnsupportedEncodingException {
		return generateXML(charsetName, null, pretty);
	}
	
	public byte[] generateXML(String charsetName, String dtd, boolean pretty) throws UnsupportedEncodingException {
		if (log.isDebugEnabled()) log.debug("Translate XmlElement in XML layout");
		
		String xml = "<?xml version=\"1.0\" encoding=\""+charsetName+"\"?>";
		if (dtd!=null) {
			xml+="<!DOCTYPE properties SYSTEM \""+dtd+"\">";
		}
		xml += elementRipper(this);	
		
		if (pretty) {
			try {
				xml = XString.prettyXML(xml);				
			} catch (Exception e) {
				log.warn("Problem \"pretty\" XML response", e);
			}		
		}
		
		try {
			return xml.getBytes(charsetName);
		} catch (UnsupportedEncodingException e) {
			throw e;
		}
		
		
	}
		
	
	public static Document loadDocument(InputStream inputStream) throws XmlParserException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = null;
		Document document = null;
		try {
			builder = factory.newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			throw new XmlParserException(e);
		}
		try {
			document = builder.parse(inputStream);
			if (document!=null) {
				document.getDocumentElement().normalize();
			}
		} catch (SAXException e) {
			throw new XmlParserException(e);				
		} catch (IOException e) {
			throw new XmlParserException(e);
		} catch (Exception e) {
			throw new XmlParserException(e);
		}
		
		return document;		
	}
	
	
	
	public boolean parseXML(InputStream inputStream) throws XmlParserException {
		if (log.isDebugEnabled()) log.debug("Load and check XML document from input stream");
		
		// Apertura del documento
		this.inputDocument = null;
		inputDocument = loadDocument(inputStream);
		if (inputDocument==null) {
			return false;
		}
				
		// Check ROOT and start parsing
		Element root = inputDocument.getDocumentElement();
		parse(NodeElement.getInstanceROOT(root, getEntityName()));	
		
		log.info("Parsing of XML document completed; generated class "+getClass().getSimpleName());
		return true;
	}
	
	/**
	 * Restituisce il document XML
	 * @return
	 */
	public Document extractInputDocument() {
		return this.inputDocument;
	}
	
	/**
	 * Restituisce XML letto in input
	 * @return
	 */
	public String extractInputXML() {
		if (this.inputDocument==null) return null;
		try {
			OutputFormat format = new OutputFormat(this.inputDocument);
	        format.setIndenting(true);
	        format.setIndent(2);
	        Writer out = new StringWriter();
	        XMLSerializer serializer = new XMLSerializer(out, format);
	        serializer.serialize(this.inputDocument);
	        return out.toString();
		} catch (Exception e) {
			log.error("Error formatting input XML",e);
			return null;
		}
	}
	
	

	@SuppressWarnings("unchecked")
	protected String elementRipper(XmlElement entity) {
		
		class AttributeBean {	
			
			String name;
			String value;
			
			public boolean check(String name, Object value) {
				if (value instanceof XmlAttribute<?>) {
					XmlAttribute<?> attribute = (XmlAttribute<?>)value;
					return check(attribute.getName()!=null?attribute.getName():name, attribute.getValue());
				}				
				if (value==null) return false;
				
				this.name = name;
				
				// ....trasformazione....
				if (value instanceof Boolean) {
					this.value = (((Boolean)value).booleanValue()?"1":"0");
				// 
				} else if (value instanceof EnterpriseCalendar) {
					this.value = ((EnterpriseCalendar)value).formatISO8601zulu();
				} else if (value instanceof Calendar) {					
					this.value = new EnterpriseCalendar((Calendar)value).formatISO8601zulu();			
				} else if (value instanceof java.sql.Date) {					
					this.value = new EnterpriseCalendar((java.sql.Date)value).formatISO8601zulu();
				} else if (value instanceof java.util.Date) {					
					this.value = new EnterpriseCalendar((java.util.Date)value).formatISO8601zulu();
				//	
				} else {
					this.value = value.toString();
				}
				
				return true;
			}
			
		}
		
		class OccurrencyBean {			
			String name;
			List<XmlElement> elements;			
		}
		
		
		
		/*
		 * Ciclo su tutti i metodi per estrarre il valore e metterlo nell HEADER
		 */
		if (log.isTraceEnabled()) log.trace(">>>> Class 2 header conversion for: "+entity.getClass().getName());
		
		List<XmlElement> entities = new ArrayList<XmlElement>();
		List<OccurrencyBean> occurrences = new ArrayList<OccurrencyBean>();
		List<AttributeBean> attributes = new ArrayList<AttributeBean>();
		
		Method[] methods = entity.getClass().getMethods();
		for (Method method : methods) {
			
			if (!method.getName().startsWith("get")) continue;
			if (method.getName().equals("getClass")) continue;
			if (method.getName().equalsIgnoreCase("getEntityName")) continue;
			if (method.getName().startsWith("getEntityValue")) continue;			
			if (log.isTraceEnabled()) log.trace(">> method: "+method.getName());
			
			
			String name = XString.toLowerCaseFirst(method.getName().substring(3));
			//Class<?> clazz = method.getReturnType();							
			
			Object value = null;			
			try {
				value = method.invoke(entity, (Object[])null);				
			} catch (IllegalArgumentException e) {
				continue;
			} catch (IllegalAccessException e) {
				continue;
			} catch (InvocationTargetException e) {
				continue;
			}		
			if (value==null) continue;
			
			if (value instanceof XmlElement) {
				
				// ENTITY
				entities.add((XmlElement)value);	
			
			} else if (value instanceof XmlList) {	
				// ENTITY* derivata da XmlList
				XmlList<XmlElement> cvalue = (XmlList<XmlElement>)value;
				if (cvalue.size()==0 && cvalue.hideIfEmpty()) continue;
				
				OccurrencyBean bean = new OccurrencyBean();
				bean.name = cvalue.getName();
				bean.elements = cvalue;
				occurrences.add(bean);	
			/*
			} else if (value instanceof XmlMap<?>) {
				
				// ENTITY* childs
				XmlMap<?> map = (XmlMap<?>)value;
				if (map.size()>0) {
					XmlList<XmlElement> lista = new XmlList<XmlElement>(map.getEntityName(), map.size());
					for (String key : map.keySet()) {
						XmlElement element = new XmlElement(key) {
							@Override
							public void parse(NodeElement father) throws XmlParserException {
								// TODO Auto-generated method stub
							}
						};
						element.setEntityValue(map.get(key));
						lista.add(element);
					}	
					
					OccurrencyBean bean = new OccurrencyBean();
					bean.name = lista.getName();
					bean.elements = lista;
					occurrences.add(bean);
				}
				*/
				
			} else if (value instanceof HashMap) {
				// ENTITY* childs nativo
				HashMap<String, List<XmlElement>> childs = (HashMap<String, List<XmlElement>>)value;
				for (String key : childs.keySet()) {
					OccurrencyBean bean = new OccurrencyBean();
					bean.name = key;
					bean.elements = childs.get(bean.name);
					occurrences.add(bean);
				}
				
			} else if ((value instanceof List) || 
					   (value instanceof ArrayList)) {
				
				// ENTITY* derivata da semplice List
				OccurrencyBean bean = new OccurrencyBean();
				bean.name = name;
				bean.elements = (List<XmlElement>)value;
				occurrences.add(bean);
			
			} else if (value instanceof XmlAttribute<?>) {	
				// ATTRIBUTE
				AttributeBean bean = new AttributeBean();
				if (bean.check(name, value)) {
					attributes.add(bean);
				}
			
			} else {
				// ATTRIBUTE
				AttributeBean bean = new AttributeBean();
				bean.check(name, value);
				attributes.add(bean);				
			}			
		}
		
		
		// TAG di apertura
		String xml = "<" + entity.getEntityName();
		// Attributi
		for (AttributeBean attributo : attributes) {
			xml += " " + attributo.name + "=\"" + attributo.value + "\"";
		}
		// .. valore o ENTITY annidate
		if (entities.size()>0 ||
			occurrences.size()>0 ||
			entity.getEntityValue()!=null) {
			xml += ">";			
			for (XmlElement element : entities) {
				xml += elementRipper(element);
			}
			for (OccurrencyBean occurrency : occurrences) {
				boolean opened = false;
				if (XString.isNotBlankNull(occurrency.name)) { 
					xml += "<" + occurrency.name + ">"; // eventuale TAG aggregatore
					opened = true;
				}
				// lista 
				for (XmlElement element : occurrency.elements) {
					xml += elementRipper(element);
				}
				
				if (opened) {
					xml += "</" + occurrency.name + ">"; // eventuale TAG aggregatore (chiusura)
				}
			}
			// FIXME se ho dei sottoTAG non devo avere ENTITY VALUE
			// ENTITY VALUE
			if (entity.getEntityValue()!=null) {
				if (apply_cdata) {
					xml += XString.isNotBlankNullTrim(entity.getEntityValue())?toCDATA(entity.getEntityValue()):entity.getEntityValue();
				} else {
					xml += entity.getEntityValue();
				}
			}
			//
			xml += "</" + entity.getEntityName() +">";			
		} else {
			xml += "/>";
		}
		return xml;
	}


	

	public boolean isApplyCData() {
		return apply_cdata;
	}

	/**
	 * Applica il tag <![CDATA[ ... ]]> agli elementi
	 * @param value
	 */
	public void applyCData() {
		this.apply_cdata = true;
	}


	
}
