package it.softecspa.s4fs;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.Properties;

import org.apache.log4j.Logger;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.s3.model.Region;

public class S4Config {
	
	
	// Configurazione componente S4
	private static final String PREFIX_S4 = "s4.";
	public static final String S4_ENABLE 			= PREFIX_S4 + "enable";
	public static final String S4_FS_ROOT 			= PREFIX_S4 + "filesystem.root";
	public static final String S4_OWNER 			= PREFIX_S4 + "owner.name";
	public static final String S4_CONTEXT 			= PREFIX_S4 + "context";
	public static final String S4_BUCKET 			= PREFIX_S4 + "bucket";
	public static final String S4_AWS_ACCESS_KEY 	= PREFIX_S4 + "aws.access.key";
	public static final String S4_AWS_SECRET_KEY 	= PREFIX_S4 + "aws.secret.key";
	public static final String S4_CLEANER_DELAY 	= PREFIX_S4 + "cleaner.delay";
	
	
	private Logger log = Logger.getLogger(getClass());
	
	
	// Path su cui sono appoggiati i file sul filesystem locale (obbligatorio)
	private String filesystemRoot;
	
	// Nome univoco dell'owner dei file  (obbligatorio)
	// Identifica il nodo su cui gira l'istanza
	private String ownerName;
	
	// Puntamento al mdatabase doive risiede la VFAT  (obbligatorio)
	private ConnectionManager cmWritable;
	
	// Contesto dell'applicazione (obbligatorio)
	private String context;

	
	
	
	// Bucket di riferimento su Amazon S3
	private String bucketName;
	
	// Credenziali di accesso a Amazon S3
	private AWSCredentials awsCredentials;
	
	// Regione su cui sono salvati i file
	private Region region;
	
	
	
	// Ciclo di vita del cleane del filesystem 
	private int cleanerDelay;
	private final int DEFAULT_DELAY = 3600; 
	
	
	
	public S4Config() {
		super();
	}
	
	
	public S4Config(Properties prop, ConnectionManager cmWritable) {
		this("", prop, cmWritable);
	}
	
	public S4Config( String prefix, Properties prop, ConnectionManager cmWritable) {
		this();
		
		setFilesystemRoot(prop.get(prefix+S4_FS_ROOT));
		setOwnerName(prop.get(prefix+S4_OWNER));
		setContext(prop.get(prefix+S4_CONTEXT));
		setBucketName(prop.get(prefix+S4_BUCKET));		
		setAwsCredentials(prop.get(prefix+S4_AWS_ACCESS_KEY),prop.get(S4_AWS_SECRET_KEY));
		setCleanerDelay(prop.getInt(S4_CLEANER_DELAY));
		
		
		setCmWritable(cmWritable);
		
	}
	
	
	public String getOwnerName() {
		return ownerName;
	}

	/**
	 * Imposta l'identificativo univoco del nodo dove sar eseguita l'istanza di {@S4Engine}
	 * @param ownerUname
	 */
	public void setOwnerName(String ownerUname) {
		this.ownerName = ownerUname;
	}

	
	public ConnectionManager getCmWritable() {
		return cmWritable;
	}

	/**
	 * Imposta il {@ConnectionManager} utilizzato da {@S4Engine}
	 * Il connection manager deve essere di lettura/scrittura
	 * @param ownerUname
	 */
	public void setCmWritable(ConnectionManager cmWritable) {
		this.cmWritable = cmWritable;
	}

	
	public String getBucketName() {
		return bucketName;
	}

	
	/**
	 * Imposta il bucket per utilizzato sui servizi Amazon S3
	 * Se non esiste  automaticamente creato
	 * @param ownerUname
	 */
	public void setBucketName(String bucketName) {
		this.bucketName = bucketName;
	}

	
	public AWSCredentials getAwsCredentials() {
		return awsCredentials;
	}

	
	/**
	 * Imposta le credenziali per i servizi Amazon S3
	 * @param awsCredentials
	 */
	public void setAwsCredentials(AWSCredentials awsCredentials) {
		this.awsCredentials = awsCredentials;
	}
	
	/**
	 * Imposta le credenziali per i servizi Amazon S3
	 * @param awsCredentials
	 */
	public void setAwsCredentials(String accessKey, String secretKey) {
		if (XString.isNotBlankNullTrim(accessKey) && XString.isNotBlankNullTrim(secretKey)) {
			this.awsCredentials = new BasicAWSCredentials(accessKey, secretKey);
		}
	}
	
	
	public Region getRegion() {
		if (region==null) region = Region.EU_Ireland;
		return region;
	}

	/**
	 * Imposta la regione dove Amazon S3 archivia i file
	 * @param region
	 */
	public void setRegion(Region region) {
		this.region = region;
	}
	
	
	
	public String getContext() {
		if (context==null) context = "/";
		return context;
	}

	/**
	 * Imposta il contest dell'applicazione web che utilizza il compnente
	 * Se non indicato si intende il contesto di ROOT
	 * @param context
	 */
	public void setContext(String context) {
		/* 
		 * Normalizzazione del context
		 *  ...DEVE iniziare e terminare con /
		 */
		if (context==null) context = ""; 
		context = context.trim();
		filesystemRoot = XString.replaceAll(filesystemRoot, "\\", "/");
		if (!context.startsWith("/")) context = "/"+context;
		if (!context.endsWith("/")) context += "/";
		//
		this.context = context;
	}


	

	/**
	 * Controlla la validit dei parametri inseriti
	 */
	public boolean check() {
		try {
			validate();
			return true;
		} catch (S4Exception e) {
			return false;
		}
	}
	
	void validate() throws S4Exception {
		// Campi obbligatori
		if (XString.isBlankNullTrim(filesystemRoot)) throw new S4Exception(S4Status.FIELD_MANDATORY,": 'filesystemRoot'");
		if (XString.isBlankNullTrim(ownerName)) throw new S4Exception(S4Status.FIELD_MANDATORY,": 'ownerUname'");
		if (cmWritable==null) throw new S4Exception(S4Status.FIELD_MANDATORY,": 'cmWritable'");
		if (XString.isBlankNullTrim(context)) {
			log.warn(S4Engine.class.getSimpleName()+": no value for 'context', use ROOT");
		}
		
		// Voglio utilizzare il servizio Amazon S3
		if (awsCredentials!=null && XString.isNotBlankNullTrim(bucketName)) {
			if (XString.isBlankNullTrim(bucketName)) throw new S4Exception(S4Status.FIELD_MANDATORY,": 'bucketName' if present 'awsCredentials'");
			if (region==null) {
				log.warn(S4Engine.class.getSimpleName()+": no value for 'region', use "+Region.class.getSimpleName()+"."+Region.EU_Ireland);
			}
		}
		
		if (cleanerDelay<=0) {
			log.warn(S4Engine.class.getSimpleName()+": no value for 'cleanerDelay', use default "+DEFAULT_DELAY+" seconds");
		}		
	}


	public String getFilesystemRoot() {
		return filesystemRoot;
	}


	public void setFilesystemRoot(String filesystemRoot) {
		/* 
		 * Normalizzazione del path della ROOT
		 *  ...NON DEVE terminare con /
		 */
		if (filesystemRoot==null) filesystemRoot="";
		filesystemRoot = filesystemRoot.trim();
		if (filesystemRoot.length()>0) {
			filesystemRoot = XString.replaceAll(filesystemRoot, "\\", "/");
			// ...tolgo la / finale al path
			if (filesystemRoot.endsWith("/")) filesystemRoot.substring(0,filesystemRoot.length()-1);
		}
		//
		this.filesystemRoot = filesystemRoot;
	}


	/**
	 * Tempo in secondi, con cui cicla il processo di pulizia del filesystem
	 * @return
	 */
	public int getCleanerDelay() {
		return cleanerDelay>0?cleanerDelay:DEFAULT_DELAY;
	}

	public void setCleanerDelay(int seconds) {
		this.cleanerDelay = seconds;
	}
	

}
