package it.softecspa.s4fs;

import it.softecspa.s4fs.vfat.Bucket;
import it.softecspa.s4fs.vfat.FileAllocation;
import it.softecspa.s4fs.vfat.FileS4;
import it.softecspa.s4fs.vfat.FilesystemCleaner;
import it.softecspa.s4fs.vfat.FilesystemPolicy;
import it.softecspa.s4fs.vfat.LockerS4;
import it.softecspa.s4fs.vfat.Owner;
import it.softecspa.s4fs.vfat.VFATException;
import it.softecspa.s4fs.vfat.VFATEngine;
import it.softecspa.s4fs.vfat.VFATStatistics;
import it.softecspa.s4fs.vfat.VirtualAllocationTable;
import it.softecspa.s4fs.vfat.WritePolicy;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.Thread.State;
import java.nio.channels.FileChannel;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;



// TODO Gestione del cambio del file su S3 con il reaload delle cache 
// TODO Gestire una statistica dei file non cancellati dal filesystem
// TODO Eliminazione dal garbage di tutti i file DRAFT pi vecchi di una settimana

public class S4Engine {

	private static S4Engine instance;
	private boolean configured;
	
	
	
	/**
	 * Restituisce una istanza di @{link S4Engine}
	 * @return
	 */
	public static S4Engine getInstance() {
		synchronized (S4Engine.class) {
			if (instance == null) new S4Engine();			
		}
		return instance;
	}
	
	/**
	 * Restituisce una istanza di @{link S4Engine} e contestulamente la configura
	 * @param config Bean contenente i parametri di configurazione
	 * @return
	 * @throws S4Exception 
	 */
	public static S4Engine getInstance(S4Config config) throws S4Exception {
		synchronized (S4Engine.class) {
			if (instance == null) {
				new S4Engine();				
			}
			instance.initialize(config);
		}
		return instance;
	}
	
	private Logger log = Logger.getLogger(getClass());
	
	private S4Config config;
	
	
	// Demon di pulizia automatizzata
	private Thread cleanerThread;
	private FilesystemCleaner cleaner;
	
	
	// Bucket su cui sono salvati i dati su Amazon S3, se utilizzato
	private S4Bucket s4bucket;
	private AmazonS3 s3service;
	
	// File Allocation Table virtuale
	private VFATEngine vfat;
	private Owner owner;
	
	
	
	
	private S4Engine() {
		super();
		synchronized (S4Engine.class) {
			if (instance == null) {
				instance = this;
				log.info(S4Engine.class.getSimpleName()+" instance is starting (Version "+new Version().toString()+"), but not inizialized!");	
			}
		}
	}
	
	/**
	 * Riconfigura l'istanza di @{link S4Engine}
	 * @param config
	 * @throws S4Exception
	 */
	public void reconfigure(S4Config config) throws S4Exception {
		configured = false;
		configure(config);
	}
	
	/**
	 * Inizializza listanza utilizzando
	 * @param config Bean contenente i parametri di configurazione
	 */
	public void initialize(S4Config config) throws S4Exception {
		if (configured) return;
		configure(config);
	}
	
	/**
	 * Inizializza listanza utilizzando
	 * @param config Bean contenente i parametri di configurazione
	 */
	private void configure(S4Config config) throws S4Exception {
		synchronized (S4Engine.class) {
			config.validate();
			this.config = config;				
		
			// recupero i dati dell'owner univoco
			try {
				owner = new Owner(config.getCmWritable(), config.getOwnerName());
			} catch (SQLException e) {
				throw new S4Exception(S4Status.CHECK_OWNER, e);
			}
			
			
			if (config.getAwsCredentials()!=null) {
				// Esiste una condivisione configurata su Amazon S3
				log.info(S4Engine.class.getSimpleName()+" use Amazon S3 filesystem: bucket = '"+config.getBucketName()+"', region = '"+config.getRegion()+"'");
				s4bucket = new S4Bucket(config.getBucketName(), config.getAwsCredentials(), config.getRegion());
				
				// Creo una istanza del client di S3
				s3service = new AmazonS3Client(s4bucket.getCredentials());
				
				// Verifica la consitenza del bucket
				checkBucket();
			}
			
			
			// Recupero o creo una istanza della VFAT;
			this.vfat = VFATEngine.newInstance(config.getCmWritable(), owner, s4bucket);		
			// Eseguo un check
			try {
				vfat.check();
			} catch (VFATException e) {
				throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
			}
			
			// Stampo nel log delle statistiche
			try {
				VFATStatistics stats = getStatistics();
				stats.toLog(log);
			} catch (S4Exception e) {
				log.warn("Statistics not publishable: "+e.toString(), e);
			}
			
			log.info(S4Engine.class.getSimpleName()+" instance configured with success!");
			configured = true;
			
			
			// Lancio il thread per la pulizia dei file
			cleaner = new FilesystemCleaner(this);
			cleanerThread = new Thread(cleaner, "S4Cleaner("+config.getOwnerName()+","+config.getContext()+")");
			cleanerThread.setDaemon(true);
			cleanerThread.start();
			log.info("Cleaner thread started");
		}
	}

	
	/**
	 * Verifica la consistenza del bucket
	 * @throws S4Exception 
	 */
	private void checkBucket() throws S4Exception {
		// Verifico o creo l'istanza del bucket
		if (log.isDebugEnabled()) log.debug("Verify if bucket exist!");
		Bucket dbBbucket = null;
		try {
			dbBbucket = new Bucket(config.getCmWritable(), s4bucket.getName());
		} catch (SQLException e) {
			throw new S4Exception(S4Status.CHECK_BUCKET, e);
		}
		
		if (dbBbucket.getUid()==null) {
			// il record non esiste
			
			/*
             * Create a new S3 bucket - Amazon S3 bucket names are globally unique,
             * so once a bucket name has been taken by any user, you can't create
             * another bucket with that same name.
             *
             * You can optionally specify a location for your bucket if you want to
             * keep your data closer to your applications or users.
             */
        	log.info("Creating S3 bucket '" + s4bucket.getName() + "', region "+s4bucket.getRegion());
        	try {
        		com.amazonaws.services.s3.model.Bucket s3Bucket = s3service.createBucket(s4bucket.getName(), s4bucket.getRegion());
        		
        		// Salvataggio informazioni su database
        		dbBbucket.setBacketName(s4bucket.getName());
        		dbBbucket.setCreation(s3Bucket.getCreationDate());        		
        		dbBbucket.save(config.getCmWritable());
        		
        	} catch (AmazonServiceException e) {
        		throw new S4Exception(e);
            } catch (AmazonClientException e) {
            	throw new S4Exception(e);      
            	
            } catch (SQLException e) {
            	throw new S4Exception(S4Status.SAVE_BUCKET, e);
			}	        		
		}
		// Aggiorno il bean del bucket con il valore dell'uid recuperato dalla VFAT
		s4bucket.setUid(dbBbucket.getUid());		
	}

	
	/**
	 * Restituisce TRUE  configurato per utilizzare i servizi di Amazon S3
	 * @return
	 */
	public boolean useS3Service() {
		return (s3service!=null);
	}
	
	
	/**
	 * Istanza del {@FilesystemCleaner}
	 * @return
	 */
	public FilesystemCleaner getCleaner() {
		return cleaner;
	}
	
	
	/**
	 * Stato del thread {@FilesystemCleaner}
	 * @return
	 */
	public State getCleanerState() {
		if (cleanerThread==null) return State.NEW;
		return cleanerThread.getState();
	}
	
	
	
	
	/**
	 * Creo una nuova istanza di {@FileS4} in modo da poter essere utilizzato
	 * L'istanza del file non deve esistere sulla VFAT
	 * @param nesting
	 * @param name
	 * @param policy
	 * @return
	 * @throws S4Exception
	 */
	public synchronized FileS4 createFile(String nesting, String name, FilesystemPolicy policy) throws S4Exception {
		return createFile(nesting, name, policy, false);
	}
	
	/**
	 * Creo una nuova istanza di {@FileS4} in modo da poter essere utilizzato
	 * @param nesting
	 * @param name
	 * @param policy
	 * @param overwrite Polici di sovrascrittura del file
	 * @return
	 * @throws S4Exception
	 */
	public synchronized FileS4 createFile(String nesting, String name, FilesystemPolicy fsPolicy, boolean overwrite) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		if (fsPolicy.useS3() && !useS3Service()) throw new S4Exception(S4Status.UNUSABLE_POLICY);
		
		WritePolicy wp = (overwrite?WritePolicy.UPDATABLE:WritePolicy.READONLY);
		try {			
			return vfat.createFile(config.getFilesystemRoot(), config.getContext(), nesting, name, fsPolicy, wp);
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}
	
	
	/**
	 * Aggiunge un file esistente su filesystem locale nella VFAT
	 * L'istanza del file non deve esistere sulla VFAT
	 * 
	 * Nota: il file  subito copiato nel filesystem della VFAT 
	 * ma  consolidato soltanto attraverso il metodo {@link FileS4#close()} 
	 * @param source
	 * @param nesting
	 * @param policy
	 * @return
	 * @throws S4Exception
	 */
	public synchronized FileS4 addFile(java.io.File source, String nesting, FilesystemPolicy policy) throws S4Exception {
		return addFile(source, nesting, policy, false);
	}
	
	/**
	 * Aggiunge un file esistente su filesystem locale nella VFAT
	 * 
	 * Nota: il file  subito copiato nel filesystem della VFAT 
	 * ma  consolidato soltanto attraverso il metodo {@link FileS4#close()} 
	 * @param source
	 * @param nesting
	 * @param policy
	 * @param overwrite
	 * @return
	 * @throws S4Exception
	 */
	public synchronized FileS4 addFile(java.io.File source, String nesting, FilesystemPolicy policy, boolean overwrite) throws S4Exception {
		return addFile(source, nesting, source.getName(), policy, false);
	}
	
	/**
	 * Aggiunge un file esistente su filesystem locale nella VFAT
	 * L'istanza del file non deve esistere sulla VFAT
	 * 
	 * Nota: il file  subito copiato nel filesystem della VFAT 
	 * ma  consolidato soltanto attraverso il metodo {@link FileS4#close()} 
	 * @param source
	 * @param nesting
	 * @param name
	 * @param policy
	 * @return
	 * @throws S4Exception
	 */
	public synchronized FileS4 addFile(java.io.File source, String nesting, String name, FilesystemPolicy policy) throws S4Exception {
		return addFile(source, nesting, name, policy, false);
	}
	
	/**
	 * Aggiunge un file esistente su filesystem locale nella VFAT
	 * 
	 * Nota: il file  subito copiato nel filesystem della VFAT 
	 * ma  consolidato soltanto attraverso il metodo {@link FileS4#close()} 
	 * @param source
	 * @param nesting
	 * @param name
	 * @param fsPolicy
	 * @param overwrite
	 * @return
	 * @throws S4Exception
	 */
	@SuppressWarnings("resource")
	public synchronized FileS4 addFile(java.io.File source, String nesting, String name, FilesystemPolicy fsPolicy, boolean overwrite) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		if (fsPolicy.useS3() && !useS3Service()) throw new S4Exception(S4Status.UNUSABLE_POLICY);
				
		if (!source.exists()) {
			throw new S4Exception(S4Status.UNUSABLE_POLICY);
		}
				
		WritePolicy wp = (overwrite?WritePolicy.UPDATABLE:WritePolicy.READONLY);
		try {			
			FileS4 file = vfat.createFile(config.getFilesystemRoot(), config.getContext(), nesting, name, fsPolicy, wp);
			
			// Copia il file nell'area della VFAT
			try {					
				FileChannel inChannel = new FileInputStream(source).getChannel();
				FileChannel outChannel = new FileOutputStream(file.getRealFile()).getChannel();
				inChannel.transferTo(0, inChannel.size(), outChannel);
				// or destinationChannel.transferFrom(sourceChannel, 0, sourceChannel.size());
				inChannel.close();
				outChannel.close();				
			} catch (IOException e) {
				throw new S4Exception(S4Status.SOURCE_FILE_COPY_ERROR, e);
			}
			
			return file;
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}		
		
		
	}
	
	
	
	/**
	 * Apre il file scegliendo la modalit di apertura
	 * Se nessun file  trovato,  restituito il valore NULL
	 * In caso di scrittura il file deve esistere
	 * @param nesting
	 * @param name
	 * @param writable
	 * @return
	 * @throws S4Exception
	 */
	public FileS4 openFile(String nesting, String name, WritePolicy writable) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		try {
			return vfat.loadFile(config.getFilesystemRoot(), config.getContext(), nesting, name, writable);			
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}
	
	/**
	 * Apre il file in sola lettura
	 * Se nessun file  trovato,  restituito il valore NULL
	 * @param nesting
	 * @param name
	 * @return
	 * @throws S4Exception
	 */
	public FileS4 openFile(String nesting, String name) throws S4Exception {
		return openFile(nesting, name, WritePolicy.READONLY);
	}
	
		
	/**
	 * Recupera le informazioni legate al file
	 * Il risultato non pu essere utilizzato per modificare il file
	 * E' restituito null se il file non esiste
	 * @param nesting
	 * @param name
	 * @return
	 * @throws S4Exception
	 */
	public FileS4 fileInfo(String nesting, String name) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		try {
			return vfat.fileInfo(config.getFilesystemRoot(), config.getContext(), nesting, name);			
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}
	
	/**
	 * Recupera le informazioni legate al file
	 * Il risultato non pu essere utilizzato per modificare il file
	 * E' restituito null se il file non esiste
	 * @param uid
	 * @return
	 * @throws S4Exception
	 */
	public FileS4 fileInfo(Integer uid) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		try {
			return vfat.fileInfo(config.getFilesystemRoot(), uid, false);			
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}
	
	/**
	 * Recupera le informazioni legate al file
	 * Il file pu essere ricercato anche nella cartella delle "cancellazioni"
	 * Il risultato non pu essere utilizzato per modificare il file
	 * E' restituito null se il file non esiste 
	 * @param uid
	 * @param garbage
	 * @return
	 * @throws S4Exception
	 */
	public FileS4 fileInfo(Integer uid, boolean garbage) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		try {
			return vfat.fileInfo(config.getFilesystemRoot(), uid, garbage);			
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}
	
	
	/**
	 * Esegue un test di verifica sulla esistenza del file sull Virtual FAT
	 * @param nesting
	 * @param name
	 * @return
	 * @throws S4Exception
	 */
	public boolean existsFile(String nesting, String name) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		try {
			return vfat.existsFile(config.getContext(), nesting, name);			
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}

	
	/**
	 * Restituisce statistiche sullo stato della {@link VirtualAllocationTable}
	 * @return
	 * @throws S4Exception
	 */
	public VFATStatistics getStatistics() throws S4Exception {
		// Passo di qui ancor prima che il sistema sia configurato
		if (vfat==null) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		if (config==null) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		try {
			return vfat.contextStatistic(config.getContext());			
		} catch (VFATException e) {
			throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
		}
	}
	
	
	
	
	/**
	 * Restituisce lo stato della configurazione del componente
	 * @return
	 */
	public boolean isConfigured() {
		return configured;
	}

	/**
	 * Restitutisce l'istanza di {@link S4Config} con cui  stato configurato l'istanza
	 * @return
	 */
	public S4Config getConfig() {
		return config;
	}

	
	/**
	 * Restituisce il {@link S4Bucket} virtuale utilizzato dalla istanza
	 * Questo  diverso da null soltanto se  configurato l'utilizzo si Amazon S3
	 * @return
	 */
	public S4Bucket getS4bucket() {
		return s4bucket;
	}

	/**
	 * Restituisce l'istanza del servizio {@link AmazonS3} 
	 * Questo  diverso da null soltanto se  configurato l'utilizzo si Amazon S3
	 * @return
	 */
	public AmazonS3 getS3Service() {
		return s3service;
	}

	/**
	 * Restituisce un puntamento alla  {@VirtualFAT}
	 * @return
	 */
	public VFATEngine getVFAT() {
		return vfat;
	}

	/**
	 * {@Owner} del processo e proprietario dei file
	 * @return
	 */
	public Owner getOwner() {
		return owner;
	}
	
	
	@Override
	protected void finalize() throws Throwable {
		if (cleaner!=null) {
			cleaner.stop();
		}
		super.finalize();
	}
	
	
	/**
	 * Restituisce una istanza di {@link Locker}
	 * @param nesting
	 * @param name
	 * @param reference
	 * @return
	 * @throws S4Exception 
	 */
	public LockerS4 getLockHandle(String nesting, String name, String reference) throws S4Exception {
		if (!isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		return vfat.getLockHandle(config.getContext(), nesting, name, reference);		
	}
	
		
	/**
	 * Restituisce un puntamento al vero file recuperando le infirmazioni dal S4 Engine
	 * @param path
	 * @param name
	 * @return
	  */
	public static java.io.File recoverRealFile(String path, String name) {
		S4Engine s4engine = getInstance();						
				
		String parent = s4engine.getConfig().getFilesystemRoot();
		if (s4engine.getConfig().getContext().equals(FileAllocation.ROOT)) {
			parent += "/ROOT";
		} 
		parent += s4engine.getConfig().getContext()+s4engine.getConfig()+path;		
		return new File(parent, name);
	}
	
	
}
