package it.softecspa.s4fs.vfat;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.database.dbconnect.DatabaseStatementUnplugged;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.s4fs.S4Engine;
import it.softecspa.s4fs.S4Exception;
import it.softecspa.s4fs.vfat.VirtualAllocationTable.Format;

import java.sql.SQLException;
import java.sql.Timestamp;

import com.mysql.jdbc.MysqlErrorNumbers;


/**
 * Buffer delle cancellazioni su filesystem
 * 
 * @author m.veroni
 * 
 */
public class ExpireAllocationFS extends ExpireAllocation {

	
	private final String NAME = "s4_015_expire_fs";
	
	// Owner del file
	private Integer ownerUid;
	
	private final String COLUMNS = "uid, owner, expire, deleted";

	
	
	
	ExpireAllocationFS(Integer fsUid, Integer ownerUid, int delay) {
		this.uid = fsUid;
		this.ownerUid = ownerUid;
		this.delay = delay;
	}
	
	ExpireAllocationFS(Integer ownerUid, Timestamp expire) {
		this.ownerUid = ownerUid;
		this.expire = expire;
	}
	
	

	ExpireAllocationFS (Integer fsUid, Integer ownerUid, Timestamp expire, boolean deleted) {
		this.uid = fsUid;
		this.ownerUid = ownerUid;
		this.expire = expire;
		this.deleted = deleted;
	}
	
	static void expireNowWithExclude(FileS4 file, boolean deleted, ConnectionManager cm, Owner exclude) throws VFATException {
		ExpireAllocationFS allocation = new ExpireAllocationFS(file.getUid(), file.getOwnerUid(), EnterpriseCalendar.now().getTimestamp(), deleted);
		allocation.saveAllExcludeOwner(cm, exclude, deleted);
	}
	
	
	
	/**
	 * Esegue un inserimento di un record per tutti gli Owner presenti sul sistema
	 * eccetto che per quello in esclusione
	 * @param cm
	 * @param exclude
	 * @throws VFATException 
	 */
	private void saveAllExcludeOwner(ConnectionManager cm, Owner exclude, boolean deleted) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				// Preventivamente cancello tutti i record associati al file
				StringBuilder sql = new StringBuilder(50);			
				sql.append("DELETE");
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE uid = ").append(getUid());
				try {
					dbsu.execute(sql.toString());			
				} catch (SQLException e1) {
					checkAndCreate(dbsu, e1);
					dbsu.execute(sql.toString());	
				}
						
				// Eseguo una select sugli OWNER per inserire i record nella "expired"
				sql = new StringBuilder(50);			
				sql.append("INSERT INTO ").append(NAME).append("(").append(COLUMNS).append(")");
				sql.append(" SELECT ").append(getUid()).append(" AS 'uid'")
									  .append("uid AS 'owner'")
									  .append(getUid()).append(" AS 'expire'")
									  .append((deleted?"1":"0")).append(" AS 'deleted'");
				sql.append(" FROM ").append(Owner.NAME);
				sql.append(" WHERE uid <> ").append(exclude.getUid());
				try {
					dbsu.execute(sql.toString());			
				} catch (SQLException e1) {
					checkAndCreate(dbsu, e1);
					dbsu.execute(sql.toString());	
				}
				
				// ---------------------------- CORE del metodo ------------------------------				
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}		
	}

	void save(DatabaseStatementUnplugged dbsu) throws SQLException {		
		try {
			// Inserisco il nuovo record
			StringBuilder sql = new StringBuilder(200);			
			sql.append("INSERT INTO ").append(NAME).append(" (").append(COLUMNS).append(")");
			sql.append(" VALUES (");
			sql.append("  ").append(getUid());
			sql.append(", ").append(getOwnerUid());
			sql.append(", ").append(Format.toString(getExpire()));
			sql.append(", 0");
			sql.append(")");
			try {
				dbsu.execute(sql.toString());			
			} catch (SQLException e1) {
				checkAndCreate(dbsu, e1);
				dbsu.execute(sql.toString());	
			}
		} catch (SQLException e2) {
			if (e2.getErrorCode()!=MysqlErrorNumbers.ER_DUP_ENTRY &&
				e2.getErrorCode()!=MysqlErrorNumbers.ER_DUP_KEY) {
				throw e2;
			}
			
			// Aggiorno il record
			StringBuilder sql = new StringBuilder(200);			
			sql.append("UPDATE ").append(NAME).append(" SET");
			sql.append("  expire = ").append(Format.toString(getExpire()));
			sql.append(" WHERE uid = ").append(getUid());
			sql.append(" AND owner = ").append(getOwnerUid());
			dbsu.execute(sql.toString());					
		}				
			
	}
	

	@Override
	String createSQL() {
		StringBuilder sql = new StringBuilder(200);
		sql.append("CREATE TABLE ").append(NAME).append(" (");
		sql.append(" uid int(10) UNSIGNED NOT NULL,");
		sql.append(" owner int(10) UNSIGNED NOT NULL,");
		sql.append(" expire datetime NOT NULL,");
		sql.append(" deleted tinyint(1) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" timerecord timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,");
		sql.append(" primary key (uid, owner)");
		sql.append(" ) ENGINE = MyISAM");
		sql.append(" COMMENT = 'Scheduled delete for filesystem record'");
		return sql.toString();
	}


	@Override
	public String getTableName() {
		return NAME;
	}


	


	public Integer getOwnerUid() {
		return ownerUid;
	}


	void setOwnerUid(Integer ownerUid) {
		this.ownerUid = ownerUid;
	}

	@Override
	String deleteSQL() {
		StringBuilder sql = new StringBuilder(100);
		sql.append("DELETE FROM ").append(NAME);
		sql.append(" WHERE uid = ").append(getUid());
		sql.append(" AND owner = ").append(getOwnerUid());
		sql.append(" AND expire <= "+Format.toString(getExpire()));
		return sql.toString();
	}

	
	/**
	 * Cancella i file dal filesystem
	 *  senza rilanciare eccezioni
	 * @param
	 * @throws VFATException 
	 */
	@Override
	boolean removeFile(FileS4 file) throws VFATException {
		VFATEngine.deleteLocalFylesystem(file);
		return true;
	}

	@Override
	String selectSQLExpired() {
		StringBuilder sql = new StringBuilder(100);
		sql.append("SELECT ").append(COLUMNS);
		sql.append(" FROM ").append(NAME);
		sql.append(" WHERE owner = ").append(getOwnerUid());
		sql.append(" AND expire <= "+Format.toString(getExpire()));
		return sql.toString();
	}

	@Override
	void terminator(FileS4 file) throws S4Exception {
		/* 
		 * Se il file scade sta solo su filesystem LOCALE (e sono il proprietario)
		 * lo posso cancellare in modo definitivo
		 */
		if (!(file.getFSPolicy()==FilesystemPolicy.ONLY_LOCAL &&
			  file.getOwnerUid().equals(getOwnerUid()))) return;			
			
		S4Engine engine = S4Engine.getInstance();
		VFATEngine vfat = engine.getVFAT();
		vfat.deleteFile(engine.getOwner(), file);			
	}

	
	

	
	
	

}
