package it.softecspa.s4fs.vfat;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.database.dbconnect.DatabaseStatementUnplugged;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.s4fs.S4Engine;
import it.softecspa.s4fs.S4Exception;
import it.softecspa.s4fs.vfat.VirtualAllocationTable.Format;

import java.sql.SQLException;
import java.sql.Timestamp;

import com.mysql.jdbc.MysqlErrorNumbers;


/**
 * Buffer delle cancellazioni su S3
 * 
 * @author m.veroni
 * 
 */
public class ExpireAllocationS3 extends ExpireAllocation {

	
	private final String NAME = "s4_016_expire_s3";
	
	// Bucket di Amazon S3
	private Integer bucketUid;
	
	private final String COLUMNS = "uid, bucket, expire, deleted";

	
	
	
	ExpireAllocationS3(Integer fsUid, Integer bucketUid, int delay) {
		this.uid = fsUid;
		this.bucketUid = bucketUid;
		this.delay = delay;
	}
	
	ExpireAllocationS3(Integer bucketUid, Timestamp expire) {
		this.bucketUid = bucketUid;
		this.expire = expire;
	}
	
	ExpireAllocationS3 (Integer fsUid, Integer bucketUid, Timestamp expire, boolean deleted) {
		this.uid = fsUid;
		this.bucketUid = bucketUid;
		this.expire = expire;
		this.deleted = deleted;
	}
	
	
	public static void expireNow(FileS4 file, boolean deleted, ConnectionManager cm) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
		
				ExpireAllocation allocation = new ExpireAllocationS3(file.getUid(), file.getBucketUid(), EnterpriseCalendar.now().getTimestamp(), deleted);
				allocation.save(dbsu);
				
				// ---------------------------- CORE del metodo ------------------------------				
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}		
	}
	
	
	
	void save(DatabaseStatementUnplugged dbsu) throws SQLException {
		try {
			// Inserisco il nuovo record
			StringBuilder sql = new StringBuilder(100);			
			sql.append("INSERT INTO ").append(NAME).append(" (").append(COLUMNS).append(")");
			sql.append(" VALUES (");
			sql.append("  ").append(getUid());
			sql.append(", ").append(getBucketUid());
			sql.append(", ").append(Format.toString(getExpire()));
			sql.append(", 0");
			sql.append(")");
			try {
				dbsu.execute(sql.toString());			
			} catch (SQLException e1) {
				checkAndCreate(dbsu, e1);
				dbsu.execute(sql.toString());	
			}
		} catch (SQLException e2) {
			if (e2.getErrorCode()!=MysqlErrorNumbers.ER_DUP_ENTRY) {
				throw e2;
			}
			
			// Aggiorno il record
			StringBuilder sql = new StringBuilder(100);			
			sql.append("UPDATE ").append(NAME).append(" SET");
			sql.append("  expire = ").append(Format.toString(getExpire()));
			sql.append(" WHERE uid = ").append(getUid());
			sql.append(" AND bucket = ").append(getBucketUid());
			dbsu.execute(sql.toString());					
		}		
	}

	@Override
	String createSQL() {
		StringBuilder sql = new StringBuilder(200);
		sql.append("CREATE TABLE ").append(NAME).append(" (");
		sql.append(" uid int(10) UNSIGNED NOT NULL,");
		sql.append(" bucket int(10) UNSIGNED NOT NULL,");
		sql.append(" expire datetime NOT NULL,");
		sql.append(" deleted tinyint(1) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" timerecord timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,");
		sql.append(" primary key (uid, bucket)");
		sql.append(" ) ENGINE = MyISAM");
		sql.append(" COMMENT = 'Scheduled delete for Amazon S3 record'");
		return sql.toString();
	}


	@Override
	public String getTableName() {
		return NAME;
	}

	public Integer getBucketUid() {
		return bucketUid;
	}

	void setBucketUid(Integer bucketUid) {
		this.bucketUid = bucketUid;
	}

	
	@Override
	String deleteSQL() {
		StringBuilder sql = new StringBuilder(100);
		sql.append("DELETE FROM ").append(NAME);
		sql.append(" WHERE uid = ").append(getUid());
		sql.append(" AND bucket = ").append(getBucketUid());
		sql.append(" AND expire <= "+Format.toString(getExpire()));
		return sql.toString();
	}
	

	/**
	 * Cancella i file dal filesystem ed il relativo record nella tabella
	 * senza rilanciare eccezioni
	 * @param
	 */
	@Override
	boolean removeFile(FileS4 file) throws S4Exception {
		S3FileAction action = new S3FileAction(file);
		action.deleteFile();
		return true;
	}

	
	@Override
	String selectSQLExpired() {
		StringBuilder sql = new StringBuilder(100);
		sql.append("SELECT ").append(COLUMNS);
		sql.append(" FROM ").append(NAME);
		sql.append(" WHERE bucket = ").append(getBucketUid());
		sql.append(" AND expire <= "+Format.toString(getExpire()));
		return sql.toString();
	}

	
	@Override
	void terminator(FileS4 file) throws S4Exception {
		/* 
		 * Se il file scade sta solo su Amazon S3 
		 * lo posso cancellare in modo definitivo
		 */
		if (file.getFSPolicy()!=FilesystemPolicy.ONLY_S3) return;			
			
		S4Engine engine = S4Engine.getInstance();
		VFATEngine vfat = engine.getVFAT();
		vfat.deleteFile(engine.getOwner(), file);		
	}
	
	

}
