package it.softecspa.s4fs.vfat;

import it.softecspa.kahuna.lang.XString;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;

import org.apache.log4j.Logger;


/**
 * Allocazione del file all'interno della tabella di allocazione
 * 
 * @author m.veroni
 *
 */
public class FileAllocation  {

	Logger log = Logger.getLogger(getClass());
	
	public final static String ROOT = "/";
	
	
	protected Integer uid;

	// Chiave logica  
	protected String context;
	protected String nesting;
	protected String name;
	
	
	// Il file  conderato draft fino a che non viene eseguito lo store
	protected boolean draft;
	
	
	// Data creazione
	protected Timestamp creation;
	// Id del nodo che ha eseguito l'inserimento
	protected Integer ownerUid;
	
	// Data ultima modifica
	protected Timestamp modify;
	// Id del nodo che ha eseguito l'ultima modifica
	protected Integer modifierUid;

	// Dimensione in byte del file
	protected Long size;
	
	
	// Data acquisizione lock
	protected Timestamp locking;
	// Id del nodo che ha eseguito il lock
	protected Integer lockerUid;
		
	// Policy di gestione file
	protected FilesystemPolicy fsPolicy;
		
	// Tempo di vita, in millisecondi del file sullo storage
	protected int[] fsExpiration = new int[3];
	
	
	// Valore di default
	String S3GRANT_DEFAULT = "0000000000";
	
	
	protected Integer bucketUid;
	protected String s3Grant;
	
	protected String s3ETag;
	protected String s3VersionId;

	int s3Status;
	String s3StatusMessage;
	
	
	
	// Varaibili di servizio
	/*
	 * Questo flag  utilizzato nel caso in cui sono cambiate le policy di storage
	 * e o le policy di S3 
	 */
	protected String compare_s3Grant;
	protected String compare_s3ETag;
	protected FilesystemPolicy compare_fsPolicy;
	protected int[] compare_fsExpiration;
	
	
	
	/**
	 * Contesto a cui appartiene il file
	 * @return
	 */
	public String getContext() {
		return context;
	}

	
	void setContext(String context) {
		this.context = context;
	}
	
	
	/**
	 * Percorso virtuale all'interno della Virtual FAT
	 * @return
	 */
	public String getNesting() {
		return nesting;
	}
	
	void setNesting(String nesting) {
		/* 
		 * Normalizzazione del nesting
		 *  ...NON DEVE iniziare / e deve terminare con / (se valorizzata)
		 */
		if (nesting==null) nesting = "";
		nesting = nesting.trim();
		if (nesting.length()>0) {
			nesting = XString.replaceAll(nesting, "\\", "/");
			// ...tolgo la / iniziale al nesting
			if (nesting.startsWith("/")) nesting = nesting.substring(1);
			if (nesting.length()>0) {
				if (!nesting.endsWith("/")) nesting += "/";
			}
		}
		this.nesting = nesting;
	}
	
	
	/**
	 * Nome del file
	 * @return
	 */
	public String getName() {
		return name;
	}

	void setName(String name) {
		this.name = name;
	}
	
	
	/**
	 * Timestamp di creazione file.
	 * Equivale alla data della prima chiamata al metodo {@link FileS4#close()}
	 * @return
	 */
	public Timestamp getCreation() {
		return creation;
	}

	void setCreation(Timestamp creation) {
		this.creation = creation;
	}

	
	/**
	 * Identificativo del prorietari del file
	 * @return
	 */
	public Integer getOwnerUid() {
		return ownerUid;
	}

	void setOwnerUid(Integer owner) {
		this.ownerUid = owner;
	}
	
	
	/**
	 * Timestamp di ultima modifica del file
	 * @return
	 */
	public Timestamp getModify() {
		return modify;
	}

	void setModify(Timestamp modify) {
		this.modify = modify;
	}
	
	
	/**
	 * Identificativo del nodo che ha modificato il file
	 * @return
	 */
	public Integer getModifierUid() {
		return modifierUid;
	}

	void setModifierUid(Integer modifier) {
		this.modifierUid = modifier;
	}
	

	
	private static class If {
		
		public static Integer integerNull(ResultSet res, String field) throws SQLException {
			return res.getString(field)==null?null:new Integer(res.getInt(field));
		}
		
		public static Long longNull(ResultSet res, String field) throws SQLException {
			return res.getString(field)==null?null:new Long(res.getLong(field));
		}
		
		public static boolean toBoolean(ResultSet res, String field) throws SQLException {
			return res.getString(field)!=null?1==res.getInt(field):false;
		}
		
	}
	
	protected void readRow(ResultSet res) throws SQLException, VFATException {
		this.uid = res.getInt("uid");

		this.context 		= res.getString("context");
		this.nesting 		= res.getString("nesting");
		this.name 			= res.getString("name");

		this.creation 		= res.getTimestamp("creation");
		this.ownerUid 		= new Integer(res.getInt("owner"));
		this.modify 		= res.getTimestamp("modify");
		this.modifierUid 	= If.integerNull(res, "modifier");
		this.size 			= If.longNull(res, "size");
		
		this.locking 			= res.getTimestamp("locking");
		this.lockerUid 		= If.integerNull(res, "locker");

		this.fsPolicy 		= FilesystemPolicy.rigenerate(If.toBoolean(res, "storage_local")
												    	, If.toBoolean(res, "storage_s3")
												    	, If.toBoolean(res, "storage_cache"));
		
		this.fsExpiration[Storage.LOCAL.index()] 	= res.getInt("timeout_local");
		this.fsExpiration[Storage.S3.index()] 		= res.getInt("timeout_s3");
		this.fsExpiration[Storage.CACHE.index()] 	= res.getInt("timeout_cache");

		this.bucketUid 		= If.integerNull(res, "s3_bucket");
		this.s3Grant 		= res.getString("s3_grant");

		this.draft 			= If.toBoolean(res,"draft");  	
		this.s3Status			= res.getInt("s3_status");
		this.s3StatusMessage 	= res.getString("s3_status_message");
		
		
		
		// ...cache di servizio utilizzata per limitare le interazioni con Amazon S3
		this.compare_s3Grant = new String(this.s3Grant);
		this.compare_fsPolicy = fsPolicy;
		this.compare_s3ETag = (this.s3ETag!=null?new String(this.s3ETag):null);
		this.compare_fsExpiration = new int[fsExpiration.length];
		System.arraycopy(fsExpiration, 0, compare_fsExpiration, 0, fsExpiration.length);
		
	}

	
	
	/**
	 * Identficativo univoco del file all'interno della Virtual FAT
	 * @return
	 */
	public Integer getUid() {
		return uid;
	}

	void setUid(Integer uid) {
		this.uid = uid;
	}

	
	/**
	 * Il file  una bozza e non   stato ancora consolidato sulla Vitual FAT
	 * @return
	 */
	public boolean isDraft() {
		return draft;
	}

	void setDraft(boolean draft) {
		this.draft = draft;
	}

	
	/**
	 * Restituisce le policy di scrittura del file
	 * {@linkplain FilesystemPolicy}
	 * @return
	 */
	public FilesystemPolicy getFSPolicy() {
		if (fsPolicy==null) fsPolicy = FilesystemPolicy.BUFFER;
		return fsPolicy;
	}
	
	void setFSPolicy(FilesystemPolicy value) {
		this.fsPolicy = value;
	}
	
	
	
	/**
	 * Restituisce un array contenente le date di scadenza dei vari {@link Storage}
	 * Il valore  in giorni
	 * @return
	 */
	public int[] getFSExpiration() {
		return fsExpiration;
	}

	/**
	 * Restituisce il valore di scadenza per lo {@link Storage#LOCAL}
	 * Il valore  in giorni
	 * @return
	 */
	public int getFSExpirationLocal() {
		return fsExpiration[Storage.LOCAL.index()];
	}
	
	/**
	 * Restituisce il valore di scadenza per lo {@link Storage#CACHE}
	 * Il valore  in giorni
	 * @return
	 */
	public int getFSExpirationCache() {
		return fsExpiration[Storage.CACHE.index()];
	}
	
	/**
	 * Restituisce il valore di scadenza per lo {@link Storage#S3}
	 * Il valore  in giorni
	 * @return
	 */
	public int getFSExpirationS3() {
		return fsExpiration[Storage.S3.index()];
	}
	
	
	
	
	
	/**
	 * Restituisce il valore delle policy associate al file condiviso su filesystem Amazon S3
	 * E' restituita una stringa contente valori 0 e 1
	 * 
	 * @return
	 */
	public String getS3Grant() {
		if (s3Grant==null) s3Grant = S3GRANT_DEFAULT;
		return s3Grant;
	}
	
	/**
	 * Resetta il valore delle policy a quello prima della modifica
	 */
	public void resetS3Grant() {
		s3Grant = compare_s3Grant;
	}
	
	void setS3Grant(String s3grant) {
		this.s3Grant = s3grant;
	}
	
	/**
	 * Restituisce lo stato della singola {@link Permission} associata al {@GroupGrantee} di Amazon S3
	 * @param grantee
	 * @param permission
	 * @return
	 */
	public boolean isS3Grant(GroupGrantee grantee, Permission permission) {
		if (s3Grant==null) return false;
		int index = (grantee.base()+permission.index());
		if (s3Grant.length()<index) return false;
		return ('1'==s3Grant.charAt(index));
	}

	
	
	
	/**
	 * identificativo univoco del "bucket" di Amazon S3
	 * @return
	 */
	public Integer getBucketUid() {
		return bucketUid;
	}

	void setBucketUid(Integer bucket) {
		this.bucketUid = bucket;
	}

	
	/**
	 * Timestamp di blocco del record
	 * @return
	 */
	public Timestamp getLock() {
		return locking;
	}
	
	void setLock(Timestamp lock) {
		this.locking = lock;
	}

	/**
	 * Restituisce TRUE se il file  bloccato
	 * @return
	 */
	public boolean isLocked() {
		return (locking!=null);
	}
	
	
	/**
	 * Identificato del nodo che sta bloccando il file
	 * @return
	 */
	public Integer getLockerUid() {
		return lockerUid;
	}

	void setLockerUid(Integer lockerUid) {
		this.lockerUid = lockerUid;
	}

	
	/**
	 * Dimansione in Byte del file
	 * @return
	 */
	public Long getSize() {
		return size;
	}

	void setSize(Long size) {
		this.size = size;
	}

	
	/**
	 * Valore dell'Etag generato da Amazon S3
	 * @return
	 */
	public String getS3ETag() {
		return s3ETag;
	}

	void setS3ETag(String s3eTagt) {
		s3ETag = s3eTagt;
	}

	
	/**
	 * Versione del file condiviso su Amanzon S3
	 * Questo valore non  sempre presente
	 * @return
	 */
	public String getS3VersionId() {
		return s3VersionId;
	}

	void setS3VersionId(String s3VersionId) {
		this.s3VersionId = s3VersionId;
	}

	
	
	/**
	 * Valore restituito dalla operazione di caricamendo del file su Amazon S3
	 * E' significativo soltanto in caso di eventuale errore di caricamento, altrimenti = 0
	 * 
	 * @return
	 */
	public int getS3Status() {
		return s3Status;
	}

	void setS3Status(int status) {
		this.s3Status = status;
	}

	
	/**
	 * Valore restituito dalla operazione di caricamendo del file su Amazon S3
	 * E' significativo soltanto in caso di eventuale errore di caricamento, altrimenti vuoto
	 * @return
	 */
	public String getS3StatusMessage() {
		return s3StatusMessage;
	}
	
	void setS3StatusMessage(String statusMessage) {
		this.s3StatusMessage = statusMessage;
	}

	
	
	
	protected String getCompareS3Grant() {
		return compare_s3Grant;
	}

	protected String getCompareS3ETag() {
		return compare_s3ETag;
	}

	protected FilesystemPolicy getCompareFSPolicy() {
		return compare_fsPolicy;
	}
	
	protected int[] getCompareFSExpiration() {
		if (compare_fsExpiration==null) compare_fsExpiration = new int[Storage.values().length];
		return compare_fsExpiration;
	}
	
	

	
}
