package it.softecspa.s4fs.vfat;

import it.softecspa.kahuna.lang.XString;
import it.softecspa.s4fs.S4Engine;
import it.softecspa.s4fs.S4Exception;
import it.softecspa.s4fs.S4Status;

import java.sql.ResultSet;
import java.sql.SQLException;



/**
 * 
 * @author m.veroni
 *
 */
public class FileS4 extends FileAllocation {
	
	 
	
	private java.io.File realFile;
	
	/*
	 * Indica se il file  APERTO oppure  gi stato finalizzato dal metodo .close() 
	 */
	private boolean open;
	
	/* 
	 * Indica se il file  aperto in scrittura
	 * Questo  necessario al fine di limitare gli aggiornamenti su S3
	 * Sono considerati aperti in scrittura tutti i file CREATI
	 * e quelli espressamente indicati come tali
	 */
	private WritePolicy writePolicy;
	
	/*
	 * Se TRUE indica la necessit di un UPDATE del file su S3
	 * La variabile pu cambiare stato soltanto se writable = UPDATABLE
	 */
	private boolean changed;
	
	
	
	protected FileS4 () {		
	}
	
	/**
	 * Crea una istanza a partire dai dati estratti dal database
	 * @param res
	 * @throws SQLException 
	 * @throws S4Exception 
	 */
	FileS4 (ResultSet res) throws SQLException, VFATException {	
		readRow(res);
	}
	
	
	/**
	 * Crea una istanza a partire dalle info prsenti in una Allocation table 
	 * @param vfile
	 * @param draft
	 */
	FileS4 (FileAllocation vfile, boolean draft) {
		// Chiave logica  
		this.context = vfile.context;
		this.nesting	 = vfile.nesting;
		this.name	 = vfile.name;
		
		this.draft	 = draft;
		
		this.creation = vfile.creation;
		this.ownerUid = vfile.ownerUid;
		
		this.modify	 = vfile.modify;		
		this.modifierUid = vfile.modifierUid;
		
		this.locking	 	= vfile.locking;		
		this.lockerUid 		= vfile.lockerUid;

		this.fsPolicy = vfile.fsPolicy;
		
		for (int i=0; i<fsExpiration.length; i++) {
			this.fsExpiration[i] = vfile.fsExpiration[i];  
		}
		
		this.bucketUid = vfile.bucketUid;
		this.s3Grant = vfile.s3Grant;
		
		this.s3Status = vfile.s3Status;
		this.s3StatusMessage = vfile.s3StatusMessage;
		
	}
	
	

	/**
	 * Crea il file su filesystem e contestulamente si preoccupa di creare le sottocartelle necessarie
	 * @param filesystemRoot
	 * @throws VFATException
	 */
	void createRealFile(String filesystemRoot) throws VFATException {
		String parent = filesystemRoot;
		if (context.equals(ROOT)) {
			parent += "/ROOT";
		} 
		parent += getContext()+getNesting();
		
		realFile = new java.io.File(razionalizzaPath(parent),getName());	
		if (realFile!=null) {
			try {
				realFile.getParentFile().mkdirs();
			} catch (SecurityException e) {
				throw new VFATException("Security exception in creation of real file path: '"+realFile.getParent()+"'", e);
			} catch (Exception e) {
				throw new VFATException("Unhandled exception in creation of real file path: '"+realFile.getParent()+"'", e);
			}
		}
	}
	
	
	
	
	private String razionalizzaPath(String path) {
		if (path == null) return "/";
		path = path.trim();
		if (path.length()<=1) return "/";
		
		path = XString.replaceAll(path, "\\", "/");
		path = XString.replaceAll(path, "//", "/"); // ...toglie i "doppioni"
		if (path.endsWith("/")) path = path.substring(0, path.length()-1);
		
		return path;
	}
	
	
	/**
	 * Restituisce il puntamento al vero file
	 */
	public final java.io.File getRealFile() {
		return realFile;
	}

	public String getCompletePath() {
		return context + nesting + name;
	}
	
	
	/**
	 * Chiude il file e lo storicizza sulla VFAT secondo le policy impostate
	 * Se necessario esegue l'upload su Amazon S3
	 * @throws S4Exception 
	 */
	public synchronized void close() throws S4Exception {
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		
		S4Engine engine = S4Engine.getInstance();
		if (!engine.isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		VFATEngine vfat = engine.getVFAT();
		vfat.closeFile(this);
	}
	
	

	/**
	 * Chiave univoca per salvare il file su Amazon S3
	 * 
	 * In seguito, a seconda delle polici il file pu essere recuperato in http
	 * http://[bucketname].s3.amazonaws.com/[context(-1)][nesting][uid]-[name]
	 * 
	 * @return
	 */
	protected final String getS3Key() {
		// devo togliere la / dal context iniziale altrimenti Amazon s'incazza e non si pu aprire in HTTP
		return (context+nesting+uid+"-"+name).substring(1);
	}

	
	
	/**
	 * Cancella il file dal filesystem locale
	 * @return
	 */
	public boolean removeFile() {
		try {
			VFATEngine.deleteLocalFylesystem(this);
			return true;
		} catch (VFATException e) {
			return false;
		}		
	}
	
	
	/**
	 * Cancella il file dalla VFAT e programma la cancellazione su filesystem distribuiti
	 * @return
	 * @throws S4Exception 
	 */
	public boolean delete() throws S4Exception {
		if (!isWritable()) throw new S4Exception(S4Status.FILE_NOT_WRITABLE);
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		
		S4Engine engine = S4Engine.getInstance();
		if (!engine.isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		VFATEngine vfat = engine.getVFAT();
		vfat.deleteFile(engine.getOwner(), this);
		
		return false;
	}

	
	/**
	 * Restituisce TRUE se il file  aperto
	 * @return
	 */
	public boolean isClosed() {
		return !open;
	}

	void setOpen(boolean open) {
		this.open = open;
	}
	
	
	
	/**
	 * Campia le permission associate al file su filesystem Amazon S3
	 * @param grantee
	 * @param permission
	 * @param value
	 * @return
	 * @throws S4Exception
	 */
	public synchronized boolean changeS3Grant(GroupGrantee grantee, Permission permission, boolean value) throws S4Exception {
		if (!isWritable()) throw new S4Exception(S4Status.FILE_NOT_WRITABLE);
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		if (!getFSPolicy().useS3())  throw new S4Exception(S4Status.REAL_FILE_NOT_USE_S3);
		
		getS3Grant();
		char[] appo = s3Grant.toCharArray();
		int index = (grantee.base()+permission.index());
		appo[index] = (value?'1':'0');
		s3Grant = new String(appo);
		return true;
	}
	
	
	
	
	/**
	 * Cambia le politiche di cancellazione dei file per lo specifico {@link Storage}
	 * Attenzione, queste sono cambiate soltanto a seguito di un {@link FileS4#close()}
	 * @param storage
	 * @param timeout in giorni
	 * @throws VFATException 
	 */
	public void setFsExpiration(Storage storage, int timeout) throws S4Exception {
		if (!isWritable()) throw new S4Exception(S4Status.FILE_NOT_WRITABLE);
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		if (storage.equals(Storage.S3) && !getFSPolicy().useS3())  throw new S4Exception(S4Status.REAL_FILE_NOT_USE_S3);
		
		this.fsExpiration[storage.index()] = timeout;
	}

	
	/**
	 * Cambia le politiche di cancellazione dei file con una sola chiamata
	 * Attenzione, queste sono cambiate soltanto a seguito di un {@linkplain FileS4#close()}
	 * @param localTimeout
	 * @param s3Timeout
	 * @param cacheTimeout
	 * @throws S4Exception
	 */
	public void setFsExpiration(int localTimeout, int s3Timeout, int cacheTimeout) throws S4Exception {
		if (!isWritable()) throw new S4Exception(S4Status.FILE_NOT_WRITABLE);
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		
		this.fsExpiration[Storage.LOCAL.index()] = localTimeout;
		this.fsExpiration[Storage.S3.index()] = s3Timeout;
		this.fsExpiration[Storage.CACHE.index()] = cacheTimeout;
	}
	
	
	/**
	 * Restituisce lo stato della {@WritePolicy}
	 * @return
	 */
	public WritePolicy getWritePolicy() {
		return writePolicy;
	}
	
	void setWritePolicy(WritePolicy value) {
		this.writePolicy = value;		
	}
	
	/**
	 * Il file  il lettura/scrittura
	 * @return
	 */
	public boolean isWritable() {
		return WritePolicy.UPDATABLE.equals(writePolicy);
	}

	/**
	 * Il file  in sola lettura
	 * @return
	 */
	public boolean isReadonly() {
		return WritePolicy.READONLY.equals(writePolicy);
	}
	
	
	
	/**
	 * Restituisce TRUE se il file  cambiato e deve quindi essere ricaricato su Amazon S3
	 * Il valore  significativo soltanto in alcuni casi
	 * @return
	 */
	protected boolean isChanged() {
		return changed;
	}

	/**
	 * Cambia lo stato del file
	 * @param changed
	 */
	void setChanged(boolean changed) {
		this.changed = changed;
	}
	
	
	
	/**
	 * Metto un marker sul file in modo che alla chiusura possa essere aggiornato su S3
	 * Questa operazione non  permessa se il file  in modalit DRAFT
	 * @throws S4Exception
	 */
	public void updateMarker() throws S4Exception {
		if (isDraft()) throw new S4Exception(S4Status.FILE_IN_DRAFT_MODE);
		if (!isWritable()) throw new S4Exception(S4Status.FILE_NOT_WRITABLE);
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		
		log.info("File manually marked as updatable!");
		setChanged(true);		
	}
	
	
	/**
	 * Tolgo il marker di update dal file
	 * Questa operazione non  permessa se il file  in modalit DRAFT
	 * @throws S4Exception
	 */
	public void revokeUpdateMarker() throws S4Exception {
		if (isDraft()) throw new S4Exception(S4Status.FILE_IN_DRAFT_MODE);
		if (!isWritable()) throw new S4Exception(S4Status.FILE_NOT_WRITABLE);
		if (isClosed()) throw new S4Exception(S4Status.FILE_IS_CLOSED);
		
		log.info("Revoke update marker!");
		setChanged(false);		
	}

	
	/**
	 * Restituisce TRUE se il file  condiviso su Amazon S3
	 * @return
	 */
	public boolean isS3Shared() {
		return (this.s3ETag!=null);
	}

	
	
	
}
